// java

package edu.washington.cac.calendar.data;

import java.sql.ResultSet;
import java.sql.SQLException;

import edu.washington.cac.calendar.db.Caldata;
import edu.washington.cac.calendar.db.DBResources;
import edu.washington.cac.calendar.db.Fields;
import edu.washington.cac.calendar.db.WhereClauses;

/**
  The location of an <code>Event</code>
  @author Greg Barnes
  @version 2.1
 */
public class Location extends CalendarObject implements LinkEntity
{
  private int locationid;
  private String address, subaddress, link;

  /** Unique ID of a location that has been deleted (An event can
      use a location created by another user, and if the user
      deletes the location, the 'deleted' location can be used
      to indicate this event to this user.) */
  public static final int DELETED_LOCATION_ID = 3;

  /** Unique ID of a 'no location' location */
  public static final int NO_LOCATION_ID = 1;

  /** Unique ID of an unknown location */
  public static final int UNKNOWN_LOCATION_ID = 2;

  /** Unique ID of the default location if none is specified */
  public static final int DEFAULT_LOCATION_ID = NO_LOCATION_ID;

  // /** A dummy location */
  // public static Location DUMMY_LOCATION;      // final, really

  /** A deleted location */
  public static Location DELETED_LOCATION;    // final, really

  static {
    /* try {
      DUMMY_LOCATION = new Location(DEFAULT_LOCATION_ID, "no location", "",
                                    null, DBResources.specialCreator(), true);
    } catch (CaldataException e) {  // shouldn't happen
      DUMMY_LOCATION = null;
    }*/

    try {
      DELETED_LOCATION = new Location(DELETED_LOCATION_ID, "deleted",
                                      "this location was deleted", null,
                                      DBResources.specialCreator(), true);
    } catch (CaldataException e) {  // shouldn't happen
      DELETED_LOCATION = null;
    }
  }

  /** create a location with all fields specified

    @param locationid unique id
    @param address main address
    @param subaddress secondary address
    @param link URL with more info
    @param creator UWNetID of user that created it
    @param isPublic is the location viewable by everyone?

    @exception CaldataException if the fields are not valid in some way
   */
  public Location(int locationid,
                  String address,
                  String subaddress,
                  String link,
                  User creator,
                  boolean isPublic)
      throws CaldataException
  {
    super(creator, isPublic);
    DataChecker.checkAddress(address);

    this.locationid = locationid;
    this.address = address;
    this.subaddress = subaddress;
    this.link = link;
  }

  /** create a location from the next row in a
       <code>java.sql.ResultSet</code>

      @param rs The <code>ResultSet</code> with the location
      @exception CaldataException if the fields are not valid in some way
      @exception SQLException if there is a problem with the ResultSet
    */
  public Location(ResultSet rs) throws SQLException, CaldataException
  {
    this(rs.getInt(Fields.LLOCATION_ID.getName()),
         rs.getString(Fields.LADDRESS.getName()),
         rs.getString(Fields.LSUBADDRESS.getName()),
         rs.getString(Fields.LLINK.getName()),
         new User(rs.getString(Fields.LCREATOR.getName())),
         rs.getString(Fields.LPUBLIC.getName()).equals(WhereClauses.TRUE));
  }

  public void setId(int locationid)
  {
    this.locationid = locationid;
  }

  /**
    Get the unique id of the location
    @return the unique id of the location
   */
  public int getId()
  {
    return locationid;
  }

  /**
    Test whether the location is actually specified
    @return whether the location is actually specified (<i>i.e.</i>,
          not <code>UNKNOWN</code>, <code>DELETED</code>,
          or <code>NO_LOCATION</code>)
   */
  public boolean isLocation()
  {
    return locationid != NO_LOCATION_ID &&
           locationid != UNKNOWN_LOCATION_ID &&
           locationid != DELETED_LOCATION_ID;
  }

  /**
    Get the main address of the location
    @return the main address of the location
   */
  public String getAddress()
  {
    return address;
  }

  /**
    Get the secondary address of the location
    @return the secondary address of the location
   */
  public String getSubaddress()
  {
    return subaddress;
  }

  /**
    Get the link for the location
    @return the link for the location
   */
  public String getLink()
  {
    return link;
  }

  /**
    Set the Location's URL
    @param link The new URL
   */
  public void setLink(String link)
  {
    this.link = link;
  }

  private String first30(String s)
  {
    if (s.length() < 30) {
      return s;
    } else {
      return s.substring(0, 30);
    }
  }

  /**
    Get a <code>String</code> representation of the location
    @return a <code>String</code> representation of the location
   */
  public String toString()
  {
//      return first25(address) + " (" + creator.getNameDB() + ")";
    return first30(address);
  }

  /**
    delete this location from the database
    @param user The UWNetID of the user requesting the deletion.
        There should be no effect if this does not match the event's
        creator.
    @exception SQLException if there is a database problem
   */
  public void delete(User user) throws SQLException
  {
//      System.err.println("Deleting location " + getId() + " for " + user.getName());
    getCaldata().deleteLocation(getId(), user);
  }

  /** replace any location in the database with the same id as this
        location with this location

    @param user The user requesting the replace.
        There should be no effect if this does not match the old event's
        creator.

    @exception SQLException if there is a database problem
   */
  public void replace(User user) throws SQLException
  {
//      System.err.println("Replacing location " + getId() + " for " + user.getNameDB());
    getCaldata().replaceLocation(this, user);
  }

  /** add this location to the database

    @param user The user requesting the add.
         This will become the location's creator in the database

    @exception SQLException if there is a database problem
   */
  public void add(User user) throws SQLException
  {
//      System.err.println("Adding location for " + user.getNameDB());
    locationid = getCaldata().addLocation(this, user);
  }

  /**
    Test whether this location is less than another location

    @param c The other location to compare.  If c is a
        <code>CalendarObject</code> that is not an location, the method returns
        <code>false</code>
    @return whether this location is less than another location
   */
  public boolean lessThan(CalendarObject c)
  {
    if (!(c instanceof Location)) {
      return false;
    }

    Location l = (Location) c;

    // If one is the special creator and the other one isn't, compare creators
    if (!getCreator().equals(l.getCreator()) &&
        (getCreator().equals(DBResources.specialCreator()) ||
         l.getCreator().equals(DBResources.specialCreator())))
    {
      return getCreator().getName().compareTo(l.getCreator().getName()) < 0;
    } else {    // compare addresses
      return (getAddress().compareTo(l.getAddress()) < 0);
    }
  }
}
