// java

package edu.washington.cac.calendar.data;

import java.sql.ResultSet;
import java.sql.SQLException;

import edu.washington.cac.calendar.db.Caldata;
import edu.washington.cac.calendar.db.Fields;
import edu.washington.cac.calendar.db.WhereClauses;

/**
  A keyword for an event
  @author Greg Barnes
  @version 2.2
 */
public class Keyword extends CalendarObject implements LongdescEntity
{
  /** Unique id for the keyword */
  private int keywordid;

  /** Display name of the keyword */
  private String word;

  /** Longer description of the keyword */
  private String longdesc;

  /**
    Create a new keyword, specifying all fields

    @param keywordid Unique ID for the keyword
    @param word Short description
    @param longdesc Long description
    @param creator The person who created the keyword
    @param isPublic Is the keyword public?

    @exception CaldataException if the fields are not valid in some way
   */
  public Keyword(int keywordid,
                 String word,
                 String longdesc,
                 User creator,
                 boolean isPublic)
      throws CaldataException
  {
    super(creator, isPublic);
    DataChecker.checkShortdesc(word);

    this.keywordid = keywordid;
    this.word = word;
    this.longdesc = longdesc;
  }

  /**
    Create a new keyword from a JDBC ResultSet

    @param rs The result set with the keyword fields
    @exception SQLException if there's a problem with the database
    @exception CaldataException if the fields are not valid in some way
   */
  public Keyword(ResultSet rs) throws SQLException, CaldataException
  {
    this(rs.getInt(Fields.KKEYWORD_ID.getName()),
         rs.getString(Fields.KWORD.getName()),
         rs.getString(Fields.KLONGDESC.getName()),
         new User(rs.getString(Fields.KCREATOR.getName())),
         rs.getString(Fields.KPUBLIC.getName()).equals(WhereClauses.TRUE));
  }

  public void setId(int keywordid)
  {
    this.keywordid = keywordid;
  }

  public int getId()
  {
    return keywordid;
  }

  public String getWord()
  {
    return word;
  }

  public String getShortdesc()
  {
    return getWord();
  }

  public String getLongdesc()
  {
    return longdesc;
  }

  public void delete(User user) throws SQLException
  {
//      System.err.println("Deleting keyword " + getId() + " for " + user.getName());
    getCaldata().deleteKeyword(getId(), user);
  }

  /**
    Replace the old version of this keyword in the database with this version
    @param user User requesting the change.  If user is not the creator,
      this should have no effect
    @exception SQLException if there's a db error
   */
  public void replace(User user) throws SQLException
  {
//  System.err.println("Replacing keyword " + getId() + " for " + user.getNameDB());
    getCaldata().replaceKeyword(this, user);
  }

  /**
    Add this keyword to the database
    @param user User requesting the add
    @exception SQLException if there's a db error
   */
  public void add(User user) throws SQLException
  {
//    System.err.println("Adding keyword for " + user.getNameDB());
    keywordid = getCaldata().addKeyword(this, user);
  }

  public boolean equals(Object obj)
  {
    if (obj == null || !(obj instanceof Keyword)) {
      return false;
    } else {
      return ((Keyword) obj).getWord().equals(getWord()) &&
             ((Keyword) obj).getCreator().equals(getCreator());
    }
  }

  /**
    @return whether this keyword is less than another
    @param c Keyword to compare to.  If <code>c</code> is not a keyword,
      return false
   */
  public boolean lessThan(CalendarObject c) {
    if (!(c instanceof Keyword)) {
      return false;
    }

    Keyword k = (Keyword) c;

    if (getWord().compareTo(k.getWord()) != 0) {
      return (getWord().compareTo(k.getWord()) < 0);
    } else if (getLongdesc().compareTo(k.getLongdesc()) != 0) {
      return (getLongdesc().compareTo(k.getLongdesc()) < 0);
    } else {
      return getCreator().getNameDB().compareTo(k.getCreator().getNameDB()) < 0;
    }
  }
}
