// java

package edu.washington.cac.calendar.data;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Iterator;

import edu.washington.cac.calendar.db.Fields;
import edu.washington.cac.calendar.db.WhereClauses;

/** An Event that is an instance of a recurring event
    @author Greg Barnes
    @version 1.0
  */
public class InstanceEvent extends Event
{
  /** The master event for this recurrence */
  private Event master;

  /** Create an instance event corresponding to the current row in a given
        <code>java.sql.ResultSet</code> of events

      @param rs The <code>ResultSet</code> holding the event
      @param master The master event for this recurrence

      @exception SQLException if there's trouble accessing the
        <code>ResultSet</code>
      @exception CaldataException if the fields are not valid in some way
    */
  public InstanceEvent(ResultSet rs, Event master)
      throws SQLException, CaldataException
  {
    super(Fields.getInstanceId(rs),
          rs.getString(Fields.ESHORTDESC.getName()),
          Fields.getEventLongdesc(rs),
          rs.getDate(Fields.MDSTARTDATE.getName()),
          rs.getTime(Fields.MDSTARTTIME.getName()),
          rs.getDate(Fields.MDENDDATE.getName()),
          rs.getTime(Fields.MDENDTIME.getName()),
          rs.getString(Fields.EPUBLIC.getName()).equals(WhereClauses.TRUE),
          rs.getString(Fields.ELINK.getName()),
          new User(rs.getString(Fields.ECREATOR.getName())),
          rs.getString(Fields.ECOST.getName()),
          new InstanceRecurrence(master.getId()),
          master.getLocationid(),
          master.getSponsorid(),
          master.getSeq(),
          master.getLastmod(),
          master.getCreated());
    this.master = master;
  }

  /**
    Get the master for this event
    @return the master for this event
   */
  public Event getMaster()
  {
    return this.master;
  }

  /**
    Set recurrence information for this event
    @param recur Recurrence information
   */
  public void setRecurrence(Recurrence recur)
  {
    if (!(recur instanceof InstanceRecurrence)) {
      throw new IllegalArgumentException(
          "InstanceEvent:setRecurrence must have an InstanceRecurrence");
    }

    super.setRecurrence(recur);
  }

  /**
    @return If the event has a particular keyword
    @param k Keyword to test
   */
  public boolean hasKeyword(Keyword k)
  {
    return this.master.hasKeyword(k);
  }

  /**
    Get an <code>Iterator</code> of the event's keywords
    @return an <code>Iterator</code> of the event's keywords
   */
  public Iterator getKeywords()
  {
    return this.master.getKeywords();
  }

  /**
    Illegal operation

    @exception IllegalStateException Always
   */
  public void addKeyword(int id) throws IllegalStateException
  {
    throw new IllegalStateException("can't add a keyword to an InstanceEvent");
  }

  /**
    Delete any missing keywords associated with the event
    @param ks Keywords cache
   */
  public void deleteMissingKeywords(Keywords ks)
  {
    this.master.deleteMissingKeywords(ks);
  }
}
