// java

package edu.washington.cac.calendar.data;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Calendar;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;

import edu.washington.cac.calendar.MyCalendar;

/**
  An interface that can be used for either an Events object or something
  that imitates it.

  @author Greg Barnes
  @version 1.0
 */
public interface EventsI extends CacheI
{
  /**
    @return If the key represents an object in the set
    @param o Key
   */
  public boolean containsKey(Object o);

  /**
    Get the event in the set with the given id

    @return the event in the set with the given id
    @param eventID Unique ID of the event
    @exception NoSuchItemException if no such event exists
   */
  public Event getEvent(int eventID) throws NoSuchItemException;

  /** Get the event in the set with the given id. Optionally load it into
   * the cache if it's not already there.
   *
   * @param eventID    int Unique ID of the event
   * @param loadIt     load the event if not in cache
   * @return Event     from the set with the given id
   * @exception NoSuchItemException if no such event exists
   */
  public Event getEvent(int eventID,
                        boolean loadIt) throws NoSuchItemException;

  /**
    get an item from the Cache

    @param id ID of the item
    @return The CalendarObject with the same id
    @exception NoSuchItemException if no item exists with that id
   */
  public CalendarObject get(int id) throws NoSuchItemException;

  /**
    @return All the items in the cache
   */
  public Iterator elements();

  /**
    Get all the items in the cache as a collection.
    @return all the items in the cache as a collection.
   */
  public Collection values();

  /**
    @return The number of items in the Cache
   */
  public int size();

  /**
    @return all the elements in the cache, in sorted order
   */
  public Iterator sortedElements();

  /**
    Get the events for a user on a given day, sorted in an appropriate
    way

    @param mc Date, represented as a <code>MyCalendar</code>
    @param loadKeywords should the keywords be loaded (if the events are?)

    @return the events for a user on a given day, sorted
    @exception SQLException if there is a problem loading from the database
    @exception ItemException if the database contains bad data
  */
  public Event[] oneDaysEvents(MyCalendar mc, boolean loadKeywords)
      throws SQLException, ItemException;

  /**
    Get the events for a user on a given day, sorted in an appropriate
    way

    @param mc Date, represented as a <code>MyCalendar</code>
    @return the events for a user on a given day, sorted
    @exception SQLException if there is a problem loading from the database
    @exception ItemException if the database contains bad data
  */
  public Event[] oneDaysEvents(MyCalendar mc)
      throws SQLException, ItemException;

  /**
    Get the events for a user over a series of days, sorted in an appropriate
    way

    @param startDate First day in the series, represented as a
        <code>MyCalendar</code>
    @param endDate Last day in the series, represented as a
        <code>MyCalendar</code>
    @return the events for a user over the series of days, sorted
    @exception SQLException if there is a problem loading from the database
    @exception CaldataException if the database contains bad data
  */
  public List manyDaysEvents(MyCalendar startDate, MyCalendar endDate)
      throws SQLException, ItemException;

//  /**
//    Add an item to the cache
//    @param c Item to add
//    @exception SQLException If there is a database problem
//    @exception ItemAccessException If there is a database problem
//   */
//  public void add(CalendarObject c) throws SQLException, ItemAccessException;
//
//  /**
//    Add an item to the cache
//    @param c Item to add
//    @param saveToDB Save to database as well?
//    @exception SQLException If there's a database problem
//   */
//  public void add(CalendarObject c, boolean saveToDB) throws SQLException;
//
//  /**
//    delete an item from the Cache
//
//    @param c the item to delete
//    @exception SQLException if there is a problem with the database
//    @exception ItemException if there's a problem with c
//   */
//  public void delete(CalendarObject c) throws SQLException, ItemException;
//
//  /**
//    replace an item in the Cache
//
//    @param c the item to replace (the current item with the same id
//      as c will be replaced with c)
//    @exception NoSuchItemException if there is no current item with
//      the same id as c
//    @exception SQLException if there is a problem with the database
//    @exception ItemAccessException if the user cannot replace this item
//   */
//  public void replace(CalendarObject c)
//      throws SQLException, NoSuchItemException, ItemAccessException;

  /** Set the keywords cache
    @param ks new Keywords cache
   */
  public void setKeywords(Keywords ks);

  /** Set the Locations cache
    @param ls new Locations cache
   */
  public void setLocations(Locations ls);

  /** Set the sponsors cache
    @param ss new Sponsors cache
   */
  public void setSponsors(Sponsors ss);

  /**
    Get the set of associated locations for a user
    @return the set of associated locations for a user
    */
  public Locations getLocations();

  /**
    Get the set of associated sponsors for a user
    @return the set of associated sponsors for a user
   */
  public Sponsors getSponsors();

  /**
     Get the set of associated keywords for a user
     @return the set of associated keywords for a user
   */
  public Keywords getKeywords();

  /**
    Alert the object that events from many days are about to be
    requested.  This need not be called, but should speed up a
    series of calls to <code>oneDaysEvents()</code> if it is

    @param start The first date in a series of dates
    @param end The last date in a series of dates
    @exception SQLException if we have trouble loading data
    @exception ItemException if the data we try to load is bad
  */
  public void prepareManyDays(Calendar start, Calendar end)
      throws SQLException, ItemException;

//  /** load a result set containing events for one day
//      @param mc The day in question
//      @param rs The <code>java.sql.ResultSet</code> containing the events
//      @exception SQLException if there is database trouble
//      @exception CaldataException if the data we get back is bad
//    */
//  public void load(MyCalendar mc, ResultSet rs)
//      throws SQLException, CaldataException;
}
