// java

package edu.washington.cac.calendar.data;

import edu.washington.cac.calendar.MyCalendar;
import edu.washington.cac.calendar.db.Caldata;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.TreeSet;
import java.util.Vector;

import org.apache.log4j.Logger;

/**
  A set of Events for a single user.

  @author Greg Barnes
  @version 2.2
 */
public class Events extends Cache implements EventsI
{
  private HashMap days = new HashMap();
  private HashMap sortedDays = new HashMap();
  private HashMap loadedDays = new HashMap();
  private Locations ls;
  private Sponsors ss;
  private Keywords ks;

  /** Initialize the set for a particular user
    @param user the user
    @param load Whether to load locations, keywords and sponsors on creation
    @exception SQLException if the auxiliary data can't be loaded
    @exception ItemException if the auxiliary data contain bad data
   */
  public Events(User user, boolean load)
      throws SQLException, ItemException
  {
    super(user, false);
    ls = new Locations(user, load);
    ss = new Sponsors(user, load);
    ks = new Keywords(user, load);
  }

  /** Initialize the set for a particular user, loading all auxiliary data
    @param user the user
    @exception SQLException if the auxiliary data can't be loaded
    @exception ItemException if the auxiliary data contain bad data
    */
  public Events(User user) throws SQLException, ItemException
  {
    this(user, true);
  }

  /** Initialize the set for a guest user, loading all auxiliary data
    @exception SQLException if the auxiliary data can't be loaded
    @exception ItemException if the auxiliary data contains bad data
    */
  public Events() throws SQLException, ItemException
  {
    this(new User());
  }

  /** Set the keywords cache
    @param ks new Keywords cache
   */
  public void setKeywords(Keywords ks)
  {
    this.ks = ks;
  }

  /** Set the Locations cache
    @param ls new Locations cache
   */
  public void setLocations(Locations ls)
  {
    this.ls = ls;
  }

  /** Set the sponsors cache
    @param ss new Sponsors cache
   */
  public void setSponsors(Sponsors ss)
  {
    this.ss = ss;
  }

  /**
    Get the set of associated locations for a user
    @return the set of associated locations for a user
   */
  public Locations getLocations() {
    return ls;
  }

  /**
    Get the set of associated sponsors for a user
    @return the set of associated sponsors for a user
   */
  public Sponsors getSponsors()
  {
    return ss;
  }

  /**
    Get the set of associated keywords for a user
    @return the set of associated keywords for a user
   */
  public Keywords getKeywords()
  {
    return ks;
  }

  protected String dateKey(MyCalendar mc)
  {
    return mc.getDateDigits();
  }

  protected String dateKey(Event e)
  {
    return dateKey(e.startCalendar());
  }

  /**
    Get the events for one day, never loading from the database

    @param date The date in question
    @return A HashMap with the events, keyed by event ids
   */
  private HashMap daysEvents(String date)
  {
    try {
      return daysEvents(date, false, false);
    } catch (ItemException e) {         // shouldn't happen
      Logger.getLogger(this.getClass()).error(
            "ItemException", e);
      //e.printStackTrace(this.log.getPrintStream());
      return new HashMap();
    } catch (SQLException e) {         // shouldn't happen
      Logger.getLogger(this.getClass()).error(
            "SQLException", e);
      //e.printStackTrace(this.log.getPrintStream());
      return new HashMap();
    }
  }

  /**
    Load all the events

    @exception SQLException if we try to load from the database and have a
      problem
    @exception ItemException if we try to load from the database and
      it holds bad data
   */
  protected void doLoadAll()
      throws SQLException, ItemException
  {
    getCaldata().loadEvents(this.user, null, null, this, ls, ss, ks);
  }

  /**
    Have the events on a particular date been loaded from the database?
    @param date The date in question
    @return Have the events on the date been loaded from the database?
   */
  protected boolean dateLoaded(String date)
  {
    return isLoaded() || this.loadedDays.containsKey(date);
  }

  /** A bogus value to put in a hashtable */
  private static final String DUMMY = "";

  /**
    Get the events for one day
    @param data The date in question
    @param load If the events for this day have not already been loaded,
      do we load them from the database?
    @param loadAllInfo If the events are loaded, do we load all associated info?
    @return A HashMap with the events, keyed by event ids
    @exception SQLException if we try to load from the database and have a
      problem
    @exception ItemException if we try to load from the database and
      it holds bad data
   */
  private HashMap daysEvents(String date, boolean load, boolean loadAllInfo)
    throws SQLException, ItemException
  {
    if (!days.containsKey(date)) {
      HashMap newDay = new HashMap();
      days.put(date, newDay);
    }

    if (load && !dateLoaded(date)) {
      getCaldata().loadDaysEvents(this.user,
                                  date, this, ls, ss, ks, loadAllInfo);
      this.loadedDays.put(date, DUMMY);
    }

    return (HashMap) days.get(date);
  }

  private HashMap daysEvents(Event e)
  {
    return daysEvents(dateKey(e));
  }

  /**
    Invalidate the sort for a date
    @param date Date to invalidate
   */
  private void invalidateSort(String date)
  {
    this.sortedDays.remove(date);
  }

  /**
    Add an Event to this set, but not the database

    @param c The <code>Event</code> to add.  There is no effect
      if c is not an <code>Event</code>
    @exception ItemAccessException if the item can't be added to the cache
   */
  protected void localAdd(CalendarObject c) throws ItemAccessException
  {
    if (!(c instanceof Event)) {
      return;
    }

    Event e = (Event) c;
    super.localAdd(e);

    Iterator dates = e.getDates();

    while (dates.hasNext()) {
      String nextDate = (String) dates.next();
      daysEvents(nextDate).put(key(e), e);
      invalidateSort(nextDate);
    }

    if (e.getRecurrence().isMaster()) {
      ((MasterRecurrence) e.getRecurrence()).addInstance(e);
    }

    /* Some events can be added with the assumption that their location
       and sponsor were already loaded (because all locations and sponsors
       were loaded.  But if the event was entered after the locations and
       sponsors were loaded, this might not be true.
       So, try to ensure the location and sponsor were loaded.
       But don't worry if there's a problem; Event.get{Location,Sponsor} will
       handle that case
     */
    try {
      this.ls.ensureLoaded(e.getLocationid());
    } catch (ItemException e2) {
    } catch (SQLException e2) {
    }

    try {
      this.ss.ensureLoaded(e.getSponsorid());
    } catch (ItemException e2) {
    } catch (SQLException e2) {
    }
  }

//  /**
//    delete an event from the Cache
//
//    @param c the item to delete
//    @exception ItemException if there's a problem with the item
//    @exception SQLException if there is a problem with the database
//   */
//  public void delete(CalendarObject c) throws SQLException, ItemException
//  {
//    if (((Event) c).isRecurring()) {
//      if (!((Event) c).getRecurrence().isMaster()) {
//        throw new ItemException(
//            "cannot delete one instance of a recurring event");
//      } else {
//        deleteInstances((Event) c);
//      }
//    }
//
//    super.delete(c);
//  }

  /**
    Do the work of deleting an Event from this set

    @param e The <code>Event</code> to delete.
    @exception ItemException if there's a problem with the event
   */
  private void doLocalDelete(Event e) throws ItemException
  {
    super.localDelete(e);
    Iterator dates = e.getDates();

    while (dates.hasNext()) {
      String nextDate = (String) dates.next();
      daysEvents(nextDate).remove(key(e));
      invalidateSort(nextDate);
    }
  }

  /**
    Delete an Event from this set, but not the database.

    @param c The <code>Event</code> to delete.  There is no effect
      if c is not an <code>Event</code>
    @exception ItemException if there's a problem with the item chosen
   */
  protected void localDelete(CalendarObject c) throws ItemException
  {
    if (!(c instanceof Event)) {
      return;
    }

    if (((Event) c).isRecurring()) {
      if (!((Event) c).getRecurrence().isMaster()) {
        throw new ItemException(
            "cannot delete one instance of a recurring event");
      } else {
        localDeleteInstances((Event) c);
      }
    }

    doLocalDelete((Event) c);
  }

  /**
    Given a recurring event, delete all instances of it (except the master
      instance)
    @param e 'master' event of the recurring event
    @exception ItemException If there's a problem with the event chosen
   */
  private void localDeleteInstances(Event e) throws ItemException
  {
    if (!e.getRecurrence().isMaster()) {
      throw new ItemException("Not a recurring event");
    }

    Iterator events = elements();

    while (events.hasNext()) {
      Event e2 = (Event) events.next();

      if (e2.isInstance(e.getId()) && !(e2.getId() == e.getId())) {
        try {
          // don't call localDelete, as it can't handle instances
          doLocalDelete(e2);
        } catch (NoSuchItemException ex) {
          Logger.getLogger(this.getClass()).error(
                "No such item, but we just found it", ex);
          throw new RuntimeException("No such item, but we just found it");
        }
      }
    }
  }

  /**
    Get the event in the set with the given id

    @return the event in the set with the given id
    @param eventID Unique ID of the event
    @exception NoSuchItemException if no such event exists
   */
  public Event getEvent(int eventID) throws NoSuchItemException
  {
    return (Event) get(eventID);
  }

  /** Get the event in the set with the given id. Optionally load it into
   * the cache if it's not already there.
   *
   * @param eventID    int Unique ID of the event
   * @param loadIt     load the event if not in cache
   * @return Event     from the set with the given id
   * @exception NoSuchItemException if no such event exists
   */
  public Event getEvent(int eventID,
                        boolean loadIt) throws NoSuchItemException {
    if (loadIt) {
      try {
        ensureLoaded(eventID);
      } catch (Throwable t) {
        // I guess we just have to asume no item.
        throw new NoSuchItemException(t.getMessage());
      }
    }

    return (Event) get(eventID);
  }

  /**
    Get the events in a day, in no particular order

    @param mc The day in question
    @param loadAllInfo should all associated info be loaded (if events are)?

    @return A Collection of the events
    @exception SQLException if we have to load the events from the database,
      and there is a problem
    @exception ItemException if we have to load the events from the
      database, and they contain bad data
   */
  private HashMap oneDaysUnsortedEvents(MyCalendar mc, boolean loadAllInfo)
    throws SQLException, ItemException
  {
    return daysEvents(dateKey(mc), true, loadAllInfo);
  }

  /**
    @return The events represented by a Vector, as an array
    @param v The vector
   */
  protected Event[] eventArray(Vector v)
  {
    int i = 0;
    Event[] ea = new Event[v.size()];
    Iterator it = v.iterator();

    for (; i < ea.length; i++) {
      ea[i] = (Event) it.next();
    }

    return ea;
  }

  /**
    Get the events for a user on a given day, sorted in an appropriate way,
    as a Vector

    @param mc Date, represented as a <code>MyCalendar</code>
    @param loadAllInfo should all associated info be loaded (if the events are?)

    @return the events for a user on a given day, as a sorted Vector
    @exception SQLException if there is a problem loading from the database
    @exception ItemException if the database contains bad data
   */
  protected Vector oneDaysVEvents(MyCalendar mc, boolean loadAllInfo)
      throws SQLException, ItemException
  {
    Vector v = (Vector) this.sortedDays.get(dateKey(mc));

    if (v == null) {
      v = new Vector(oneDaysUnsortedEvents(mc, loadAllInfo).values());
      Collections.sort(v);
      this.sortedDays.put(dateKey(mc), v);
    }

    return v;
  }

  /**
    Get the events for a user on a given day, sorted in an appropriate
    way, as an array

    @param mc Date, represented as a <code>MyCalendar</code>
    @param loadAllInfo should all associated info be loaded (if the events are?)

    @return the events for a user on a given day, as a sorted array
    @exception SQLException if there is a problem loading from the database
    @exception ItemException if the database contains bad data
  */
  public Event[] oneDaysEvents(MyCalendar mc, boolean loadAllInfo)
      throws SQLException, ItemException
  {
    return eventArray(oneDaysVEvents(mc, loadAllInfo));
  }

  /**
    Get the events for a user on a given day, sorted in an appropriate
    way, as an array

    @param mc Date, represented as a <code>MyCalendar</code>
    @return the events for a user on a given day, sorted
    @exception SQLException if there is a problem loading from the database
    @exception ItemException if the database contains bad data
  */
  public Event[] oneDaysEvents(MyCalendar mc)
      throws SQLException, ItemException
  {
    return oneDaysEvents(mc, true);
  }

  /**
    Get the events for a user over a series of days, sorted in an appropriate
    way

    @param startDate first day in the series, represented as a
        <code>MyCalendar</code>
    @param endDate last day in the series, represented as a
        <code>MyCalendar</code>

    @return the events for a user over the series of days, sorted
    @exception SQLException if there is a problem loading from the database
    @exception ItemException if the database contains bad data
  */
  public List manyDaysEvents(MyCalendar startDate, MyCalendar endDate)
      throws SQLException, ItemException
  {
    prepareManyDays(startDate, endDate);
    TreeSet s = new TreeSet();

    MyCalendar current = new MyCalendar(startDate.getDateDigits());

    while (!endDate.isEarlierDay(current)) {
      s.addAll(oneDaysUnsortedEvents(current, true).values());
      current = current.tomorrow();
    }

    return new Vector(s);
  }

  /**
    Add an empty day to our collection of days
    @mc The day in question
   */
  private void addEmptyDay(MyCalendar mc)
  {
    daysEvents(dateKey(mc));
  }

  private MyCalendar nextLoadedDay(MyCalendar current, MyCalendar end)
  {
    while (!end.isEarlierDay(current) && !dateLoaded(dateKey(current)))
    {
      current = current.tomorrow();
    }

    return current;
  }

  private MyCalendar nextUnloadedDay(MyCalendar current, MyCalendar end)
  {
    while (!end.isEarlierDay(current) && dateLoaded(dateKey(current)))
    {
      current = current.tomorrow();
    }

    return current;
  }

  private void markLoaded(MyCalendar start, MyCalendar end)
  {
    MyCalendar current;

    for (current = start; !end.isEarlierDay(current);
         current = current.tomorrow())
    {
      this.loadedDays.put(dateKey(current), DUMMY);
    }
  }

  /**
    Alert the object that events from many days are about to be
    requested.  This need not be called, but should speed up a
    series of calls to <code>oneDaysEvents()</code> if it is

    @param start The first date in a series of dates
    @param end The last date in a series of dates
    @exception SQLException if we have trouble loading data
    @exception ItemException if the data we try to load is bad
  */
  public void prepareManyDays(MyCalendar start, MyCalendar end)
      throws SQLException, ItemException
  {
    // copy original start calendar to avoid side effects
    MyCalendar currentStart = new MyCalendar(start.getDateDigits());

    // heuristic for now:  one request for each series of unloaded days
    while (!end.isEarlierDay(currentStart)) {
      currentStart = nextUnloadedDay(currentStart, end);
      MyCalendar currentEnd = nextLoadedDay(currentStart, end).yesterday();
//new this.log.println(dateKey(currentStart) + " " + dateKey(currentEnd));

      if (!currentEnd.isEarlierDay(currentStart)) {
        getCaldata().loadEvents(this.user, dateKey(currentStart),
                                 dateKey(currentEnd), this, ls, ss, ks);
        markLoaded(currentStart, currentEnd);
      }

      currentStart = currentEnd.tomorrow();
    }
  }

  /**
    Alert the object that events from many days are about to be
    requested.  This need not be called, but should speed up a
    series of calls to <code>oneDaysEvents()</code> if it is

    @param start The first date in a series of dates
    @param end The last date in a series of dates
    @exception SQLException if we have trouble loading data
    @exception ItemException if the data we try to load is bad
  */
  public void prepareManyDays(Calendar start, Calendar end)
      throws SQLException, ItemException
  {
    prepareManyDays(new MyCalendar(start), new MyCalendar(end));
  }

//  /** load a result set containing events for one day
//      @param mc The day in question
//      @param rs The <code>ResultSet</code> containing the events
//      @exception SQLException if there is database trouble
//      @exception CaldataException if the data we get back is bad
//    */
//  public void load(MyCalendar mc, ResultSet rs)
//      throws SQLException, CaldataException
//  {
//    addEmptyDay(mc);
//    getCaldata().loadEvents(rs, this, ls, ss);
//    markLoaded(mc, mc);
//  }

  /**
    Was this event loaded by this cache?
    @param e Event
    @return Was this event loaded by this cache?
   */
  protected boolean loadedByThisCache(Event e)
  {
    return e.getCreator().equals(this.user) && !e.isPublic();
  }

  /**
    Is there a recurring event during a given series of days in the cache
    (We don't count events that were loaded from another cache)?

    @param startDate First day in the series
    @param endDate Last day in the series
    @return Is there a recurring event during a given series of days?
    @exception CaldataException If the dates are in a bad format
   */
  public boolean containsRecurringEvent(String startDate, String endDate)
      throws CaldataException
  {
    MyCalendar endCalendar = new MyCalendar(endDate);

    for (MyCalendar currentDate = new MyCalendar(startDate);
         !endCalendar.isEarlierDay(currentDate);
         currentDate = currentDate.tomorrow())
    {
      Iterator events = daysEvents(dateKey(currentDate)).values().iterator();

      while (events.hasNext()) {
        Event e = (Event) events.next();

        if (e.isRecurring() && loadedByThisCache(e)) {
          return true;
        }
      }
    }

    return false;
  }

  /**
    Load a single event into the cache
    @param id Id of the item to load
    @exception SQLException If there's a problem accessing the database
    @exception CaldataException If there's a problem with the data in the db
    @exception NoSuchItemException If the item does not exist
    @exception ItemAccessException If the item can't be added to the cache
   */
  protected void loadOne(int id)
      throws SQLException, CaldataException, NoSuchItemException,
             ItemAccessException
  {
    getCaldata().loadOneEvent(id, this);
  }

  /**
    Ensure that an event is loaded in the proper cache
    @param eventid Id of the event to load
    @param isPublic Is this a public event?
    @exception SQLException If there's a problem accessing the database
    @exception CaldataException If there's a problem with the data in the db
    @exception NoSuchItemException If the event does not exist
    @exception ItemAccessException If the event can't be loaded
   */
  protected void ensureLoaded(int eventid, boolean isPublic)
      throws SQLException, CaldataException, NoSuchItemException,
             ItemAccessException
  {
    if (isPublic) {
      PublicEvents.getPublicEvents().ensureLoaded(eventid);
    } else {
      ensureLoaded(eventid);
    }
  }

  /**
    Add an instance of a recurrence
    Also, if the master event for the recurrence is not loaded yet, load it.

    @param masterid ID of the master event for the instance
    @param instanceid ID of the instance event
    @param rs Result set containing all the information about the instance

    @exception SQLException If the master has to be loaded, and there's a
       problem with the database
    @exception CaldataException If the master has to be loaded, and
       there's a problem with the data in the db
    @exception NoSuchItemException If the master event doesn't exist
    @exception ItemAccessException If the event can't be added to the cache
   */
  public void addRecurInstance(int masterid, int instanceid, ResultSet rs)
      throws SQLException, CaldataException, NoSuchItemException,
             ItemAccessException
  {
    ensureLoaded(masterid);
    Event e;

    try {
      e = getEvent(masterid);
    } catch (NoSuchItemException ex) {
      throw new IllegalStateException("can't find event " + masterid +
                                      ", but just called ensureLoaded()");
    }

    localAdd(new InstanceEvent(rs, e));

    try {
      e.addRecurInstance(getEvent(instanceid));
    } catch (NoSuchItemException ex) {
      throw new IllegalStateException("can't find event " + instanceid +
                                      ", but we just added it");
    }
  }

  /**
    Invalidate all cached dates.  Only the flags that indicate which days
    have been loaded and sorted are invalidated; actual events are not
    affected.
   */
  protected void invalidateAllDays()
  {
    this.loadedDays = new HashMap();
    this.sortedDays = new HashMap();
  }

  /**
    Get number of loaded days.  Used by Unit Tests
    @return number of loaded days
   */
  protected int numberOfLoadedDays()
  {
    return this.loadedDays.size();
  }

  /**
    Get number of sorted days.  Used by Unit Tests
    @return number of sorted days
   */
  protected int numberOfSortedDays()
  {
    return this.sortedDays.size();
  }
}
