// java

package edu.washington.cac.calendar.data;

import java.sql.Date;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.Calendar;
import java.util.Iterator;
import java.util.Vector;

import org.apache.log4j.Logger;

import edu.washington.cac.calendar.MyCalendar;
import edu.washington.cac.calendar.db.Caldata;
import edu.washington.cac.calendar.db.DBResources;
import edu.washington.cac.calendar.db.Fields;
import edu.washington.cac.calendar.db.WhereClauses;

/** An Event in UWCal
    @author Greg Barnes
    @version 2.11 slh
  */
public class Event extends CalendarObject implements LinkEntity, LongdescEntity
{
  private int eventid;
  private String shortdesc, longdesc;
  private java.sql.Date startdate, enddate;
  private Time starttime, endtime;
  private String link, cost;

  private int locationid;
  private int sponsorid;

  /** Information about whether this event is a recurring event, and
    if it is, how it recurs
   */
  private Recurrence recur = NoRecurrence.NO_RECUR;

  private Vector keywords = new Vector();

  private int seq;

  private Timestamp lastmod;

  private Timestamp created;

  private CaldataWarning w = null;

  /** Create an event by specifying all its fields

      @param eventid Unique ID for the event
      @param shortdesc Short description
      @param longdesc Long description
      @param startdate Start Date
      @param starttime Starting Time (can be null)
      @param enddate End Date (can be null)
      @param endtime End Time (can be null)
      @param isPublic Is the event viewable by everyone?
      @param link URL with more info
      @param creator The person who created the event
      @param cost Cost to attend
      @param recur Recurrence information
      @param locationid Unique ID of the event's location
      @param sponsorid Unique ID of the event's sponsor
      @param seq          int sequence number
      @param lastmod      Timestamp last modification time
      @param created      Timestamp creation time
      @exception CaldataException if the fields are not valid in some way
    */
  public Event(int eventid,
               String shortdesc,
               String longdesc,
               java.sql.Date startdate,
               Time starttime,
               java.sql.Date enddate,
               Time endtime,
               boolean isPublic,
               String link,
               User creator,
               String cost,
               Recurrence recur,
               int locationid,
               int sponsorid,
               int seq,
               Timestamp lastmod,
               Timestamp created)
      throws CaldataException
  {
    super(creator, isPublic);
    DataChecker.checkShortdesc(shortdesc);

    try {
      DataChecker.checkDateTimes(startdate, starttime, enddate, endtime);
    } catch (CaldataWarning x) {
      Logger.getLogger(this.getClass()).warn(
            "id: " + eventid, x);
      w = x;

      if (w instanceof EndBeforeStartWarning ||
          w instanceof NoEnddateWarning)
      {
        enddate = startdate;
        endtime = null;
      }
    }

    this.eventid = eventid;
    this.shortdesc = shortdesc;
    this.longdesc = longdesc;
    this.startdate = startdate;
    this.starttime = starttime;
    this.enddate = enddate;
    this.endtime = endtime;
    this.link = link;
    this.cost = cost;
    this.recur = recur;
    this.locationid = locationid;
    this.sponsorid = sponsorid;
    this.seq = seq;
    this.lastmod = lastmod;
    this.created = created;

    normalizeDatesAndTimes();
  }

  /** Create an event by specifying all its fields

      @param eventid Unique ID for the event
      @param shortdesc Short description
      @param longdesc Long description
      @param startdate Start Date
      @param starttime Starting Time (can be null)
      @param enddate End Date (can be null)
      @param endtime End Time (can be null)
      @param isPublic Is the event viewable by everyone?
      @param link URL with more info
      @param creator The person who created the event
      @param cost Cost to attend
      @param location the event's location
      @param sponsor the event's sponsor
      @param seq          int sequence number
      @param lastmod      Timestamp last modification time
      @param created      Timestamp creation time
      @exception CaldataException if the fields are not valid in some way
    */
  public Event(int eventid,
               String shortdesc,
               String longdesc,
               java.sql.Date startdate,
               Time starttime,
               java.sql.Date enddate,
               Time endtime,
               boolean isPublic,
               String link,
               User creator,
               String cost,
               Recurrence recur,
               Location location,
               Sponsor sponsor,
               int seq,
               Timestamp lastmod,
               Timestamp created)
    throws CaldataException
  {
    this(eventid, shortdesc, longdesc, startdate, starttime, enddate,
         endtime, isPublic, link, creator, cost, recur,
         location != null ? location.getId() : Location.NO_LOCATION_ID,
         sponsor != null ? sponsor.getId() : Sponsor.NO_SPONSOR_ID,
         seq, lastmod, created);
  }

  /** Create an event corresponding to the current row in a given
        <code>java.sql.ResultSet</code> of events

      @param rs The <code>ResultSet</code> holding the event

      @exception SQLException if there's trouble accessing the
       <code>ResultSet</code>
      @exception CaldataException if the fields are not valid in some way
    */
  public Event(ResultSet rs) throws SQLException, CaldataException
  {
    this(rs.getInt(Fields.EEVENT_ID.getName()),
         rs.getString(Fields.ESHORTDESC.getName()),
         Fields.getEventLongdesc(rs),
         rs.getDate(Fields.ESTARTDATE.getName()),
         rs.getTime(Fields.ESTARTTIME.getName()),
         rs.getDate(Fields.EENDDATE.getName()),
         rs.getTime(Fields.EENDTIME.getName()),
         rs.getString(Fields.EPUBLIC.getName()).equals(WhereClauses.TRUE),
         rs.getString(Fields.ELINK.getName()),
         new User(rs.getString(Fields.ECREATOR.getName())),
         rs.getString(Fields.ECOST.getName()),
         Recurrence.create(rs.getString(Fields.ERECURRING_STATUS.getName()),
                           rs.getInt(Fields.EEVENT_ID.getName())),
         rs.getInt(Fields.ELOCATION_ID.getName()),
         rs.getInt(Fields.ESPONSOR_ID.getName()),
         rs.getInt(Fields.ESEQ.getName()),
         rs.getTimestamp(Fields.ELASTMOD.getName()),
         rs.getTimestamp(Fields.ECREATED.getName()));
  }

  /**
    Set recurrence information for this event
    @param recur Recurrence information
   */
  public void setRecurrence(Recurrence recur)
  {
    this.recur = recur;
  }

  /**
    Get recurrence information for this event
    @return Reccurrence information
   */
  public Recurrence getRecurrence()
  {
    return this.recur;
  }

  /**
    Is this a recurring event?
    @return Is this a recurring event?
   */
  public boolean isRecurring()
  {
    return this.recur.recurs();
  }

  /**
    Set the id for this event
    @param eventid Event id
   */
  public void setId(int eventid)
  {
    this.eventid = eventid;
  }

  /* the start and end times supplied are supposed to have a date
     of January 1, 1970.  This does not always seem to be true.

     Similarly, the dates are supposed to have the hour, minute, second
     and milliseconds set to 0, but I have reason to believe this may not
     be the case with all JVMs.

     So, here we normalize both the dates and times so that they can
     be compared.
   */
  private void normalizeDatesAndTimes()
  {
    this.startdate = MyCalendar.normalizedDate(startdate);
    this.enddate = MyCalendar.normalizedDate(enddate);
    this.starttime = MyCalendar.normalizedTime(starttime);
    this.endtime = MyCalendar.normalizedTime(endtime);
  }

  public CaldataWarning getWarnings()
  {
    return w;
  }

  /**
      Get the event's short description
      @return The event's short description
    */
  public String getShortdesc()
  {
    return shortdesc;
  }

  /**
      Get the event's long description
      @return The event's long description
    */
  public String getLongdesc()
  {
    return longdesc;
  }

  /** The 'hour' of an event that doesn't start at any particular hour,
      (i.e., has no start time)
    */
  public int NO_START_HOUR = -1;

  private int startHour()
  {
    return starttime == null ? NO_START_HOUR :
           new MyCalendar(starttime).get(Calendar.HOUR_OF_DAY);
  }

  /**
    Get this event's start hour (0-23) relative to another day.

    @param relativeToDay Day to compare to determine if this is the
          'same day'
    @return This event's start hour (0-23) relative to another day.
          If this event has not start time, or doesn't start on the
          same day, returns NO_START_HOUR
   */
  public int startHour(MyCalendar relativeToDay)
  {
    return startCalendar().isSameDay(relativeToDay) ? startHour() :
                                                      NO_START_HOUR;
  }

  private MyCalendar myCalendar(java.util.Date date, Time time)
  {
    if (time == null && date == null) {
      return null;
    } else {
      return new MyCalendar(date, time);
    }
  }

  /**
    Get the event's starting day and time

    @return A <code>MyCalendar</code> corresponding to the event's
          starting day and time
   */
  public MyCalendar startCalendar()
  {
      return myCalendar(startdate, starttime);
  }

  /**
    Get the event's ending day and time

    @return A <code>MyCalendar</code> corresponding to the event's
      ending day and time
   */
  public MyCalendar endCalendar()
  {
    return myCalendar(enddate, endtime);
  }

  /** Get the event's starting date (not time)
    @return The event's starting date (not time)
   */
  public java.sql.Date getStartdate()
  {
    return startdate;
  }

  /** Get the event's starting time (not date)
    @return The event's starting time (not date)
   */
  public Time getStarttime()
  {
    return starttime;
  }

  /**
    Get the event's starting date as a <code>String</code>
    @return The event's starting date as a <code>String</code>
   */
  public String getStarttimeString()
  {
    return startCalendar().timeString();
  }

  /**
    Get the event's ending date (not time)
    @return The event's ending date (not time)
   */
  public java.sql.Date getEnddate()
  {
    return enddate;
  }

  /**
    Get the event's ending time (not date)
    @return The event's ending time (not date)
   */
  public Time getEndtime()
  {
    return endtime;
  }

  /**
    Get the event's ending date as a <code>String</code>
    @return The event's ending date as a <code>String</code>
   */
  public String getEndtimeString()
  {
    return endCalendar().timeString();
  }

  /**
    Get the event's URL
    @return the event's URL
   */
  public String getLink()
  {
    return this.link;
  }

  /**
    Set the event's URL
    @param link The new URL
   */
  public void setLink(String link)
  {
    this.link = link;
  }

  /**
    Get the event's cost
    @return the event's cost
   */
  public String getCost()
  {
    return cost;
  }

  /**
    Get the event's unique id
    @return the event's unique id
   */
  public int getId()
  {
    return eventid;
  }

  /**
    Get the unique id of the event's location
    @return the unique id of the event's location
   */
  public int getLocationid()
  {
    return locationid;
  }

  /**
    Get the unique id of the event's sponsor
    @return the unique id of the event's sponsor
   */
  public int getSponsorid()
  {
    return sponsorid;
  }

  /**
    @return If the event has a particular keyword
    @param k Keyword to test
   */
  public boolean hasKeyword(Keyword k)
  {
    return keywords.contains(new Integer(k.getId()));
  }

  /**
    Get an <code>Iterator</code> of the event's keywords
    @return an <code>Iterator</code> of the event's keywords
   */
  public Iterator getKeywords()
  {
    return keywords.iterator();
  }

  /** Add a keyword to the set of keywords associated with this event

    @param id The unique ID of the keyword to add
   */
  public void addKeyword(int id)
  {
    if (!keywords.contains(new Integer(id))) {
      keywords.addElement(new Integer(id));
    }
  }

  /** Delete any of this event's keywords that aren't in another
      set of keywords

      @param ks The set of Keywords to check against
    */
  public void deleteMissingKeywords(Keywords ks)
  {
    int i;

    for (i = 0; i < keywords.size(); i++) {
      if (!ks.containsKey(keywords.elementAt(i))) {
        keywords.removeElementAt(i);
        i--;        // everything gets shifted down 1 by removeElementAt()
      }
    }
  }

  /**
    Get the event's location

    @param ls The set of Locations
    @return the event's location
   */
  public Location getLocation(Locations ls)
  {
    try {
      return ls.getLocation(locationid);
    } catch (NoSuchItemException e) {
      // probably deleted, but print a message to the log
      Logger.getLogger(this.getClass()).warn(
            "Event.getLocation: location " + locationid +
            " not found for event " + getId());
      locationid = Location.DELETED_LOCATION_ID;

      try {
        ls.localAdd(Location.DELETED_LOCATION);
      } catch (ItemAccessException e2) {
        Logger.getLogger(this.getClass()).error(
              "Couldn't add deleted location!", e2);
        throw new RuntimeException("Couldn't add deleted location!");
      }

      try {
        return ls.getLocation(locationid);
      } catch (NoSuchItemException e2) {
        Logger.getLogger(this.getClass()).error(
              "no location, but we just added it", e2);
        throw new RuntimeException("no location, but we just added it");
      }
    }
  }

  /**
    Set the event's location

    @param l The Location
   */
  public void setLocation(Location l)
  {
    this.locationid = l.getId();
  }

  /**
    Get the event's location

    @param es The set of Events, which
       contains a handle for the set of Locations
    @return the event's location
   */
  public Location getLocation(EventsI es)
  {
    return getLocation(es.getLocations());
  }

  /**
    Get the event's sponsor

    @param ss The set of Sponsors
    @return the event's sponsor
   */
  public Sponsor getSponsor(Sponsors ss)
  {
    try {
      return ss.getSponsor(sponsorid);
    } catch (NoSuchItemException e) {
      // probably deleted, but print a message to the log
      Logger.getLogger(this.getClass()).warn(
            "Event.getSponsor: sponsor " + sponsorid +
            " not found for event " + getId());
      sponsorid = Sponsor.DELETED_SPONSOR_ID;

      try {
        ss.localAdd(Sponsor.DELETED_SPONSOR);
      } catch (ItemAccessException e2) {
        Logger.getLogger(this.getClass()).error(
              "Should be able to add deleted sponsor!", e2);
        throw new RuntimeException("Should be able to add deleted sponsor!");
      }

      try {
        return ss.getSponsor(sponsorid);
      } catch (NoSuchItemException e2) {
        Logger.getLogger(this.getClass()).error(
              "no sponsor, but we just added it", e2);
        throw new RuntimeException("no sponsor, but we just added it");
      }
    }
  }

  /**
    Get the event's sponsor

    @param es The set of Events, which contains a handle for the set of
      sponsors
    @return the event's sponsor
   */
  public Sponsor getSponsor(EventsI es)
  {
    return getSponsor(es.getSponsors());
  }

  /** Set the seq for this event
   *
   * @param val   int sequence number
   */
  public void setSeq(int val) {
    seq = val;
  }

  /** Get the events seq
   *
   * @return int    the events seq
   */
  public int getSeq() {
    return seq;
  }

  public void setLastmod(Timestamp val) {
    lastmod = val;
  }

  public Timestamp getLastmod() {
    return lastmod;
  }

  public void setCreated(Timestamp val) {
    created = val;
  }

  public Timestamp getCreated() {
    return created;
  }

  /** delete this event for the user in the database.

    @param user The user requesting the deletion.
    @exception SQLException if there is a database problem
   */
  protected void delete(User user) throws SQLException
  {
//    new CalLogFactory().create(user.getName()).println(
//        "Deleting event " + getId() + " for " + user.getName());
    getCaldata().delete(getId(), user);
  }

  /** replace any event in the database with the same id as this event
        with this event

      @param user The user requesting the replace.
        There should be no effect if this does not match the old event's
        creator.

      @exception SQLException if there is a database problem
    */
  protected void replace(User user) throws SQLException
  {
//  new CalLogFactory().create(user.getNameDB()).println("Replacing event " + getId() + " for " + user.getNameDB());
    getCaldata().replace(this, user);
  }

  /** add this event to the database

    @param user The UWNetID of the user requesting the add.
           This will become the event's creator in the database

    @exception SQLException if there is a database problem
   */
//  protected void add(User user) throws SQLException
/* Make this public again until the problems with access are fixed.MARD
 */
  public void add(User user) throws SQLException
  {
//new CalLogFactory().create(user.getNameDB()).println("Adding event for " + user.getNameDB());
    eventid = getCaldata().add(this, user);
  }

  private static void printDate(java.util.Date d)
  {
    if (d == null) {
      System.err.println(d);
    } else {
      System.err.println(d.getTime());
    }
  }

  /** compare two dates

    @param a LHS of &lt;
    @param b RHS of &lt;

    @return whether <code>a</code> is strictly less than <code>b</code>
   */
  public static boolean lessThan(java.util.Date a, java.util.Date b)
  {
//printDate(a);
//printDate(b);
      return (a == null && b != null) ||
        (a != null && b != null && a.before(b));
  }

  /**
    Test whether this event has an earlier start date than another

    @param e The other event to compare
    @return whether this event has an earlier start date than another
   */
  private boolean hasEarlierStartDateThan(Event e)
  {
    return lessThan(getStartdate(), e.getStartdate());
  }

  /**
    Test whether this event has an earlier end date than another
    @param e The other event to compare
    @return whether this event has an earlier end date than another
   */
  private boolean hasEarlierEndDateThan(Event e)
  {
      return lessThan(getEnddate(), e.getEnddate());
  }

  /**
    Test whether this event has an earlier start time than another
    @param e The other event to compare
    @return whether this event has an earlier start time than another
   */
  private boolean hasEarlierStartTimeThan(Event e)
  {
    return lessThan(getStarttime(), e.getStarttime());
  }

  /**
    Test whether this event has an earlier end time than another
    @param e The other event to compare
    @return whether this event has an earlier end time than another
   */
  private boolean hasEarlierEndTimeThan(Event e)
  {
    return lessThan(getEndtime(), e.getEndtime());
  }

  /**
    Test whether this event is less than another event

    @param c The other event to compare.  If c is a
       <code>CalendarObject</code> that is not an event, the method returns
       a consistent but useless result (by comparing the toString values
       of both objects)
    @return whether this event is less than another event
   */
  public boolean lessThan(CalendarObject c)
  {
    if (!(c instanceof Event)) {
      return toString().compareTo(c.toString()) < 0;
    }

    Event e = (Event) c;

    if (hasEarlierStartDateThan(e)) {
      return true;
    } else if (e.hasEarlierStartDateThan(this)) {
      return false;
    } else if (hasEarlierStartTimeThan(e)) {
      return true;
    } else if (e.hasEarlierStartTimeThan(this)) {
      return false;
    } else if (hasEarlierEndDateThan(e)) {
      return true;
    } else if (e.hasEarlierEndDateThan(this)) {
      return false;
    } else if (hasEarlierEndTimeThan(e)) {
      return true;
    } else if (e.hasEarlierEndTimeThan(this)) {
      return false;
    } else if (getShortdesc().compareToIgnoreCase(e.getShortdesc()) < 0) {
      return true;
    } else if (e.getShortdesc().compareToIgnoreCase(getShortdesc()) < 0) {
      return false;
    } else {
      /* assumably these can never be the same, so the final test */
      return getId(  ) < e.getId(  );
    }
  }

  /** A simple iterator over the dates an event takes place
   */
  private class Dates implements Iterator
  {
    /** The next date in the iteration */
    MyCalendar nextDate;

    /**
      Create a new instance
     */
    public Dates()
    {
      nextDate = startCalendar();
    }

    /**
      Is there another date left?
      @return is there another date left?
     */
    public boolean hasNext()
    {
      return !endCalendar().isEarlierDay(nextDate);
    }

    /**
      Get the next date
      @return the next date
     */
    public Object next()
    {
      String rVal = nextDate.getDateDigits();
      nextDate = nextDate.tomorrow();
      return rVal;
    }

    /**
      not supported
      @exception UnsupportedOperationException always thrown
     */
    public void remove() throws UnsupportedOperationException
    {
      throw new
          UnsupportedOperationException("Can't remove a date from an event");
    }
  }

  /**
    Get all the dates this event occurs.

    @return an <code>Iterator</code> of all the dates this
      event occurs.  Each data is represented by a
      <code>MyCalendar</code>
   */
  public Iterator getDates()
  {
    return new Dates();
  }

  /**
    Is this an instance of a recurring event
    @param masterId master id of the recurring event
    @return Is the event an instance of the recurring event
   */
  boolean isInstance(int masterId)
  {
     return this.recur.getMasterId() == masterId;
  }

  /**
    Add an instance to the list of instances of a master event for a recurrence
    @param e instance event
   */
  void addRecurInstance(Event e)
  {
    ((MasterRecurrence) this.recur).addInstance(e);
  }

  /**
    Was this event loaded by default?

    Whenever a user requests 'all' events on a certain day, some events
    are not loaded.  Events that are loaded are said to be 'loaded by
    default.'  (The other events can still be loaded by specific request).

    @return Was this event loaded by default?
   */
  boolean wasLoadedByDefault()
  {
    return !this.creator.equals(DBResources.notToBeLoadedByDefaultCreator());
  }
}
