// java

package edu.washington.cac.calendar.data;

import java.sql.Date;
import java.sql.Time;
import java.util.Calendar;

import org.apache.log4j.Logger;

import edu.washington.cac.calendar.MyCalendar;

/** A class to check various data fields for consistency and correctness
   @author Greg Barnes
   @version 1.1
  */
public class DataChecker {
  /** These are also in
        uwcal.out.forms.FormsDisplayText
     The back end code should have no dependencies on applications.
     For the time being duplicate it here
   */
  public static final String ADDRESS = "Main Address";
  public static final String NAME = "Title/Summary";

  static final int DATE_STRING_LENGTH = 8;
  static final int YEAR_START_INDEX = 0;
  static final int MONTH_START_INDEX = 4;
  static final int DAY_START_INDEX = 6;

  /**
   check that a date string, purportedly in YYYYMMDD format, actually is
   and represents a valid date.

   Note that not all errors are flagged.  In particular, days that
   are too large for a given month (e.g., Feb 30) or months too large
   for a given year (not possible in the Gregorian calendar, but
    perhaps in others) are not flagged as long as the day/month
   represent valid values in *some* month/year.  These 'overflow'
   dates are handled per the explanation in the
   <code>java.util.Calendar</code> documentation (e.g., Feb 30, 1999
   becomes Mar 2, 1999).

   @param dateString The string to check
   @return The original dataString
   @exception CaldataException if there's something wrong with the
      string
   */
  public static String checkDateString(String dateString)
      throws CaldataException
  {
    if (dateString == null || dateString.length() != DATE_STRING_LENGTH) {
      throw new CaldataException("Bad date: " + dateString);
    }

    for (int i = 0; i < DATE_STRING_LENGTH; i++) {
      if (dateString.charAt(i) < '0' ||
          dateString.charAt(i) > '9') {
        Logger.getLogger("edu.washington.cac.calendar.data.DataChecker").
                  error("bogus date string2: " + dateString);
        throw new CaldataException("Bad date");
      }
    }

    if (monthNum(dateString) < 0 || dayNum(dateString) < 1 ||
        monthNum(dateString) > new MyCalendar().getMaximum(Calendar.MONTH) ||
        dayNum(dateString) > new MyCalendar().getMaximum(Calendar.DATE) ||
        yearNum(dateString) < 1)        // there was no year zero
    {
      Logger.getLogger("edu.washington.cac.calendar.data.DataChecker").error(
                  "bogus date string3: " + dateString);
      throw new CaldataException("Bad date");
    }

    return dateString;
  }

    /**
     Extract the year from an eight digit date of the form YYYYMMDD

     @param eightDigitDate The eight digit date
     @return The year number
     */
  public static int yearNum(String eightDigitDate) {
     return Integer.parseInt(eightDigitDate.substring(YEAR_START_INDEX,
       YEAR_START_INDEX + 4));
  }

    /**
     Extract the month number from an eight digit date of the
     form YYYYMMDD.  Following java.util.Date, the first month
     is month number 0

     @param eightDigitDate The eight digit date
     @return The month number
     */
  public static int monthNum(String eightDigitDate) {
     return Integer.parseInt(eightDigitDate.substring(MONTH_START_INDEX,
        MONTH_START_INDEX + 2)) - 1;
  }

    /**
     Extract the day number from an eight digit date of the
     form YYYYMMDD.

     @param eightDigitDate The eight digit date
     @return The day number
     */
  public static int dayNum(String eightDigitDate) {
     return Integer.parseInt(eightDigitDate.substring(DAY_START_INDEX,
       DAY_START_INDEX + 2));
  }

    /**
     Check that a given hour and minute have reasonable values.

     @param hour A supposed hour in the day, using the 12 hour am/pm
      format
     @param minute A supposed minute in an hour
     @exception CaldataException If either value is too high or low
      */
  public static void checkTime(int hour, int minute)
    throws CaldataException
  {
    if (hour > 12 || minute > 59 || hour < 1 || minute < 0) {
      Logger.getLogger("edu.washington.cac.calendar.data.DataChecker").error(
                  "bogus hour/minute: " + hour + ":" + minute);
      throw new CaldataException("Bad time");
    }
  }

  /**
   Check that a given set of start dates and times are consistent.

   @param startdate A starting date
   @param starttime A starting time
   @param enddate An ending date
   @param endtime An ending time

   @exception CaldataException If the start date is missing
   @exception NoEnddateWarning If the end date is missing
   @exception EndBeforeStartWarning If the end date/time is before the
     start date/time
   */
  public static void checkDateTimes(Date startdate,
                                    Time starttime,
                                    Date enddate,
                                    Time endtime)
    throws CaldataException, NoEnddateWarning, EndBeforeStartWarning
  {
    if (startdate == null) {
      throw new CaldataException("Event missing start date");
    } else if (enddate == null) {
      throw new NoEnddateWarning("End date unspecified!");
    } else if ((endtime == null &&
      new MyCalendar(enddate).isEarlierDay(new MyCalendar(startdate))) ||
          (endtime != null &&
           new MyCalendar(enddate, endtime).getCalendar().
               before(new MyCalendar(startdate, starttime).getCalendar())))
    {
      throw new EndBeforeStartWarning("Event ends before it begins!");
    }
  }

  private static boolean blank(String s) {
    return s == null || s.trim().length() == 0;
  }

  private static void checkImportantString(String s, String displayName)
    throws CaldataException {
    if (blank(s)) {
      throw new CaldataException(displayName + " cannot be blank.");
    }
  }

  static void checkShortdesc(String shortdesc) throws CaldataException {
    checkImportantString(shortdesc, NAME);
  }

  static void checkAddress(String address) throws CaldataException {
    checkImportantString(address, ADDRESS);
  }

  /**
    compress spaces in a string:  any sequence of whitespace characters
    is converted to a single space.
    @param s The String to compress
    @return s, compressed
   */
  public static String compressSpace(String s) {
    StringBuffer sb = new StringBuffer(s);
    StringBuffer sb2 = new StringBuffer(s.length());
    boolean prevCharWasSpace = false;

    for (int i = 0; i < sb.length(); i++) {
      if (!Character.isWhitespace(sb.charAt(i))) {
        sb2.append(sb.charAt(i));
        prevCharWasSpace = false;
      } else {
       if (!prevCharWasSpace) {
         sb2.append(' ');
       }

        prevCharWasSpace = true;
      }
    }

    return sb2.toString();
  }

  /**
    normalize a string.  Since our strings will be used on the web,
    right now this means removing any whitespace at the beginning and
    end, and replacing any other sequence of whitespace characters with
    a single space.  If the string is blank when we're finished, return
    null.
       @param s The String to normalize
       @return the string normalized
   */
  public static String normalize(String s) {
    if (compressSpace(s).trim().length() == 0) {
      return null;
    } else {
      return compressSpace(s).trim();
    }
  }

  public static void main(String[] args) {
    System.out.println("XX" + normalize("    ") + "XX");
    System.out.println("XX" + normalize("  a  ") + "XX");
    System.out.println("XX" + normalize("   a b  c ") + "XX");
    System.out.println("XX" + normalize("   a b  c ") + "XX");
    System.out.println("XX" + normalize("a \nb\t\t\tc\r d") + "XX");
  }
}
