// java

package edu.washington.cac.calendar.data;

import java.sql.SQLException;

//import edu.washington.cac.calendar.CalLogFactory;

/** Something that can be stored in a Cache
  @author Greg Barnes
  @version 2.2
 */
public abstract class CalendarObject extends DBObject implements Comparable
{
  /** creator of the object */
  protected User creator;

  /** Access rights for the object */
  private AccessRights ar;

  /** Is the object public or not? */
  private boolean isPublic;

  /**
    Create a new instance
    @param creator the creator
    @param isPublic Is the object public or not?
   */
  protected CalendarObject(User creator, boolean isPublic)
  {
//    super(new CalLogFactory().create(creator.getNameApp()));
    super();
    this.creator = creator;
    this.isPublic = isPublic;
    this.ar = isPublic ? (AccessRights) new PublicAccessRights(this) :
                                        new PrivateAccessRights(this);
  }

  /**
    Get the creator of the object
    @return the creator of the object
   */
  public User getCreator()
  {
    return this.creator;
  }

  /**
    Is the object public?
    @return is the object public?
   */
  public boolean isPublic()
  {
    return this.isPublic;
  }

  public static final int NO_ID = -1;
  public static final int DELETED_ID = 0;

  public abstract int getId();
  protected abstract void add(User user) throws SQLException;

  /** Delete this object from the database */
  protected abstract void delete(User user) throws SQLException;

  protected abstract void replace(User user) throws SQLException;

  /** Is this object less than another
    @param c Object to compare to
    @return Is this object less than the other
   */
  public abstract boolean lessThan(CalendarObject c);

  /**
    Compares this object with the specified object for order.

    @param o Object to compare to.  If o is a CalendarObject but not of
      the same type as this object, the result will be consistent
      but not necessarily useful.
    @return a negative integer, zero, or a positive integer as this object
      is less than, equal to, or greater than the specified object.
    @exception ClassCastException If o is not a CalendarObject
   */
  public int compareTo(Object o) throws ClassCastException
  {
    CalendarObject c = (CalendarObject) o;

    if (this.lessThan(c)) {
      return -1;
    } else if (c.lessThan(this)) {
      return 1;
    } else {
      return 0;
    }
  }

  // Convenience methods to handle access rights
  /** Can a user read (search for, include in their cache) the object?
    @param user User to test
    @return Can the user read (search for, include in their cache) the object?
   */
  boolean canBeReadBy(User user)
  {
    return this.ar.canBeReadBy(user);
  }

  /** Can a user modify the object?
    @param user User to test
    @return Can the user modify the object?
   */
  public boolean canBeModifiedBy(User user)
  {
    return this.ar.canBeModifiedBy(user);
  }

  /** Can a user create the object?
    @param user User to test
    @return Can the user create the object?
   */
  boolean canBeCreatedBy(User user)
  {
    return this.ar.canBeCreatedBy(user);
  }

  /** Can a user delete the object?
    @param user User to test
    @return Can the user delete the object?
   */
  boolean canBeDeletedBy(User user)
  {
    return this.ar.canBeDeletedBy(user);
  }
}
