// java

package edu.washington.cac.calendar.data;

import java.sql.SQLException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Vector;

import edu.washington.cac.calendar.db.Caldata;
//import edu.washington.cac.calendar.CalLogFactory;
//import edu.washington.cac.calendar.logging.Log;

import org.apache.log4j.Logger;

/**
  A set of cached items
  @author Greg Barnes
  @version 1.2
 */
public abstract class Cache implements CacheI
{
  protected HashMap hash = new HashMap();

  /** The user who owns the cache */
  protected User user;

  /** Are all the elements in the cache loaded from the database? */
  private boolean isLoaded = false;

//  /** File to print messages to */
//  protected Log log;

  /* Object for logging */
  Logger logger = Logger.getLogger(this.getClass());

  /** Create a cache for a specific user
    @param user The user
    @param load Load the cache on creation?
    @exception SQLException If there's an error loading the cache
    @exception ItemException If the data loaded is bad
   */
  public Cache(User user, boolean load)
      throws SQLException, ItemException
  {
    this.user = user;
//    this.log = new CalLogFactory().create(user.getNameApp());

    if (load) {
      loadAll();
    }
  }

  /** Create a cache for a specific user, without loading it
    @param user The user
   */
  public Cache(User user)
  {
    this.user = user;
//    this.log = new CalLogFactory().create(user.getNameApp());
  }

  /**
    Create a cache for the guest
   */
  public Cache()
  {
    this(new User());
  }

  /**
    Get a Caldata object.
    @return the Caldata object
   */
  protected Caldata getCaldata()
  {
//    return new Caldata(this.log);
    return new Caldata();
  }

  /**
    Get the number of items in the Cache
    @return The number of items in the Cache
   */
  public int size()
  {
    return hash.size();
  }

  /**
    Get all the items in the cache, as an enumeration
    @return All the items in the cache
   */
  public Iterator elements()
  {
    return hash.values().iterator();
  }

  /**
    Get all the items in the cache, as a Collection
    @return All the items in the cache
   */
  public Collection values()
  {
    return hash.values();
  }

  /**
    Turn an integer id into an object suitable to be a key
    @param id The integer id
    @return the id, as an object
   */
  protected Integer key(int id)
  {
    return new Integer(id);
  }

  /**
    Get the key for a CalendarObject
    @param c The CalendarObject
    @return the key for the CalendarObject
   */
  protected Integer key(CalendarObject c)
  {
    return key(c.getId());
  }

  /**
    Add an item to the cache
    @param c item to add
    @exception ItemAccessException If the item cannot be added to this cache
   */
  protected void localAdd(CalendarObject c) throws ItemAccessException
  {
    if (!c.canBeReadBy(this.user)) {
      throw new ItemAccessException(this.user + " cannot add " + c +
                                    " to their cache.");
    }

    hash.put(key(c), c);
    invalidateSort();
  }

  /**
    Add an item to the cache, and save it to the database
    @param c Item to add
    @exception SQLException If there's a db problem
    @exception ItemAccessException If the item cannot be added to this cache
   */
  public void add(CalendarObject c) throws SQLException, ItemAccessException
  {
    add(c, true);
  }

  /**
    Add an item to the cache
    @param c Item to add
    @param saveToDB Save to database as well?
    @exception SQLException If there's a db problem
    @exception ItemAccessException If the item cannot be added to this cache
        or the database
   */
  public void add(CalendarObject c, boolean saveToDB)
    throws SQLException, ItemAccessException
  {
//this.log.println("here");
    if (saveToDB) {
      if (c.canBeCreatedBy(this.user)) {
        c.add(this.user);
      } else {
        throw new ItemAccessException(this.user + " cannot add " + c +
                                      " to the database.");
      }
    }
//this.log.println("here2");

    if (!hash.containsKey(key(c))) {
      localAdd(c);
    }
//this.log.println("here3");
  }

  /**
    Add a collection of <code>CalendarObjects</code> to the cache
    @param c Collection of items
    @param saveToDB Save to database as well?
    @exception SQLException If there's a db problem
    @exception ItemAccessException If the item cannot be added to this cache
        or the database
   */
  public void addAll(Collection c, boolean saveToDB)
      throws SQLException, ItemAccessException
  {
    Iterator i = c.iterator();

    while (i.hasNext()) {
      add((CalendarObject) i.next(), saveToDB);
    }
  }

  /**
    Delete an item from this cache, but not from the database
    @param c Item to delete
    @exception ItemException if there's a problem with the item specified
   */
  protected void localDelete(CalendarObject c) throws ItemException
  {
    if (hash.containsKey(key(c))) {
      hash.remove(key(c));
    } else {
      throw new NoSuchItemException("localDelete: couldn't find item " +
                                    key(c));
    }

    invalidateSort();
  }

  /**
    delete an item from the Cache and the database

    @param c the item to delete
    @exception SQLException if there is a problem with the database
    @exception ItemException if there's a problem with c, such as it
        doesn't exist, or this user cannot delete it
   */
  public void delete(CalendarObject c) throws SQLException, ItemException
  {
    if (hash.containsKey(key(c))) {
      /* must test whether the user can delete the *old* item, not
         *this* item
       */
      if (get(c.getId()).canBeDeletedBy(this.user)) {
        c.delete(this.user);
        localDelete((CalendarObject) hash.get(key(c)));
      } else {
        throw new ItemAccessException(this.user + " cannot delete " + c);
      }
    } else {
      throw new NoSuchItemException("delete: couldn't find item " + key(c));
    }
  }

  /**
    replace an item in the Cache

    @param c the item to replace (the current item with the same id
      as c will be replaced with c)
    @exception SQLException if there is a problem with the database
    @exception ItemException if there's a problem with the item specified
   */
  public void replace(CalendarObject c)
      throws SQLException, ItemException
  {
    if (hash.containsKey(key(c))) {
      /* must test whether the user can edit the *old* event, not
         *this* event
       */
      if (get(c.getId()).canBeModifiedBy(this.user)) {
        c.replace(this.user);
        localDelete((CalendarObject) hash.get(key(c)));
        localAdd(c);
      } else {
        throw new ItemAccessException(this.user + " cannot update " + c);
      }
    } else {
      throw new NoSuchItemException("replace: couldn't find item " +
                                    key(c));
    }

    invalidateSort();
  }

  /**
    get an item from the Cache

    @param id ID of the item
    @return The CalendarObject with the same id
    @exception NoSuchItemException if no item exists with that id
   */
  public CalendarObject get(int id) throws NoSuchItemException
  {
    if (hash.containsKey(key(id))) {
      return (CalendarObject) hash.get(key(id));
    } else {
      throw new NoSuchItemException("get: couldn't find item " + id);
    }
  }

  /**
    @return If the key refers to an object in the cache
    @param o key to check
   */
  public boolean containsKey(Object o)
  {
    return hash.containsKey(o);
  }

  /** The elements in the cache, sorted */
  private Vector sorted = null;

  /**
    Invalidate the sorted object
   */
  // synchronized to avoid problems in sortedElements, below
  private synchronized void invalidateSort()
  {
    sorted = null;
  }

  /**
    Get all the elements in the cache, in sorted order
    @return all the elements in the cache, in sorted order
   */
  // synchronized to avoid race conditions with invalidateSort, above
  public synchronized Iterator sortedElements()
  {
    if (sorted == null) {
      sorted = new Vector(values());
      Collections.sort(sorted);
    }

    return sorted.iterator();
  }

  /**
    Actually load all the elements in the cache
    @exception SQLException If there's a problem with the database
    @exception ItemException If the database contains bad data
   */
  protected abstract void doLoadAll() throws SQLException, ItemException;

  /**
    Load all the elements in the cache
    @exception SQLException If there's a problem with the database
    @exception ItemException If the database contains bad data
   */
  protected void loadAll() throws SQLException, ItemException
  {
    if (this.isLoaded) {
      return;
    }

    doLoadAll();
    this.isLoaded = true;
  }

  /**
    Are all the elements in the cache loaded?
    @return Are all the elements in the cache loaded?
   */
  public boolean isLoaded()
  {
    return this.isLoaded;
  }

  /**
    Load a single item into the cache
    @param id Id of the item to load
    @exception SQLException If there's a problem accessing the database
    @exception CaldataException If there's a problem with the data in the db
    @exception NoSuchItemException If the item does not exist
    @exception ItemAccessException If the item can't be added to the cache
   */
  protected abstract void loadOne(int id)
      throws SQLException, CaldataException, NoSuchItemException,
             ItemAccessException;

  /**
    Ensure that an item is loaded in the cache
    @param id Id of the item to load
    @exception SQLException If there's a problem accessing the database
    @exception CaldataException If there's a problem with the data in the db
    @exception NoSuchItemException If the item does not exist
    @exception ItemAccessException If the item can't be loaded into the cache
   */
  void ensureLoaded(int id)
      throws SQLException, CaldataException, NoSuchItemException,
             ItemAccessException
  {
    if (!containsKey(key(id))) {
      loadOne(id);
    }

    get(id);            // throws NoSuchItemException if it didn't work
  }
}
