// java

package edu.washington.cac.calendar;

import java.io.Serializable;
import java.sql.Time;
import java.text.DateFormat;
import java.text.FieldPosition;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

import edu.washington.cac.calendar.data.DataChecker;
import edu.washington.cac.calendar.data.CaldataException;

/** A wrapper around the <code>java.util.Calendar</code> class implementing
    useful, but perhaps esoteric Calendar functions for UWCal.

    A <code>MyCalendar</code> object may or may not have a particular time
    of day associated with it.

    @author Greg Barnes
    @version 2.2
 */
public class MyCalendar implements Serializable
{
  private Calendar cal;
  private java.util.Date time;

  /** Create a MyCalendar object representing a particular date and time.

    @param c The <code>java.util.Calendar</code> representing the
          date and time.
   */
  public MyCalendar(Calendar c)
  {
    this.time = c.getTime();
    this.cal = c;
  }

  /** Create a MyCalendar object representing the current date and time.
   */
  public MyCalendar()
  {
    this(Calendar.getInstance());
  }

  /**
    Create a MyCalendar object representing a particular date.
    A <code>MyCalendar</code> created using this constructor will
    not have an associated time.

    @param eightDigitDate The date in question, represented as an
       8 character <code>String</code> of the form
       <code>YYYYMMDD</code>
   */
  public MyCalendar(String eightDigitDate) throws CaldataException
  {
    this(Calendar.getInstance());
    this.time = null;
    DataChecker.checkDateString(eightDigitDate);
    this.cal.set(DataChecker.yearNum(eightDigitDate),
                 DataChecker.monthNum(eightDigitDate),
                 DataChecker.dayNum(eightDigitDate), 0, 0, 0);
    this.cal.set(Calendar.MILLISECOND, 0);

    /* The next line is to workaround some nasty JDK behavior.

       if you set() as above, then set the DAY_OF_WEEK (as
       getFirstDayOfThisWeek() and other methods below do), the DAY_OF_MONTH
       we entered above is thrown out when determining the new DAY_OF_MONTH
       in favor of the unset WEEK_OF_MONTH (or WEEK_OF_YEAR).

       To avoid this, we make the following unnecessary call to get
       the WEEK_OF_YEAR value, which should recompute all the other 'fields'
       of the CALENDAR.
      */
    this.cal.get(Calendar.WEEK_OF_YEAR);
  }

  /**
    Create a MyCalendar object representing a particular date and,
    optionally, time.

    @param date The desired date.  This parameter should not be
      <code>null</code>
    @param time The desired time.  This parameter may be
      <code>null</code>, in which case the <code>MyCalendar</code>
      has no associated time.
   */
  public MyCalendar(java.util.Date date, Time time)
  {
    this.time = time;

    if (time == null) {
      setup(normalizedDate(date));
    } else {
      setup(time);
      Calendar cal2 = Calendar.getInstance();
      cal2.setTime(date);
      this.cal.set(cal2.get(Calendar.YEAR), cal2.get(Calendar.MONTH),
                   cal2.get(Calendar.DATE));
    }
  }

  /**
    Create a MyCalendar object representing a particular date and
    time.

    @param time The <code>java.util.Date</code> representing the
      date and time.
   */
  public MyCalendar(java.util.Date time)
  {
    this.time = time;
    setup(time);
  }

  private void setup(java.util.Date time)
  {
    this.cal = Calendar.getInstance();
    this.cal.setTime(time);
  }

  /** Convert this object into a <code>java.util.Calendar</code>

    @return this object converted into a
      <code>java.util.Calendar</code>
   */
  public Calendar getCalendar()
  {
    return this.cal;
  }

  /** Test whether the object represent a particular time
    (as well as a particular date)

    @return Does the object represent a particular time,
    as well as a particular date?
   */
  public boolean hasTime()
  {
    return this.time != null;
  }

  /**
    Convert this object into a <code>java.util.Date</code>

    @return this object converted into a <code>java.util.Date</code>
   */
  public Date getTime()
  {
    return this.cal.getTime();
  }

  /** Get a <code>String</code> representation of a particular time
      field of the object.

    @param field The field to be returned,
      <i>e.g.</i>, <code>MONTH_FIELD</code>.  For possible values, see
      the constants in <code>java.text.DateFormat</code>
    @param dateFormat The style of <code>DateFormat</code> to use,
      <i>e.g.</i>, <code>SHORT</code>.  For possible values, see
      the constants in <code>java.text.DateFormat</code>.
    @return A <code>String</code> representation of a particular time
      field of the object.
   */
  public String getComponent(int field, int dateFormat)
  {
    FieldPosition f = new FieldPosition(field);
    StringBuffer s = DateFormat.
        getDateTimeInstance(dateFormat, dateFormat).
        format(getTime(), new StringBuffer(), f);
    return s.toString().substring(f.getBeginIndex(), f.getEndIndex());
  }

  /** Get a <code>String</code> representation of a particular time
      field of the object.

    @param field The field to be returned,
      <i>e.g.</i>, <code>MONTH_FIELD</code>.  For possible values, see
      the constants in <code>java.text.DateFormat</code>
    @return A <code>String</code> representation of a particular time
      field of the object, in the <code>SHORT DateFormat</code> style.
   */
  public String getComponent(int field)
  {
    return getComponent(field, DateFormat.SHORT);
  }

  /**
    Get a <code>String</code> representation of the time of day

    @return A short <code>String</code> representation of the time of
       day of this <code>MyCalendar</code>.  If <code>hasTime</code>
       is false, returns a blank string
   */
  public String timeString()
  {
    return timeString(DateFormat.getTimeInstance(DateFormat.SHORT));
  }

  /**
    Get a <code>String</code> representation of the time of day

    @param df Format for the result
    @return The time of day, formatted per <code>df</code>
       If <code>hasTime</code> is false, returns a blank string
   */
  public String timeString(DateFormat df)
  {
    if (!hasTime()) {
      return " ";
    } else {
      return df.format(getTime());
    }
  }

  /**
    Get the first day of the week

    @return the first day of the week <i>e.g.</i>, Sunday in US,
    Monday in France
   */
  public int getFirstDayOfWeek()
  {
    return this.cal.getFirstDayOfWeek();
  }

  /** Get the last day of the week

    @return the last day of the week <i>e.g.</i>, Saturday in US,
    Sunday in France
   */
  /* % is a remainder function, not the modulo we really want, so we have to
     add daysInWeek() to handle negative values.  E.g., Java says
     -6 % 7 = -6, but we want a positive value.
   */
  public int getLastDayOfWeek()
  {
    return (daysInWeek() + this.cal.getFirstDayOfWeek() - 1) % daysInWeek();
  }

  /**
    Test whether the given day is the first day of the week

    @param dow The day of the week to test
    @return whether the given day is the first day of the week
   */
  boolean isFirstDayOfWeek(int dow)
  {
    return getFirstDayOfWeek() == dow;
  }

  /**
    Test whether the given day is the last day of the week

    @param dow The day of the week to test
    @return whether the given day is the last day of the week
   */
  public boolean isLastDayOfWeek(int dow)
  {
    return getLastDayOfWeek() == dow;
  }

  /**
    Get the minimum value for the given unit of time

    @param unit The unit of time, <i>e.g.</i>, <code>MONTH</code>.
      For possible values, see the constants in
      <code>java.util.Calendar</code>
    @return the minimum value for the given unit of time
   */
  public int getMinimum(int unit)
  {
    return this.cal.getMinimum(unit);
  }

  /** Get the maximum value for the given unit of time

    @param unit The unit of time, <i>e.g.</i>, <code>MONTH</code>.
      For possible values, see the constants in
      <code>java.util.Calendar</code>
    @return the maximum value for the given unit of time
   */
  public int getMaximum(int unit)
  {
    return this.cal.getMaximum(unit);
  }

  /**
    Set the value for the given unit of time for this
      <code>MyCalendar</code>

    @param unit The unit of time, <i>e.g.</i>, <code>MONTH</code>.
      For possible values, see the constants in
      <code>java.util.Calendar</code>
    @param value Value for the unit of time
   */
  public void set(int unit, int val)
  {
    this.cal.set(unit, val);
    // See note in MyCalendar(String eightDigitDate) as to why this line is here
    this.cal.get(Calendar.WEEK_OF_YEAR);
  }

  /**
    Get the value for the given unit of time for this
      <code>MyCalendar</code>

    @param unit The unit of time, <i>e.g.</i>, <code>MONTH</code>.
      For possible values, see the constants in
      <code>java.util.Calendar</code>
    @return the value for the given unit of time for this
      <code>MyCalendar</code>
   */
  public int get(int unit)
  {
    return this.cal.get(unit);
  }

  /**
    Get the number of the month for this <code>MyCalendar</code>.
    Note:  The first month is number 1.

    @return the month number for this <code>MyCalendar</code>
   */
  public int month()
  {
    return this.cal.get(Calendar.MONTH) + 1;
  }

  /**
    Get the number of the day of the month for this <code>MyCalendar</code>.
    @return the day of the month for this <code>MyCalendar</code>
   */
  public int day()
  {
    return this.cal.get(Calendar.DATE);
  }

  /**
    Get the number of the year for this <code>MyCalendar</code>.
    @return the year for this <code>MyCalendar</code>
   */
  public int year()
  {
    return this.cal.get(Calendar.YEAR);
  }

  /**
    Get a two-digit <code>String</code> representation of a
      one to two-digit number

    @param i A one or two digit number
    @return A two-digit <code>String</code> representation of the
      number
   */
  public static String twoDigit(int i)
  {
    return (i < 10 ? "0" : "") + i;
  }

  /**
    Get a two-digit <code>String</code> representation of the day of the month

    @return A two-digit <code>String</code> representation of the
      day of the month for this <code>MyCalendar</code>.
    */
  public String twoDigitDay()
  {
    return twoDigit(day());
  }

  /**
    Get a two-digit <code>String</code> representation of the
      month of year

    @return A two-digit <code>String</code> representation of the
      month of year for this <code>MyCalendar</code>.
   */
  public String twoDigitMonth()
  {
    return twoDigit(month());
  }

  /**
    Get a four-digit <code>String</code> representation of the
      year for this <code>MyCalendar</code>.

    @return A four-digit <code>String</code> representation of the
      year for this <code>MyCalendar</code>.
   */
  public String fourDigitYear()
  {
    return (this.cal.get(Calendar.YEAR) + "");
  }

  /**
    Get an eight-digit <code>String</code> representation of the
      date

    @return An eight-digit <code>String</code> representation of the
      date for this <code>MyCalendar</code>, of the form
       <code>YYYYMMDD</code>
   */
  public String getDateDigits()
  {
    return fourDigitYear() + twoDigitMonth() + twoDigitDay();
  }

  /** Get a calendar some time later or earlier than this one

    @param c The calendar to start with
    @param unit The unit of time to add or subtract, <i>e.g.</i>,
      <code>MONTH</code>.  For possible values, see the constants in
      <code>java.util.Calendar</code>
    @param amount The number of units to add or subtract.  A positive
      value means to add, a negative value to subtract
    @return A <code>java.util.Calendar</code> equivalent to
      this calendar some time later or earlier
   */
  static Calendar add(Calendar c, int unit, int amount)
  {
    c.add(unit, amount);
    return c;
  }

  /** Get a calendar some time later or earlier than this one

    @param unit Units to add/subtract, <i>e.g.</i>,
      <code>Calendar.DATE</code> to add days
    @param amount Number of units to add or subtract.  Use negative
      numbers to subtract
    @return a <code>MyCalendar</code> corresponding to this
      <code>MyCalendar</code> +/- the appropriate number of units
   */
  public MyCalendar addTime(int unit, int amount)
  {
    return new MyCalendar(add((Calendar) this.cal.clone(), unit, amount));
  }

  /**
    Get a <code>MyCalendar</code> one day earlier.

    @return A <code>MyCalendar</code> equivalent to
      this <code>MyCalendar</code> one day earlier.
   */
  public MyCalendar yesterday()
  {
    return addTime(Calendar.DATE, -1);
  }

  /**
    Get a <code>MyCalendar</code> one day later.

    @return A <code>MyCalendar</code> equivalent to
      this <code>MyCalendar</code> one day later.
   */
  public MyCalendar tomorrow()
  {
    return addTime(Calendar.DATE, 1);
  }

  /**
    Get a <code>String</code> representation of the date

    @return A <code>String</code> representation of this
      <code>MyCalendar</code>'s date.
    @param format The style of <code>DateFormat</code> to use,
      <i>e.g.</i>, <code>SHORT</code>.  For possible values, see
      the constants in <code>java.text.DateFormat</code>.
   */
  public String dateString(int format)
  {
    return DateFormat.getDateInstance(format).format(getTime());
  }

  /**
    Get a <code>String</code> representation of the date

    @return A <code>String</code> representation of this
      <code>MyCalendar</code>, in the <code>DateFormat.FULL</code>
      style.
   */
  public String dateString()
  {
    return dateString(DateFormat.FULL);
  }

  /**
    Get a Calendar for the current month

    @return A <code>java.util.Calendar</code> representing some time
      in the current month
   */
  static Calendar currentMonth()
  {
    return Calendar.getInstance();
  }

  /**
    Get the number of days in a week.

    @return The number of days in a week.
   */
  public int daysInWeek()
  {
    return this.cal.getMaximum(Calendar.DAY_OF_WEEK) -
           this.cal.getMinimum(Calendar.DAY_OF_WEEK) + 1;
  }

  /**
    Get the number of days in a week, as a <code>String</code>.

    @return The number of days in a week, as a <code>String</code>.
   */
  String daysInWeekString()
  {
    return new Integer(daysInWeek()).toString();
  }

  /**
    Get the number of months in a year

    @return The number of months in a year
   */
  public int monthsInYear()
  {
    return this.cal.getMaximum(Calendar.MONTH) -
           this.cal.getMinimum(Calendar.MONTH) + 1;
  }

  /**
    Get a <code>String</code> representation of the month

    @return A <code>String</code> representation of the
      month of year for this <code>MyCalendar</code>.
   */
  public String monthString()
  {
    return getComponent(DateFormat.MONTH_FIELD, DateFormat.LONG);
  }

  /**
    Get the first day of the year for this <code>MyCalendar</code>.

    @return A <code>java.util.Calendar</code> representing the
        first day of the year for this <code>MyCalendar</code>.
        <i>e.g.</i>, if this <code>MyCalendar</code> represents
        February 5, 2000, returns a <code>Calendar</code> representing
        January 1, 2000.
   */
  public Calendar firstDayOfThisYear()
  {
    Calendar firstDay = (Calendar) this.cal.clone();
    firstDay.set(Calendar.DAY_OF_YEAR,
                 this.cal.getMinimum(Calendar.DAY_OF_YEAR));
    return firstDay;
  }

  /**
    Get the last day of the year for this <code>MyCalendar</code>.

    @return A <code>java.util.Calendar</code> representing the
          last day of the year for this <code>MyCalendar</code>.
          <i>e.g.</i>, if this <code>MyCalendar</code> represents
          February 5, 2000, returns a <code>Calendar</code> representing
          December 31, 2000.
   */
  public Calendar lastDayOfThisYear()
  {
    return add(add(firstDayOfThisYear(), Calendar.YEAR, 1),
               Calendar.DATE, -1);
  }

  /**
    Get the first day of the month for this <code>MyCalendar</code>.

    @return A <code>java.util.Calendar</code> representing the
        first day of the month for this <code>MyCalendar</code>.
        <i>e.g.</i>, if this <code>MyCalendar</code> represents
        February 5, 2000, returns a <code>Calendar</code> representing
        February 1, 2000.
   */
  public Calendar firstDayOfThisMonth()
  {
    Calendar firstDay = (Calendar) this.cal.clone();
    firstDay.set(Calendar.DAY_OF_MONTH,
                 this.cal.getMinimum(Calendar.DAY_OF_MONTH));
    return firstDay;
  }

  /**
    Get the last day of the month for this <code>MyCalendar</code>.

    @return A <code>java.util.Calendar</code> representing the
          last day of the month for this <code>MyCalendar</code>.
          <i>e.g.</i>, if this <code>MyCalendar</code> represents
          February 5, 2000, returns a <code>Calendar</code> representing
          February 29, 2000.
   */
  public Calendar lastDayOfThisMonth()
  {
    return add(add(firstDayOfThisMonth(), Calendar.MONTH, 1),
               Calendar.DATE, -1);
  }

  /**
    Test whether this object represent a weekend day.

    @return does this object represent a weekend day?
   */
  public boolean isWeekendDay()
  {
    return isWeekendDay(dayOfWeek());
  }

  /**
    Test whether a given day of the week represent a weekend day.

    @param weekpos The day of the week to test
    @return does a given day of the week represent a weekend day?
   */
  public static boolean isWeekendDay(int weekpos)
  {
    // strange but true: the week 'end' is at the beginning
    return weekpos < 2;
  }

  /**
    Get the day of the week for this <code>MyCalendar</code>.
    @return The day of the week for this <code>MyCalendar</code>.
   */
  public int dayOfWeek()
  {
    return dayOfWeek(this.cal);
  }

  /**
    Get the day of the week for a <code>java.util.Calendar</code>.

    @param c The date to evaluate
    @return The day of the week for a <code>java.util.Calendar</code>.
   */
  public int dayOfWeek(Calendar c)
  {
    return c.get(Calendar.DAY_OF_WEEK) % daysInWeek();
  }

  /**
    Get the day of the week of the first day of the month for
      this <code>MyCalendar</code>

    @return The day of the week of the first day of the month for
      this <code>MyCalendar</code> (see
      <code>firstDayOfthisMonth()</code>)
   */
  public int dayOfWeekOfFirstDay()
  {
    return dayOfWeek(firstDayOfThisMonth());
  }

  /**
    Get the day of the week of the last day of the month for
      this <code>MyCalendar</code>

    @return The day of the week of the last day of the month for
      this <code>MyCalendar</code> (see
      <code>lastDayOfthisMonth()</code>)
   */
  public int dayOfWeekOfLastDay()
  {
    return dayOfWeek(lastDayOfThisMonth());
  }

  /**
    Get the first day of the week for this <code>MyCalendar</code>.

    @return A <code>java.util.Calendar</code> representing the
      first day of the week for this <code>MyCalendar</code>.
      Analogous to <code>firstDayOfthisMonth()</code>.
   */
  public Calendar firstDayOfThisWeek()
  {
    Calendar firstDay = (Calendar) this.cal.clone();
    firstDay.set(Calendar.DAY_OF_WEEK,
                 this.cal.getMinimum(Calendar.DAY_OF_WEEK));
    return firstDay;
  }

  /**
    Get the last day of the week for this <code>MyCalendar</code>.

    @return A <code>java.util.Calendar</code> representing the
      last day of the week for this <code>MyCalendar</code>.
      Analogous to <code>lastDayOfthisMonth()</code>.
   */
  public Calendar lastDayOfThisWeek()
  {
    return add(add(firstDayOfThisWeek(), Calendar.WEEK_OF_YEAR, 1),
               Calendar.DATE, -1);
  }

  /**
    Get a <code>String</code> representation of the
      day of the week for a given date

    @param d The date to evaluate
    @return A <code>String</code> representation of the
      day of the week for the date
   */
  static String dayName(java.util.Date d)
  {
    SimpleDateFormat formatter = new SimpleDateFormat("EEEE");
    return formatter.format(d);
  }

  /**
    Get a <code>String</code> representation of the
      given day of the week

    @param dow An integer representation of the day of the week in
      question.
    @return A <code>String</code> representation of the
      given day of the week
   */
  public static String dayOfWeekName(int dow)
  {
    Calendar c = Calendar.getInstance();
    c.set(Calendar.DAY_OF_WEEK, dow);
    return dayName(c.getTime());
  }

  /**
    Get a <code>String</code> abbreviation of the
      given day of the week.

    @param dow An integer representation of the day of the week in
      question.
    @return A <code>String</code> abbreviation of the
      given day of the week.  Returns "?" if the
      day's name is too short to abbreviate.
   */
  public String dayOfWeekAbbr(int dow)
  {
    try {
      return dayOfWeekName(dow).substring(0, 2);
    } catch (StringIndexOutOfBoundsException e) {
      return "?";
    }
  }

  /**
    Get a different day in the same month

    @param dayNumber The new day of the month
    @return A <code>MyCalendar</code> representing a different
      day in the same month as this <code>MyCalendar</code>.
   */
  public MyCalendar differentDayInSameMonth(int dayNumber)
  {
    Calendar newDay = (Calendar) this.cal.clone();
    newDay.set(Calendar.DAY_OF_MONTH, dayNumber);
    return new MyCalendar(newDay);
  }

  /**
    Get the first month of the same year

    @return A <code>java.util.Calendar</code> representing the
      first month in the same year as this <code>MyCalendar</code>.
   */
  public Calendar firstMonthOfThisYear()
  {
    Calendar firstMonth = (Calendar) this.cal.clone();
    firstMonth.set(Calendar.MONTH,
                   this.cal.getMinimum(Calendar.MONTH));
    return firstMonth;
  }

  /**
    Test whether another <code>MyCalendar</code> represents the same
      date as this one.

    @param c The <code>MyCalendar</code> to test.
    @return <code>true</code> if the other object represents the same
      date as this one, <code>false</code> otherwise.
   */
  public boolean isSameDay(MyCalendar c)
  {
    return dateString().equals(c.dateString());
  }

  /**
    Test whether another object represents an earlier date than this one.

    @param c The <code>MyCalendar</code> to test.
    @return <code>true</code> if the other object represents an earlier
      date than this one, <code>false</code> otherwise.
   */
  public boolean isEarlierDay(MyCalendar c)
  {
    return getDateDigits().compareTo(c.getDateDigits()) < 0;
  }

  /**
    Remove time information from a Date.  This is one way to 
    normalize an instance of java.sql.Date (which not all JVMs do correctly).
    @param d java.util.Date to normalize
    @return normalized java.sql.Date version of d 
   */
  public static java.sql.Date normalizedDate(java.util.Date d)
  {
    if (d == null) {
      return null;
    }

    Calendar c = Calendar.getInstance();
    c.setTime(d);
    c.set(Calendar.HOUR, 0);
    c.set(Calendar.MINUTE, 0);
    c.set(Calendar.SECOND, 0);
    c.set(Calendar.MILLISECOND, 0);
    return new java.sql.Date(c.getTime().getTime());
  }

  /**
    Not all JVMs appear to normalize java.sql.Time correctly.  This
    routine will do so.
    @param t java.sql.Time to normalize
    @return normalized version of t
   */
  public static java.sql.Time normalizedTime(java.sql.Time t)
  {
    if (t == null) {
      return null;
    }

    /* per the java.sql.Time documentation, set the 'date' components to
       January 1, 1970, and the milliseconds to 0
     */
    Calendar c = Calendar.getInstance();
    c.setTime(t);
    c.set(1970, Calendar.JANUARY, 1);
    c.set(Calendar.MILLISECOND, 0);
    return new java.sql.Time(c.getTime().getTime());
  }
}
