/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.cct.uwcal.webclient;

import edu.rpi.cct.uwcal.calsvci.TimeDateComponents;
import edu.rpi.cct.uwcal.common.TimeView;
import edu.rpi.cct.uwcal.common.UWCalDefs;

import edu.washington.cac.calfacade.shared.MyCalendarVO;

import javax.servlet.http.HttpServletRequest;
import java.util.Date;

/**
 * Action to go to a given date and/or set a day/week/month/year view.
 * <p>Request parameter<br>
 *      "date" date to move to.
 * <p>Optional request parameter<br>
 *      "viewType" type of view we want, day, week, etc.
 *
 */
public class UWCalGoToAction extends UWUserCalAbstractAction {
  /** This is the routine which does the work.
   *
   * @param request   Needed to locate session
   * @param form      Action form
   * @return String   forward name
   */
  public String doAction(HttpServletRequest request,
                         UWCalActionForm form) throws Throwable {
    gotoDateView(this, form,
                  form.getDate(),
                  form.getViewTypeI(),
                  debug);

    return "success";
  }

  /** Set the current date and/or view. The date may be null indicating we
   * should switch to a new view based on the current date.
   *
   * <p>newViewTypeI may be less than 0 indicating we stay with the current
   * view but switch to a new date.
   *
   * @param form         UWCalActionForm
   * @param date         String yyyymmdd date or null
   * @param newViewTypeI new view index or -1
   */
  public static void gotoDateView(UWUserCalAbstractAction action,
                                  UWCalActionForm form,
                                  String date,
                                  int newViewTypeI,
                                  boolean debug) throws Throwable {
    /* We get a new view if either the date changed or the view changed.
     */
    boolean newView = false;

    if (debug) {
      action.logIt("ViewTypeI=" + newViewTypeI);
    }

    MyCalendarVO dt;
    TimeView tv = form.getCurTimeView();

    if (newViewTypeI == UWCalDefs.todayView) {
      // dt = new MyCalendarVO(new Date(System.currentTimeMillis()));
      dt = new MyCalendarVO();
      newView = true;
      newViewTypeI = UWCalDefs.dayView;
    } else if (date == null) {
      if (newViewTypeI == UWCalDefs.dayView) {
        // selected specific day to display from personal event entry screen.
        dt = new MyCalendarVO(form.getViewStartDate().getDateString());
        newView = true;
      } else {
       if (debug) {
         action.logIt("No date supplied: go with current date");
       }

       // Just stay here
       dt = tv.getCurDay();
      }
    } else {
      if (debug) {
        action.logIt("Date=" + date + ": go with that");
      }

      dt = new MyCalendarVO(date);
      newView = true;
    }

    if ((newViewTypeI >= 0) &&
        (newViewTypeI != form.getCurView())) {
      // Change of view
      newView = true;
    }

    if (newView && (newViewTypeI < 0)) {
      newViewTypeI = form.getCurView();
      if (newViewTypeI < 0) {
        newViewTypeI = UWCalDefs.defaultView;
      }
    }

    TimeDateComponents viewStart = form.getViewStartDate();

    if (!newView) {
      /* See if we were given an explicit date as view start date components.
         If so we'll set a new view of the same period as the current.
       */
      String vsdate = viewStart.getDateString();

      if (!(vsdate.equals(form.getCurTimeView().getFirstDay().getDateDigits()))) {
        newView = true;
        newViewTypeI = form.getCurView();
        dt = new MyCalendarVO(vsdate);
      }
    }

    if (newView) {
      form.setCurView(newViewTypeI);
      form.setViewMcDate(dt);
      form.refreshIsNeeded();
    }

    tv = form.getCurTimeView();
    // dt = tv.getCurDay();

    /** Set first day, month and year
     */

    MyCalendarVO firstDay = tv.getFirstDay();

    viewStart.setDay(firstDay.getTwoDigitDay());
    viewStart.setMonth(firstDay.getTwoDigitMonth());
    viewStart.setYear(firstDay.getFourDigitYear());

    form.getEventStartDate().setTimeInMillis(tv.getCurDay().getTimeInMillis());
    form.getEventEndDate().setTimeInMillis(tv.getCurDay().getTimeInMillis());
  }
}
