/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.cct.uwcal.webclient;

import edu.rpi.cct.uwcal.calsvci.CalSvcI;
import edu.washington.cac.calfacade.shared.LocationVO;

import javax.servlet.http.HttpServletRequest;

/**
 * Action to edit a Location
 *
 * <p>Request parameters<ul>
 *      <li>"updateLocation=anything" means we should try to update the current
 * location in form.editLocation.</li>
 *      <li>"locationId=nnn"    id of location to fetch for editing.</li>
 * </ul>
 *.
 * <p>Forwards to:<ul>
 *      <li>"doNothing"    for guest mode/invalid id or non-exiting location.</li>
 *      <li>"edit"         to edit the location.</li>
 *      <li>"success"      changes made.</li>
 * </ul>
 *
 * <p>Errors:<ul>
 *      <li>org.ucal.calendar.error.noaccess - when user has insufficient
 *            access (tries to edit public location)</li>
 * </ul>
 */
public class UWCalEditLocationAction extends UWUserCalAbstractAction {
  /** This is the routine which does the work.
   *
   * @param request   Needed to locate session
   * @param form      Action form
   * @return String   forward name
   */
  public String doAction(HttpServletRequest request,
                         UWCalActionForm form) throws Throwable {
    if (form.getGuest()) {
      // Just ignore this
      return "doNothing";
    }

    String reqpar = request.getParameter("updateLocation");
    if (reqpar != null) {
      return updateLocation(request, form);
    }

    int locationId = form.getLocationId();

    if (locationId < 0) {
      // Do nothing
      return "doNothing";
    }

    LocationVO loc = form.getCalSvcI().getLocation(locationId);

    if (loc == null) {
      return "doNothing";
    }

    if (loc.isPublic()) {
      err.emit("org.ucal.calendar.error.noaccess", "for that action");
      return "doNothing";
    }

    form.setEditLocation(loc);

    return "edit";
  }

  /** Update the db with the location in editLocation.
   *
   * @param request   Needed to locate session
   * @param form      Action form
   * @return String   forward name
   */
  public String updateLocation(HttpServletRequest request,
                               UWCalActionForm form) throws Throwable {
    CalSvcI svci = form.getCalSvcI();
    LocationVO loc = form.getEditLocation();

    svci.replaceLocation(loc);

    form.refreshIsNeeded();

    return "success";
  }
}
