/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.cct.uwcal.webclient;

import edu.rpi.cct.uwcal.calsvci.CalSvcI;
import edu.rpi.cct.uwcal.calsvci.TimeDateComponents;
import edu.washington.cac.calfacade.shared.EventVO;
import edu.washington.cac.calfacade.shared.LocationVO;

import javax.servlet.http.HttpServletRequest;

/**
 * Action to edit an Event
 *
 * <p>Request parameters<ul>
 *      <li>"updateEvent=anything" means we should try to update the current
 * event in form.editEvent.</li>
 *      <li>"eventId=nnn" id of event to fetch for editing.</li>
 * </ul>
 *.
 * <p>Forwards to:<ul>
 *      <li>"doNothing"    for guest mode/invalid id or non-existing event.</li>
 *      <li>"edit"         to edit the event.</li>
 *      <li>"success"      changes made.</li>
 * </ul>
 *
 * <p>Errors:<ul>
 *      <li>org.ucal.calendar.error.noaccess - when user has insufficient
 *            access (tries to edit public event)</li>
 * </ul>
 *
 * <p>Messages:<ul>
 *      <li>org.ucal.calendar.message.deleted.locations - when location is
 *            deleted</li>
 *      <li>org.ucal.calendar.message.added.locations - when location is
 *            added</li>
 * </ul>
 */
public class UWCalEditEventAction extends UWUserCalAbstractAction {
  /** This is the routine which does the work.
   *
   * @param request   Needed to locate session
   * @param form      Action form
   * @return String   forward name
   */
  public String doAction(HttpServletRequest request,
                         UWCalActionForm form) throws Throwable {
    if (form.getGuest()) {
      // Just ignore this
      return "doNothing";
    }

    String reqpar = request.getParameter("updateEvent");
    if (reqpar != null) {
      return updateEvent(request, form);
    }

    int eventId = form.getEventId();

    if (eventId < 0) {
      // Do nothing
      return "doNothing";
    }

    EventVO ev = form.getCalSvcI().getEvent(eventId);

    if (ev == null) {
      return "doNothing";
    }

    if (ev.isPublic()) {
      err.emit("org.ucal.calendar.error.noaccess", "for that action");
      return "doNothing";
    }

    form.setEditEvent(ev);

    LocationVO loc = ev.getLocation();

    form.setEditLocation(null);

    if (loc != null) {
      form.setLocationId(loc.getId());
    }

    return "edit";
  }

  /** Update the db with the event in editEvent.
   *
   * @param request   Needed to locate session
   * @param form      Action form
   * @return String   forward name
   */
  public String updateEvent(HttpServletRequest request,
                            UWCalActionForm form) throws Throwable {
    CalSvcI svci = form.getCalSvcI();
    EventVO ev = form.getEditEvent();

    ev.setStartdate(form.getEventStartDate().getDate());
    ev.setStarttime(form.getEventStartDate().getTime());
    ev.setEnddate(form.getEventEndDate().getDate());
    ev.setEndtime(form.getEventEndDate().getTime());

    int oldid = setLocation(ev, form);

    svci.replaceEvent(ev);

    if (oldid > 0) {
      if ((!svci.checkLocationRefs(oldid)) &&
          (svci.deleteLocation(oldid))) {
        msg.emit("org.ucal.calendar.message.deleted.locations", 1);
      }
    }

    form.refreshIsNeeded();

    return "success";
  }

  /** Change the events location maybe.
   *
   * @param form      Action form
   * @return int      old location id if the location changed
   */
  private int setLocation(EventVO ev,
                           UWCalActionForm form) throws Throwable {
    /** If a different location was selected from the drop down list
     * use that.
     */

    LocationVO loc = ev.getLocation();
    int formLocid = form.getLocationId();

    int oldid = -1;

    if (formLocid <= LocationVO.maxReservedId) {
      formLocid = -1;
    }

    if (formLocid > 0) {
      /* location selected */

      if ((loc == null) || (loc.getId() != formLocid)) {
        /* Use this one */
        if (loc != null) {
          oldid = loc.getId();
        }

        ev.setLocation(form.getCalSvcI().getLocation(formLocid));
        ev.setLocationid(formLocid);

        return oldid;
      }
    }

    /* No location was selected or it was the same as the current location.
       Check for a new location
     */
    LocationVO eloc = form.tidyLocation(form.getEditLocation());
    String locAddr = eloc.getAddress();

    if (locAddr != null) {
      if ((loc == null) || (!loc.getAddress().equals(locAddr))) {
        if (loc != null) {
          oldid = loc.getId();
        }

        ev.setLocationid(form.getCalSvcI().addLocation(eloc));
        ev.setLocation(eloc);
        msg.emit("org.ucal.calendar.message.added.locations", 1);

        return oldid;
      }
    }

    if ((loc != null) && (formLocid < 0) && (locAddr == null)) {
      ev.setLocation(null);
      ev.setLocationid(-1);

      return loc.getId();
    }

    /* Didn't change anything */

    return -1;
  }

}
