/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.cct.uwcal.webclient;

import edu.rpi.sss.util.jsp.JspUtil;

import edu.rpi.cct.uwcal.calsvci.TimeDateComponents;
import edu.rpi.cct.uwcal.common.DayView;
import edu.rpi.cct.uwcal.common.EventFormatter;
import edu.rpi.cct.uwcal.common.MonthView;
import edu.rpi.cct.uwcal.common.TimeView;
import edu.rpi.cct.uwcal.common.UWCalDefs;
import edu.rpi.cct.uwcal.common.WeekView;
import edu.rpi.cct.uwcal.common.YearView;
import edu.rpi.cct.uwcal.webcommon.UWCalActionFormBase;

import edu.washington.cac.calfacade.shared.CheckData;
import edu.washington.cac.calfacade.shared.EventVO;
import edu.washington.cac.calfacade.shared.LocationVO;
import edu.washington.cac.calfacade.shared.MyCalendarVO;

import java.sql.Date;
import java.sql.Time;
import java.util.HashMap;
import java.util.TreeSet;
import javax.servlet.http.HttpServletRequest;
import org.apache.struts.action.ActionMapping;

public class UWCalActionForm extends UWCalActionFormBase {
  private String date;

  /** These provide the labels and actions for view selection.
   * These should be localized
   */
  private String[] viewLabels =
    {"Today", "Day", "Week", "Month", "Year"};

  private static HashMap viewTypeMap = new HashMap();

  static {
    for (int i = 0; i < UWCalDefs.viewTypeNames.length; i++) {
      viewTypeMap.put(UWCalDefs.viewTypeNames[i], new Integer(i));
    }
  }

  /** Index of the view type set when the page was last generated
   */
  private int curView;

  /** This will be set if a refresh is needed - we do it on the way out.
   */
  private boolean refreshNeeded;

  /** We retrieve this to see if the public events were updated allowing us
      to force a refresh
   */
  private long publicLastmod;

  /** Index of the view type requested this time round. We set curView to
   * viewTypeI. This allows us to see if the view changed as a result of the
   * request.
   */
  private int viewTypeI;

  /** one of the viewTypeNames values
   */
  private String viewType;

  /** The current view (dsay, week, month etc)
   */
  private TimeView curTimeView;

  /** MyCalendarVO version of the start date
   */
  private MyCalendarVO viewMcDate;

  private TimeDateComponents viewStartDate;

  /** ...................................................................
   *                   Calendar fields
   *  ................................................................... */

  /** id of calendar we are requesting or currently viewing.
   */
  private int calId;

  /** ...................................................................
   *                   Subscriptions
   *  ................................................................... */

  private TreeSet subscriptions;

  /** ...................................................................
   *                   Event fields
   *  ................................................................... */

  /** id of event we are requesting or currently viewing.
   */
  private int eventId;

  /** Formatter for the current event
   */
  private EventFormatter curEventFmt;

  private String evShortdesc;
  private String evLink;

  /** ...................................................................
   *                   Location fields
   *  ................................................................... */

  /** Location selected by the user from given list.
   */
  private int locationId;

  /** Location address
   */
  private String laddress;

  /** ===================================================================
   *                   Methods
   *  =================================================================== */

  public TimeDateComponents getViewStartDate() {
    if (viewStartDate == null) {
      viewStartDate = getNowTimeComponents();
    }

    return viewStartDate;
  }

  /** This often appears as the request parameter specifying the date for an
   * action. Always YYYYMMDD format
   *
   * @param  val   String date in YYYYMMDD format
   */
  public void setDate(String val) {
    if (!CheckData.checkDateString(val)) {
      date = new MyCalendarVO().getDateDigits();
    } else {
      date = val;
    }
  }

  public String getDate() {
    return date;
  }

  /** ===================================================================
   *                   View methods
   *  =================================================================== */

  public String[] getViewLabels() {
    return viewLabels;
  }

  public String[] getViewTypeNames() {
    return UWCalDefs.viewTypeNames;
  }

  public String getViewTypeName(int i) {
    return UWCalDefs.viewTypeNames[i];
  }

  /** Index of the view type set when the page was last generated
   *
   * @param val  int valid view index
   */
  public void setCurView(int val) {
    curView = val;
  }

  public int getCurView() {
    return curView;
  }

  /** Index of the view type requested this time round. We set curView to
   * viewTypeI. This allows us to see if the view changed as a result of the
   * request.
   */
  public void setViewTypeI(int val) {
    viewTypeI = val;
  }

  public int getViewTypeI() {
    return viewTypeI;
  }

  /** This often appears as the request parameter specifying the view.
   * It should be one of the viewTypeNames
   *
   * @param  val   String viewType
   */
  public void setViewType(String val) {
    viewType = checkNull(val);

    if (viewType == null) {
      viewTypeI = -1;
      return;
    }

    Integer i = (Integer)viewTypeMap.get(viewType);

    if (i == null) {
      viewTypeI = UWCalDefs.defaultView;
    } else {
      viewTypeI = i.intValue();
    }
  }

  public String getViewType() {
    return viewType;
  }

  /** Date of the view as a MyCalendar object
   */
  public void setViewMcDate(MyCalendarVO val) {
    viewMcDate = val;
  }

  /** return true if the parameter is the index of the current view.
   * UW considers 0 to be today. We return selected if the current view is
   * day and it is for today.
   *
   * @param  i       int view index
   * @return boolean true if i is the current view index
   */
  public boolean getViewSelected(int i) {
    if (i == UWCalDefs.todayView) {
      if (curView != UWCalDefs.dayView) {
        return false;
      }

      return getCurTimeView().getFirstDay().isSameDate(new MyCalendarVO());
    }
    return (i == curView) /*|| ((curView == 1 && (i == 0))) */;
  }

  /** The current view (day, week, month etc)
   */
  public void setCurTimeView(TimeView val) {
    curTimeView = val;
  }

  /** The current view (day, week, month etc)
   */
  public TimeView getCurTimeView() {
    /* We might be called before any time is set. Set a week view by
       default
       */

    try {
      if (curTimeView == null) {
        /** Figure out the default from the properties
         */
        String vn;

        try {
          vn = JspUtil.getProperty(getMres(),
                                  "org.ucal.calendar.defaultview",
                                  "week");
        } catch (Throwable t) {
          System.out.println("Exception setting current view");
          vn = "week";
        }

        curView = -1;

        for (int i = 1; i < UWCalDefs.viewTypeNames.length; i++) {
          if (UWCalDefs.viewTypeNames[i].startsWith(vn)) {
            curView = i;
            break;
          }
        }

        if (curView < 0) {
          curView = UWCalDefs.weekView;
        }

        setViewMcDate(new MyCalendarVO());
        refreshView();
      }
    } catch (Throwable t) {
      t.printStackTrace();
    }

    if (curTimeView == null) {
      System.out.println("Null time view!!!!!!!!!!!");
    }

    return curTimeView;
  }

  public void refreshIsNeeded() {
    refreshView();
    refreshNeeded = true;
  }

  public boolean isRefreshNeeded() {
    return refreshNeeded;
  }

  public void checkRefresh() {
    if (!refreshNeeded){
      /* See if the events were updated */
      long lastmod;

      try {
        lastmod = getCalSvcI().getPublicLastmod();
      } catch (Throwable t) {
        // Not much we can do here
        err.emit(t);
        return;
      }

      if (lastmod == publicLastmod) {
        return;
      }

      publicLastmod = lastmod;
    }

    refreshView();
    refreshNeeded = false;
  }

  /** Reset the view according to the current setting of curView.
   * May be called when we change the view or if we need a refresh
   */
  public void refreshView() {
    if (debug) {
      System.out.println(" set new view to ViewTypeI=" + curView);
    }

    try {
      getCalSvcI().refreshEvents();
    } catch (Throwable t) {
      // Not much we can do here
      err.emit(t);
    }

    switch (curView) {
      case UWCalDefs.dayView:
        setCurTimeView(new DayView(viewMcDate, getCalSvcI(), debug));
        break;
      case UWCalDefs.weekView:
        setCurTimeView(new WeekView(viewMcDate, getCalSvcI(), debug));
        break;
      case UWCalDefs.monthView:
        setCurTimeView(new MonthView(viewMcDate, getCalSvcI(), debug));
        break;
      case UWCalDefs.yearView:
        setCurTimeView(new YearView(viewMcDate, getCalSvcI(),
                       getShowYearData(), debug));
        break;
    }
  }

  /** ===================================================================
   *                   Calendars
   *  =================================================================== */

  public void setCalId(int val) {
    calId = val;
  }

  public int getCalId() {
    return calId;
  }

  /** ===================================================================
   *                   Subscriptions
   *  =================================================================== */

  public void setSubscribed(int i, boolean val) {
    if (subscriptions == null) {
      subscriptions = new TreeSet();
    }

    subscriptions.add(new Integer(i));
  }

  public TreeSet getSubscriptions() {
    if (subscriptions == null) {
      subscriptions = new TreeSet();
    }

    return subscriptions;
  }

  public boolean getSubscribed(int i) {
    try {
      return getCalSvcI().getSubscribed(i);
    } catch (Throwable t) {
      err.emit(t);
      return false;
    }
  }

  /** ===================================================================
   *                   Events
   *  =================================================================== */

  public void setEventId(int val) {
    eventId = val;
  }

  public int getEventId() {
    return eventId;
  }

  public void setCurEventFmt(EventFormatter val) {
    curEventFmt = val;
  }

  public EventFormatter getCurEventFmt() {
    return curEventFmt;
  }

  public void setEvShortdesc(String val) {
    evShortdesc = checkNull(val);
  }

  public String getEvShortdesc() {
    return evShortdesc;
  }

  public void setEvLink(String val) {
    evLink = checkNull(val);
  }

  public String getEvLink() {
    return evLink;
  }

  /* * ===================================================================
   *                   Alerts
   *  =================================================================== * /

  public EventVO[] getTodaysAlerts() {
    try {
      EventVO[] alerts = getCalSvcI().getDaysAlerts(getToday());

      if (alerts == null) {
        return new EventVO[0];
      }

      return alerts;
    } catch (Throwable t) {
      err.emit(t);
      return new EventVO[0];
    }
  }*/

  /** ===================================================================
   *                   Locations
   *  =================================================================== */

  public void setLocationId(int val) {
    locationId = val;
  }

  public int getLocationId() {
    return locationId;
  }

  public void setLaddress(String val) {
    laddress = checkNull(val);
  }

  public String getLaddress() {
    return laddress;
  }

  /**
   * Reset properties to their default values.
   *
   * @param mapping The mapping used to select this instance
   * @param request The servlet request we are processing
   */
  public void reset(ActionMapping mapping, HttpServletRequest request) {
    if (debug) {
      getLog().debug("--" + getClass().getName() + ".reset--");
    }

    super.reset(mapping, request);

    date = null;
    viewTypeI = -1;
    eventId = -1;
    locationId = EventVO.DefaultLocationId;
    laddress = null;

    subscriptions = null;
  }
}

