/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.cct.uwcal.admin;

import edu.rpi.cct.uwcal.calsvci.CalSvcI;
import edu.rpi.cct.uwcal.webcommon.UWCalActionFormBase;
import edu.rpi.cct.uwcal.webcommon.UWCalSession;

import edu.washington.cac.calfacade.shared.Calintf;
import edu.washington.cac.calfacade.shared.SponsorVO;
import edu.washington.cac.calfacade.shared.UserAuth;

import javax.servlet.http.HttpServletRequest;

/** This action adds or modifies sponsor.
 *
 * @author Mike Douglass   douglm@rpi.edu
 */
public class PESponsorAction extends PEAbstractAction {
  /** This is the routine which does the work.
   *
   * @param request   Needed to locate session
   * @param sess      UWCalSession calendar session object
   * @param form      Action form
   * @return String   forward name
   */
  public String doAction(HttpServletRequest request,
                         UWCalSession sess,
                         PEActionForm form) throws Throwable {
    /** Check access
     */
    if (!form.getAuthorisedUser()) {
      return "noAccess";
    }

    /** =================================================================
     *                   Initial sponsor actions
     *  ================================================================= */

    String reqpar = request.getParameter("initAddSponsor");

    if (reqpar != null) {
      /** Set the objects to null so we get new ones.
       */
      form.initFields();

      return "addSponsor";
    }

    reqpar = request.getParameter("listUpdateSponsors");

    if (reqpar != null) {
      return "listSponsors";
    }

    reqpar = request.getParameter("listSponsors");

    if (reqpar != null) {
      return "listSponsors";
    }

    /** =================================================================
     *                   Sponsor update actions
     *  ================================================================= */

    reqpar = request.getParameter("deleteSponsor");
    if (reqpar != null) {
      return deleteSponsor(form);
    }

    reqpar = request.getParameter("addSponsor");

    if (reqpar != null) {
      if (!form.validateNewSponsor()) {
        return "retryAdd";
      }

      updateSponsor(form, true);
      return "done";
    }

    reqpar = request.getParameter("action");

    if (String.valueOf(reqpar).equals("fetch")) {
      /** User requested a sponsor from the list. Retrieve it, embed it in
       * the form so we can display the page
       */
      int id = form.getSponsor().getId();

      SponsorVO sponsor = form.getCalSvcI().getPublicSponsor(id);

      if (debug) {
        if (sponsor == null) {
          logIt("No sponsor with id " + id);
        } else {
          logIt("Retrieved sponsor " + sponsor.getId());
        }
      }

      form.setSponsor(sponsor);
      if (sponsor == null) {
        err.emit("org.ucal.calendar.pubevents.error.nosuchsponsor",
                 id);
        return "notFound";
      }

      return "updateSponsor";
    }

    /** We are just updating from the current form values.
     */
    if (!form.validateNewSponsor()) {
      return "retryMod";
    }

    return updateSponsor(form, false);
  }

  private String deleteSponsor(PEActionForm form) throws Throwable {
    int id = form.getSponsor().getId();
    CalSvcI svci = form.getCalSvcI();

    /** Only allow delete if not in use
     */
    if (svci.checkSponsorRefs(id)) {
      msg.emit("org.ucal.calendar.pubevents.message.sponsor.referenced");
      return "sponsorInUse";
    }

    /* Remove from preferences */
    UserAuth ua = form.retrieveUserAuth();

    ua.removeSponsor(null, id);

    if (!svci.deleteSponsor(id)) {
       err.emit("org.ucal.calendar.pubevents.error.nosuchsponsor", id);
      return "noSuchSponsor";
    }

    msg.emit("org.ucal.calendar.pubevents.message.sponsor.deleted");
    svci.touch();

    return "done";
  }

  private String updateSponsor(PEActionForm form,
                               boolean add) throws Throwable {
    /* if a sponsor with the same name, phone, e-mail, link, and creator exist
       in sponsors table then retrieve its sponsorid, otherwise add this
       sponsor to the database and then retrieve its sponsorid.
     */

    SponsorVO sp = form.getSponsor();

    if (!add && (sp.getId() <= SponsorVO.maxReservedId)) {
      // claim it doesn't exist
      err.emit("org.ucal.calendar.pubevents.error.nosuchsponsor",
               sp.getId());
      return "noSuchSponsor";
    }

    if (add) {
      form.getCalSvcI().ensureSponsorExists(sp);
    } else {
      form.getCalSvcI().replaceSponsor(sp);
    }

    form.updatePrefs(null, sp, null);
    form.getCalSvcI().touch();

    if (add) {
      msg.emit("org.ucal.calendar.pubevents.message.sponsor.added");
    } else {
      msg.emit("org.ucal.calendar.pubevents.message.sponsor.updated");
    }

    return "done";
  }
}

