/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.cct.uwcal.admin;

import edu.rpi.cct.uwcal.calsvci.CalSvcI;
import edu.rpi.cct.uwcal.webcommon.UWCalSession;

import edu.washington.cac.calfacade.shared.Calintf;
import edu.washington.cac.calfacade.shared.LocationVO;
import edu.washington.cac.calfacade.shared.UserAuth;

import javax.servlet.http.HttpServletRequest;

/** This action adds or modifies location.
 *
 * @author Mike Douglass   douglm@rpi.edu
 */
public class PELocationAction extends PEAbstractAction {
  /** This is the routine which does the work.
   *
   * @param request   Needed to locate session
   * @param sess      UWCalSession calendar session object
   * @param form      Action form
   * @return String   forward name
   */
  public String doAction(HttpServletRequest request,
                         UWCalSession sess,
                         PEActionForm form) throws Throwable {
    /** Check access
     */
    if (!form.getAuthorisedUser()) {
      return "noAccess";
    }

    /** =================================================================
     *                   Initial location actions
     *  ================================================================= */

    String reqpar = request.getParameter("initAddLocation");

    if (reqpar != null) {
      /** Set the objects to null so we get new ones.
       */
      form.initFields();

      return "addLocation";
    }

    reqpar = request.getParameter("listUpdateLocations");

    if (reqpar != null) {
      return "listLocations";
    }

    reqpar = request.getParameter("listLocations");

    if (reqpar != null) {
      return "listLocations";
    }

    /** =================================================================
     *                   Location update actions
     *  ================================================================= */

    reqpar = request.getParameter("deleteLocation");
    if (reqpar != null) {
      return deleteLocation(form);
    }

    reqpar = request.getParameter("addLocation");

    if (reqpar != null) {
      if (!form.validateNewLocation()) {
        return "retryAdd";
      }

      updateLocation(form, true);
      return "done";
    }

    reqpar = request.getParameter("action");

    if (String.valueOf(reqpar).equals("fetch")) {
      /** User requested a location from the list. Retrieve it, embed it in
       * the form so we can display the page
       */
      int id = form.getLocation().getId();

      LocationVO location = form.getCalSvcI().getPublicLocation(id);

      if (debug) {
        if (location == null) {
          logIt("No location with id " + id);
        } else {
          logIt("Retrieved location " + location.getId());
        }
      }

      form.setLocation(location);
      if (location == null) {
        err.emit("org.ucal.calendar.pubevents.error.nosuchlocation",
                 id);
        return "notFound";
      }

      return "updateLocation";
    }

    /** We are just updating from the current form values.
     */
    if (!form.validateNewLocation()) {
      return "retryMod";
    }

    return updateLocation(form, false);
  }

  private String deleteLocation(PEActionForm form) throws Throwable {
    int id = form.getLocation().getId();
    CalSvcI svci = form.getCalSvcI();

    /** Only allow delete if not in use
     */
    if (svci.checkLocationRefs(id)) {
      msg.emit("org.ucal.calendar.pubevents.message.location.referenced");
      return "locationInUse";
    }

    /* Remove from preferences */
    UserAuth ua = form.retrieveUserAuth();

    ua.removeLocation(null, id);

    if (!svci.deletePublicLocation(id)) {
      err.emit("org.ucal.calendar.pubevents.error.nosuchlocation", id);
      return "noSuchLocation";
    }

    svci.touch();
    msg.emit("org.ucal.calendar.pubevents.message.location.deleted");
    return "done";
  }

  private String updateLocation(PEActionForm form,
                               boolean add) throws Throwable {
    /* if a location with the same name, phone, e-mail, link, and creator exist
       in locations table then retrieve its locationid, otherwise add this
       location to the database and then retrieve its locationid.
     */

    LocationVO l = form.getLocation();
    CalSvcI svci = form.getCalSvcI();

    if (!add && (l.getId() <= LocationVO.maxReservedId)) {
      // claim it doesn't exist
      err.emit("org.ucal.calendar.pubevents.error.nosuchlocation",
               l.getId());
      return "noSuchLocation";
    }

    if (add) {
      svci.ensurePublicLocationExists(l);
    } else {
      svci.replaceLocation(l);
    }

    form.updatePrefs(null, null, l);
    svci.touch();

    if (add) {
      msg.emit("org.ucal.calendar.pubevents.message.location.added");
    } else {
      msg.emit("org.ucal.calendar.pubevents.message.location.updated");
    }

    return "done";
  }
}

