/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.cct.uwcal.admin;

import edu.rpi.cct.uwcal.calsvci.CalSvcI;
import edu.rpi.cct.uwcal.webcommon.UWCalActionFormBase;
import edu.rpi.cct.uwcal.webcommon.UWCalSession;

import edu.washington.cac.calfacade.shared.AuthUserVO;
import edu.washington.cac.calfacade.shared.EventVO;
import edu.washington.cac.calfacade.shared.KeywordVO;
import edu.washington.cac.calfacade.shared.LocationVO;
import edu.washington.cac.calfacade.shared.SponsorVO;
import edu.washington.cac.calfacade.shared.UserAuth;

import javax.servlet.http.HttpServletRequest;

/** This has the logic for processing events and alerts.
 *
 * @author Mike Douglass   douglm@rpi.edu
 */
public abstract class PEEventAlertProcess extends PEAbstractAction {
  private static final String[] alertReqPars = {
    "initAddAlert",
    "listUpdateAlerts",
    "listAlerts",
    "deleteAlerts",
    "deleteAlert",
    "deleteAlertOK",
    "addAlert",
    "updateAlert",
    "copyAlert",
    "listAllEvents",
  };

  private static final String[] eventReqPars = {
    "initAddEvent",
    "listUpdateEvents",
    "listEvents",
    "deleteEvents",
    "deleteEvent",
    "deleteEventOK",
    "addEvent",
    "updateEvent",
    "copyEvent",
    "listAllEvents",
  };

  private static final int reqInitAdd = 0;
  private static final int reqUpdateList = 1;
  private static final int reqList = 2;
  private static final int reqDeleteMultiple = 3;
  private static final int reqDelete = 4;
  private static final int reqDeleteOK = 5;
  private static final int reqAdd = 6;
  private static final int reqUpdate = 7;
  private static final int reqCopy = 8;
  private static final int reqListAll = 9;

  /** This is the routine which does the work.
   *
   * @param request   Needed to locate session
   * @param sess      UWCalSession calendar session object
   * @param form      Action form
   * @param alerts    true for alerts, false for events.
   * @return String   forward name
   */
  public String processAction(HttpServletRequest request,
                             UWCalSession sess,
                             PEActionForm form,
                             boolean alerts) throws Throwable {
    String[] reqPars;
    CalSvcI svci = form.getCalSvcI();

    /** Check access and set request parameters
     */
    if (alerts) {
      if (!form.getUserAuth().isAlertUser()) {
        return "noAccess";
      }
      reqPars = alertReqPars;
    } else {
      if (!form.getAuthorisedUser()) {
        return "noAccess";
      }
      reqPars = eventReqPars;
    }

    /** =================================================================
     *                   Initial actions
     *  ================================================================= */

    String reqpar = request.getParameter(reqPars[reqInitAdd]);

    if (reqpar != null) {
      /** Set the objects to null so we get new ones.
       */
      form.initFields();
      form.setEvent(null);
      form.setAlertEvent(alerts);
      form.resetEvent();

      return "add";
    }

    reqpar = request.getParameter(reqPars[reqUpdateList]);

    if (reqpar != null) {
      form.setAlertEvent(alerts);
      return "list";
    }

    /** See if it's the list all/active radio */
    reqpar = request.getParameter(reqPars[reqListAll]);

    if (reqpar != null) {
      return "list";
    }

    reqpar = request.getParameter(reqPars[reqList]);

    if (reqpar != null) {
      form.setAlertEvent(alerts);
      return "list";
    }

    reqpar = request.getParameter(reqPars[reqCopy]);

    if (reqpar != null) {
      /** Make the event look new
       */
      form.getEvent().setId(-1);
      form.setAlertEvent(alerts);
      return "add";
    }

    /** =================================================================
     *                   Update actions
     *  ================================================================= */

    reqpar = request.getParameter(reqPars[reqDeleteMultiple]);

    if (reqpar != null) {
      /** They selected some events and they want them deleted
       */
      form.setAlertEvent(alerts);
      return deleteEvents(sess, form, svci);
    }

    reqpar = request.getParameter(reqPars[reqDelete]);

    if (reqpar != null) {
      /** They want an event deleted
       */
      form.setAlertEvent(false);
      return "confirmDelete";
    }

    reqpar = request.getParameter(reqPars[reqDeleteOK]);

    if (reqpar != null) {
      /** They really do want an event deleted
       */
      return deleteEvent(sess, form, svci);
    }

    reqpar = request.getParameter(reqPars[reqAdd]);

    if (reqpar != null) {
      if (!form.validateEvent()) {
        return "retryAdd";
      }

      addEvent(form, svci, alerts);
      return "continue";
    }

    reqpar = request.getParameter(reqPars[reqUpdate]);

    if (reqpar != null) {
      /** We are just updating from the current form values.
       */
      if (!form.validateEvent()) {
        return "retryMod";
      }

      updateEvent(sess, form, svci, alerts);

      return "continue";
    }

    reqpar = request.getParameter("action");

    if (String.valueOf(reqpar).equals("fetch")) {
      return fetchEvent(request, form, svci);
    }

    err.emit("org.ucal.calendar.pubevents.error.unknownaction");
    return "error";
  }

  /** ===================================================================
   *                   Private methods
   *  =================================================================== */

  /** Fetch an event for editing
   *
   * @param request   Needed to locate session
   * @param form      Action form
   * @return String   forward name
   */
  private String fetchEvent(HttpServletRequest request,
                            PEActionForm form,
                            CalSvcI svci) throws Throwable {
    /** User requested an event from the list. Retrieve it, embed it in
     * the form so we can display the page
     */
    int id = form.getEvent().getId();
    if (id == 0) {
      return "success"; // Back to main menu
    }

    if (id < 0) {
      err.emit("org.ucal.calendar.pubevents.error.nosuchevent", id);
      return "notFound";
    }

    EventVO event = svci.getEvent(id);

    if (debug) {
      if (event == null) {
        log.debug("No event with id " + id);
      } else {
        log.debug("Retrieved event " + event.getId());
        log.debug("          start=" + event.getStartdate());
      }
    }

    /** ************************************
        We should ensure the alert status is correct
        ********************************* */

    form.setEvent(event);
    if (event == null) {
      err.emit("org.ucal.calendar.pubevents.error.nosuchevent", id);
      return "notFound";
    }

    return "continue";
  }

  /** Add a single event specified by the form event
   *
   * @param form      Action form
   * @param svci      CalSvcI service interface
   * @param alerts    true if we're doing an alert
   */
  private void addEvent(PEActionForm form,
                        CalSvcI svci,
                        boolean alerts) throws Throwable {
    EventVO ev = form.getEvent();

    ev.setStartdate(form.getEventStartDate().getDate());
    ev.setStarttime(form.getEventStartDate().getTime());
    ev.setEnddate(form.getEventEndDate().getDate());
    ev.setEndtime(form.getEventEndDate().getTime());
    ev.setPublic(true);

    SponsorVO sp = form.getSponsor();
    LocationVO l = form.getLocation();

    svci.addPublicEvent(l, sp, ev);
    if (!alerts) {
      form.updatePrefs(ev.getKeywords(), sp, l);
    }

    form.resetEvent();

    msg.emit("org.ucal.calendar.pubevents.message.event.added");
  }

  /** Update a single event specified by the form event
   *
   * @param sess      UWCalSession calendar session object
   * @param form      Action form
   * @param svci      CalSvcI service interface
   * @param alerts    true if we're doing an alert
   */
  private void updateEvent(UWCalSession sess,
                           PEActionForm form,
                           CalSvcI svci,
                           boolean alerts) throws Throwable {
    EventVO ev = form.getEvent();

    ev.setStartdate(form.getEventStartDate().getDate());
    ev.setStarttime(form.getEventStartDate().getTime());
    ev.setEnddate(form.getEventEndDate().getDate());
    ev.setEndtime(form.getEventEndDate().getTime());
    ev.setPublic(true);

    SponsorVO sp = form.getSponsor();
    LocationVO l = form.getLocation();

    // If keyword changed, we need to change event_keywords entry
    int oldkid = form.getOriginalKeywordId();
    int kid = form.getKeywordId();

    if (debug) {
      if (oldkid != kid) {
        log.debug("Old keyword id = " +  oldkid +
                  " New event keyword id = " + kid);
      } else {
        log.debug("No old keyword");
      }
    }

    svci.updatePublicEvent(l, sess.getAutoDeleteLocations(),
                           sp, sess.getAutoDeleteSponsors(),
                           oldkid, kid, ev);

    if (!alerts) {
      form.updatePrefs(ev.getKeywords(), sp, l);
    }

    form.resetEvent();
    svci.touch();

    msg.emit("org.ucal.calendar.pubevents.message.event.updated");
  }

  /** Delete a number of events specified by the form flags.
   * Unused and untested
   *
   * @param sess      UWCalSession calendar session object
   * @param form      Action form
   * @param svci      CalSvcI service interface
   * @return String   forward name
   */
  private String deleteEvents(UWCalSession sess,
                              PEActionForm form,
                              CalSvcI svci) throws Throwable {
    int[] flags = form.getFlags();

    if (flags == null) {
      return "done";
    }

    /** This could do with a confirmation page but that will require some
        restructuring.

        Do later.
     */
    for (int i = 0; i < flags.length; i++) {
      int eventid = flags[i];

      if (debug) {
        log.debug("About to delete " + eventid);
      }

      svci.deletePublicEvent(sess.getAutoDeleteLocations(),
                             sess.getAutoDeleteSponsors(),
                             svci.getEvent(eventid));
    }

    return "done";
  }

  /** Delete a single event specified by the form event
   *
   * @param sess      UWCalSession calendar session object
   * @param form      Action form
   * @param svci      CalSvcI service interface
   * @return String   forward name
   */
  private String deleteEvent(UWCalSession sess,
                             PEActionForm form,
                             CalSvcI svci) throws Throwable {
    int eventid = form.getEvent().getId();

    if (debug) {
      log.debug("About to delete event " + eventid);
    }

    svci.deletePublicEvent(sess.getAutoDeleteLocations(),
                           sess.getAutoDeleteSponsors(),
                           svci.getEvent(eventid));

    msg.emit("org.ucal.calendar.pubevents.message.event.deleted");

    return "done";
  }
}

