/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.cct.uwcal.admin;

import edu.rpi.cct.uwcal.webcommon.UWCalActionFormBase;
import edu.rpi.cct.uwcal.webcommon.UWCalSession;

import edu.washington.cac.calfacade.shared.AdminGroups;
import edu.washington.cac.calfacade.shared.AdminGroupVO;

import javax.servlet.http.HttpServletRequest;

/** This action manages Admin Groups.
 *
 * @author Mike Douglass   douglm@rpi.edu
 */
public class PEAdminGroupsAction extends PEAbstractAction {
  /** This is the routine which does the work.
   *
   * @param request   Needed to locate session
   * @param sess      UWCalSession calendar session object
   * @param form      Action form
   * @return String   forward name
   */
  public String doAction(HttpServletRequest request,
                         UWCalSession sess,
                         PEActionForm form) throws Throwable {
    /** =================================================================
     *                   Selecting a group - any access if no group set
     *  ================================================================= */

    AdminGroups adgrps = form.getCalSvcI().getAdminGroups();
    form.assignChoosingGroup(false); // reset

    /** Check access
     */
    if (!form.getUserAuth().isContentAdminUser()) {
      return "noAccess";
    }

    String reqpar = request.getParameter("initSwitchGroup");

    if (reqpar != null) {
      form.assignGroupSet(false);
      form.assignChoosingGroup(false);

      // Back to main menu. Abstract action will do the rest.

      String temp = checkGroup(request, form, false);
      if (temp == null) {
        err.emit("org.ucal.calendar.pubevents.error.choosegroupsuppressed");
        return "error";
      }

      return temp;
    }

    /** Check access
     */
    if (!form.getUserAuth().isSuperUser()) {
      return "noAccess";
    }

    /** =================================================================
     *                   Initial admin group actions
     *  ================================================================= */

    reqpar = request.getParameter("listAdminGroups");

    if (reqpar != null) {
      return "listAdminGroups";
    }

    /** =================================================================
     *                   Admin group actions
     *  ================================================================= */

    reqpar = request.getParameter("fetch");

    if (reqpar != null) {
      /** User requested an admin group from the list or by entering the name.
       */
      String name = request.getParameter("adminGroupName");
      if (name == null) {
        return "done";
      }

      AdminGroupVO ag = adgrps.findGroup(name);

      if (debug) {
        if (ag == null) {
          logIt("No group with name " + name);
        } else {
          logIt("Retrieved admin group " + ag.getName());
        }
      }

      form.setUpdAdminGroup(ag);

      return "updateAdminGroup";
    }

    /** =================================================================
     *      Following act on the updAdminGroup
     *  ================================================================= */

    AdminGroupVO updgrp = form.getUpdAdminGroup();

    if (updgrp == null) {
      // That's not right
      return "done";
    }

    reqpar = request.getParameter("newAdminGroup");
    if (reqpar != null) {
      form.setUpdAdminGroup(null);
      return "newAdminGroup";
    }

    reqpar = request.getParameter("removeAdminGroup");
    if (reqpar != null) {
      return "removeAdminGroup";
    }

    reqpar = request.getParameter("removeAdminGroupOK");
    if (reqpar != null) {
      return removeAdminGroup(form, adgrps, updgrp);
    }

    reqpar = request.getParameter("updateAdminGroup");
    if (reqpar != null) {
      return updateAdminGroup(form, adgrps, updgrp);
    }

    reqpar = request.getParameter("addAdminGroup");
    if (reqpar != null) {
      return addAdminGroup(form, adgrps, updgrp);
    }

    reqpar = request.getParameter("addGroupMember");

    if (reqpar != null) {
      /** Add a user to the group we are updating.
       */
      return addMember(form, adgrps, updgrp);
    }

    reqpar = request.getParameter("removeGroupMember");

    if (reqpar != null) {
      /** Remove a user from the group we are updating.
       */
      return removeMember(form, adgrps, updgrp);
    }

    return "success";
  }

  private String removeAdminGroup(PEActionForm form,
                                  AdminGroups adgrps,
                                  AdminGroupVO updgrp) throws Throwable {
    adgrps.removeGroup(updgrp.getName());
    form.setUpdAdminGroup(null);

    return "done";
  }

  private String updateAdminGroup(PEActionForm form,
                                  AdminGroups adgrps,
                                  AdminGroupVO updgrp) throws Throwable {
    if (!form.validateActionGroup()) {
      return "updateAdminGroup";
    }

    adgrps.updateGroup(updgrp);

    /** Refetch the group
     */
    form.setUpdAdminGroup(adgrps.findGroup(updgrp.getName()));

    return "continue";
  }

  private String addAdminGroup(PEActionForm form,
                               AdminGroups adgrps,
                               AdminGroupVO updgrp) throws Throwable {
    if (!form.validateActionGroup()) {
      return "newAdminGroup";
    }

    adgrps.makeGroup(updgrp.getName(),
                     updgrp.getDescription(),
                     updgrp.getGroupOwnerid(),
                     updgrp.getOwnerid());

    /** Refetch the group
     */
    form.setUpdAdminGroup(adgrps.findGroup(updgrp.getName()));

    return "continue";
  }

  /** Add a member to the group. Do nothing if they didn't give us a member
   * to add.
   *
   */
  private String addMember(PEActionForm form,
                           AdminGroups adgrps,
                           AdminGroupVO updgrp) throws Throwable {
    if (form.getUpdGroupMember() == null) {
      return "done";
    }

    adgrps.addMember(updgrp.getName(), form.getUpdGroupMember());

    return updateAdminGroup(form, adgrps, updgrp);
  }

  private String removeMember(PEActionForm form,
                              AdminGroups adgrps,
                              AdminGroupVO updgrp) throws Throwable {
    if (form.getUpdGroupMember() == null) {
      return "done";
    }

    adgrps.removeMember(updgrp.getName(), form.getUpdGroupMember());

    return updateAdminGroup(form, adgrps, updgrp);
  }
}

