/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.cct.uwcal.admin;

import edu.rpi.cct.uwcal.common.IntSelectId;
import edu.rpi.cct.uwcal.webcommon.UWCalActionFormBase;

import edu.washington.cac.calfacade.shared.AdminGroupROVO;
import edu.washington.cac.calfacade.shared.AdminGroups;
import edu.washington.cac.calfacade.shared.AdminGroupVO;
import edu.washington.cac.calfacade.shared.AuthUserVO;
import edu.washington.cac.calfacade.shared.CalFacadeException;
import edu.washington.cac.calfacade.shared.CheckData;
import edu.washington.cac.calfacade.shared.EventVO;
import edu.washington.cac.calfacade.shared.KeywordAttrsVO;
import edu.washington.cac.calfacade.shared.KeywordVO;
import edu.washington.cac.calfacade.shared.LocationVO;
import edu.washington.cac.calfacade.shared.SponsorVO;
import edu.washington.cac.calfacade.shared.UserAuth;

import java.sql.Date;
import java.sql.Time;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Vector;
import javax.servlet.http.HttpServletRequest;
import org.apache.struts.action.ActionMapping;

public class PEActionForm extends UWCalActionFormBase implements PEDefs {
  /** Id doing administration, May be a group id */
  private String adminUserId;

  /** Entry for the currently logged in user
   */
  private AuthUserVO curAuthUser;

  private String calendarDefs;
  /** ...................................................................
   *                   Flagged items - order not significant
   *  ................................................................... */

  private HashMap flags;

  /** ...................................................................
   *                   Event fields
   *  ................................................................... */

  private EventVO event;

  /** True if we are adding an alert
   */
  private boolean alertEvent;

  /** True if we should list all events
   */
  private boolean listAllEvents;

  /** ...................................................................
   *                   Keywords
   *  ................................................................... */

  private KeywordVO keyword;

  /** Set if we appear to be changing the event keyword.
   */
//  private KeywordVO oldKeyword;

  private IntSelectId keyId;

  /** true if this request had alert checkbox set.
   */
  private boolean keyAlertFlag;

  private KeywordVO[] preferredKeywords;

  /** ...................................................................
   *                   Sponsor fields
   *  ................................................................... */

  private SponsorVO sponsor;

  private boolean doUnmakePhone = true;

  private String phoneAreaCode; // Default from properties

  private String phoneBegin; // Default from properties

  private String phoneEnd;

  private String phoneExtra;

  private IntSelectId spId;

  private SponsorVO[] preferredSponsors;

  /** ...................................................................
   *                   Locations
   *  ................................................................... */

  private LocationVO location;

  private IntSelectId locId;

  private LocationVO[] preferredLocations;

  /** ...................................................................
   *                   Authorised user fields
   *  ................................................................... */

  /** Entry we are modifying
   */
  private AuthUserVO authUser;

  /** Value built out of checked boxes.
   */
  private int newAuthUserType;

  /** User we want to fetch or modify
   */
  private String authUserId;

  /** ...................................................................
   *                   Admin group fields
   *  ................................................................... */

  /** User's current group or null. */
  private AdminGroupROVO adminGroup;

  /** True if we have set the user's group.
   */
  private boolean groupSet;

  /** True if we are choosing the user's group.
   */
  private boolean choosingGroup;

  /** The groups of which our user is a member
   */
  private AdminGroupVO[] userAdminGroups;

  private AdminGroupVO updAdminGroup;

  /** Group member to add/delete
   */
  private String updGroupMember;

  /** True to show members in list
   */
  private boolean showAgMembers;

  /** ===================================================================
   *                   UserAuth Methods
   * DO NOT return userAuth. We don't want the user auth object
   * accessible to the request. Use the RO object instead.
   *  =================================================================== */

  /** See if user has some form of access
   */
  public boolean getAuthorisedUser() {
    if (userAuth == null) {
      return false;
    }

    try {
      return userAuth.getUsertype() != UserAuth.noPrivileges;
    } catch (Throwable t) {
      err.emit(t);
      return false;
    }
  }

  /** This will default to the current user. Superusers will be able to
   * specify a creator.
   *
   * @param val    String creator used to limit searches
   */
  public void setAdminUserId(String val) {
    adminUserId = val;
  }

  public String getAdminUserId() {
    return adminUserId;
  }

  public void setCurAuthUser(AuthUserVO val) {
    curAuthUser = val;
  }

  public AuthUserVO getCurAuthUser() {
    return curAuthUser;
  }

  /** ===================================================================
   *                   Methods
   *  =================================================================== */

  public void setCalendarDefs(String val) {
    calendarDefs = val;
  }

  public String getCalendarDefs() {
    return calendarDefs;
  }

  /** ===================================================================
   *                   Flags - indexed checkboxes
   *  =================================================================== */

  public void setDelFlag(int i, boolean val) {
    if (!val) {
      return;
    }

    if (flags == null) {
      flags = new HashMap();
    }

    flags.put(new Integer(i), null);
  }

  public boolean getDelFlag(int i) {
    if (flags == null) {
      return false;
    }

    return flags.containsKey(new Integer(i));
  }

  public int[] getFlags() {
    if ((flags == null) || (flags.size() == 0)) {
      return null;
    }

    int[] flagArray = new int[flags.size()];

    Iterator it = flags.keySet().iterator();
    int i = 0;

    while (it.hasNext()) {
      flagArray[i] = ((Integer)it.next()).intValue();
      i++;
    }

    return flagArray;
  }

  /** ===================================================================
   *                   Events
   *  =================================================================== */

  public void setListAllEvents(boolean val) {
    listAllEvents = val;
  }

  public boolean getListAllEvents() {
    return listAllEvents;
  }

  public void setEvent(EventVO val) {
    event = val;

    if (val == null) {
      resetEventStartDate();
      resetEventEndDate();

      getEvent();

      long now = System.currentTimeMillis();

      event.setStartdate(new Date(now));
      event.setStarttime(new Time(now));
      event.setEnddate(new Date(now));
      event.setEndtime(new Time(now));
    }

    getEventStartDate().setDateTime(event.getStartdate(), event.getStarttime());
    getEventEndDate().setDateTime(event.getEnddate(), event.getEndtime());

    resetEvent();
  }

  public void resetEvent() {
    getEvent(); // Make sure we have one

    /* Implant the current id(s) in new entries */
    int id = 0;
    KeywordVO k = event.getKeyword(0);
    if (k != null) {
      id = k.getId();
      setKeyword(k);
    }

    /* A is the All box, B is the user preferred values. */
    keyId = new IntSelectId(id, IntSelectId.AHasPrecedence);

    SponsorVO s = event.getSponsor();
    id = 0;
    if (s != null) {
      id = s.getId();
      setSponsor(s);
    }

    spId = new IntSelectId(id, IntSelectId.AHasPrecedence);

    LocationVO l = event.getLocation();
    id = 0;
    if (l != null) {
      id = l.getId();
      setLocation(l);
    }

    locId = new IntSelectId(id, IntSelectId.AHasPrecedence);
  }

  /** If an event object exists, return that otherwise create an empty one.
   *
   * @return EventVO  populated event value object
   */
  public EventVO getEvent() {
    if (event == null) {
      event = new EventVO();
    }
    return event;
  }

  /** Return events, if doing alerts we pick them out otherwise exclude them
   *
   * @return EventVO[]  populated event value objects
   */
  public EventVO[] getEvents() {
    try {
      EventVO[] evs;

      if (listAllEvents) {
        evs = getCalSvcI().getPublicEvents(alertEvent);
      } else {
        evs = getCalSvcI().getActivePublicEvents(alertEvent);
      }

      if (evs == null) {
        return new EventVO[0];
      }

      return evs;
    } catch (Throwable t) {
      err.emit(t);
      return new EventVO[0];
    }
  }

  /** Get the i'th keyword id from the event object. Return &lt; 0 for none.
   *
   * @param i        int index into events vector of keywords
   * @return int     Keyword index or -1
   */
  public int getEventKeywordId(int i) {
    Vector v = getEvent().getKeywords();

    if ((v == null) || (i >= v.size())) {
      return -1;
    }

    KeywordVO k = (KeywordVO)v.elementAt(i);

    return k.getId();
  }

  /** Get the i'th keyword from the event object. Return null for none.
   *
   * @param i           int index into events vector of keywors
   * @return KeywordVO  Keyword object or null
   */
  public KeywordVO getEventKeyword(int i) {
    Vector v = getEvent().getKeywords();

    if ((v == null) || (i >= v.size())) {
      return null;
    }

    return (KeywordVO)v.elementAt(i);
  }

  public void setAlertEvent(boolean val) {
    alertEvent = val;
  }

  public boolean getAlertEvent() {
    return alertEvent;
  }

  public int getMaxDescriptionLength() {
    return EventVO.maxLongdescLength;
  }

  /** ===================================================================
   *                   Keywords
   *  =================================================================== */

  public void setKeyword(KeywordVO val) {
    keyword = val;
    if (val != null) {
      keyAlertFlag = getKeyAlert(val.getId());
    }
  }

  /** If a Keyword object exists, return that otherwise create an empty one.
   *
   * @return KeywordVO  Keyword value object
   */
  public KeywordVO getKeyword() {
    if (keyword == null) {
      keyword = new KeywordVO();
    }

    return keyword;
  }

  /** We have a preferred and all keywords form field. One of them will be
   * unset so we ignore negative values.
   */
  public void setKeywordId(int val) {
    if (val >= 0) {
      keyId.setA(val);
    }
  }

  public int getKeywordId() {
    return getKeyword().getId();
  }

  public int getOriginalKeywordId() {
    if (keyId == null) {
      return 0;
    }

    return keyId.getOriginalVal();
  }

  public void setPrefKeywordId(int val) {
    if (val >= 0) {
      keyId.setB(val);
    }
  }

  public int getPrefKeywordId() {
    return getKeyword().getId();
  }

  /** Return public keywords excluding alerts.
   */
  public KeywordVO[] getPublicKeywords() {
    try {
      KeywordVO[] ks = getCalSvcI().getPublicKeywords();
      if (ks == null) {
        return new KeywordVO[0];
      }

      return ks;
   } catch (Throwable t) {
      err.emit(t);
      return new KeywordVO[0];
    }
  }

  /** Return public keywords including alerts.
   */
  public KeywordVO[] getAllPublicKeywords() {
    try {
      KeywordVO[] ks = getCalSvcI().getAllPublicKeywords();
      if (ks == null) {
        return new KeywordVO[0];
      }

      return ks;
    } catch (Throwable t) {
      err.emit(t);
      return new KeywordVO[0];
    }
  }

  /** Return alert keywords only
   */
  public KeywordVO[] getAlertKeywords() {
    try {
      KeywordVO[] ks = getCalSvcI().getAlertKeywords();
      if (ks == null) {
        return new KeywordVO[0];
      }

      return ks;
   } catch (Throwable t) {
      err.emit(t);
      return new KeywordVO[0];
    }
  }

  /* * Set the old keyword (before any change)
   *
   * @param val    KeywordVO old Keyword or null
   * /
  public void setOldKeyword(KeywordVO val) {
    oldKeyword = val;
  }

  /* * Get the old keyword (before any change)
   *
   * @return KeywordVO    old keyword or null
   * /
  public KeywordVO getOldKeyword() {
    return oldKeyword;
  } */

  /** Is given keyword an alert?
   *
   * @param id    keyword id
   */
  public boolean getKeyAlert(int id) {
    try {
      return getCalSvcI().getKeyAlert(id);
    } catch (Throwable t) {
      err.emit(t);
      return false;
    }
  }

  public void setKeyAlertFlag(boolean val) {
    keyAlertFlag = val;
  }

  public boolean getKeyAlertFlag() {
    return keyAlertFlag;
  }

  /** Get the preferred keywords for the current user
   *
   * @return KeywordVO[]  preferred keywords
   */
  public KeywordVO[] getPreferredKeywords() {
    try {
      if (preferredKeywords == null) {
        preferredKeywords = getCalSvcI().getPublicKeywords(
                getCurAuthUser().getPreferredKeywords());
      }
    } catch (Throwable t) {
      err.emit(t);
    }

    return preferredKeywords;
  }

  /** Clear the preferred keywords for the current user after a change
   */
  public void clearPreferredKeywords() {
    preferredKeywords = null;
  }

  /** ===================================================================
   *                   Sponsors
   *  =================================================================== */

  public int getMaxReservedSponsorId() {
    return SponsorVO.maxReservedId;
  }

  public void setSponsor(SponsorVO val) {
    sponsor = val;
    doUnmakePhone = true;
  }

  /** If a sponsor object exists, return that otherwise create an empty one.
   *
   * @return SponsorVO  sponsor value object
   */
  public SponsorVO getSponsor() {
    if (sponsor == null) {
      sponsor = new SponsorVO();
      defaultPhone();
    } else {
      unMakePhone();
    }

    return sponsor;
  }

  /** We have a preferred and all sponsors form field. One of them may be
   * unset so we ignore negative values.
   */
  public void setSponsorId(int val) {
    if (val >= 0) {
      spId.setA(val);
    }
  }

  /** Needed for a select box
   */
  public int getSponsorId() {
    return getSponsor().getId();
  }

  public int getOriginalSponsorId() {
    if (spId == null) {
      return 0;
    }

    return spId.getOriginalVal();
  }

  public void setPrefSponsorId(int val) {
    if (val >= 0) {
      spId.setB(val);
    }
  }

  /** Needed for a select box
   */
  public int getPrefSponsorId() {
    return getSponsor().getId();
  }

  public void setPhoneAreaCode(String val) {
    phoneAreaCode = checkNull(val);
  }

  public String getPhoneAreaCode() {
    return phoneAreaCode;
  }

  public void setPhoneBegin(String val) {
    phoneBegin = checkNull(val);
  }

  public String getPhoneBegin() {
    return phoneBegin;
  }

  public void setPhoneEnd(String val) {
    phoneEnd = checkNull(val);
  }

  public String getPhoneEnd() {
    return phoneEnd;
  }

  public void setPhoneExtra(String val) {
    phoneExtra = checkNull(val);
  }

  public String getPhoneExtra() {
    return phoneExtra;
  }

  /** Get the preferred sponsors for the current user
   *
   * @return SponsorVO[]  preferred sponsors
   */
  public SponsorVO[] getPreferredSponsors() {
    try {
      if (preferredSponsors == null) {
        preferredSponsors = getCalSvcI().getPublicSponsors(
                getCurAuthUser().getPreferredSponsors());
      }
    } catch (Throwable t) {
      err.emit(t);
    }

    return preferredSponsors;
  }

  /** Clear the preferred sponsors for the current user after a change
   */
  public void clearPreferredSponsors() {
    preferredSponsors = null;
  }

  /** ===================================================================
   *                   Locations
   *  =================================================================== */

  public void setLocation(LocationVO val) {
    location = val;
  }

  /** If a location object exists, return that otherwise create an empty one.
   *
   * @return LocationVO  populated location value object
   */
  public LocationVO getLocation() {
    if (location == null) {
      location = new LocationVO();
    }

    return location;
  }

  /** We have a preferred and all locations form field. One of them will be
   * unset so we ignore negative values.
   */
  public void setLocationId(int val) {
    if (val >= 0) {
      locId.setA(val);
    }
  }

  public int getLocationId() {
    return getLocation().getId();
  }

  public int getOriginalLocationId() {
    if (locId == null) {
      return 0;
    }

    return locId.getOriginalVal();
  }

  public void setPrefLocationId(int val) {
    if (val >= 0) {
      locId.setB(val);
    }
  }

  public int getPrefLocationId() {
    return getLocation().getId();
  }

  /** Get the preferred locations for the current user
   *
   * @return LocationVO[]  preferred locations
   */
  public LocationVO[] getPreferredLocations() {
    try {
      if (preferredLocations == null) {
        preferredLocations = getCalSvcI().getPublicLocations(
                getCurAuthUser().getPreferredLocations());
      }
    } catch (Throwable t) {
      err.emit(t);
    }

    return preferredLocations;
  }

  /** Clear the preferred locations for the current user after a change
   */
  public void clearPreferredLocations() {
    preferredLocations = null;
  }

  /** ===================================================================
   *                   Authorised user
   *  =================================================================== */

  /** Show whether user entries can be displayed or modified with this
   * class. Some sites may use other mechanisms.
   *
   * @return boolean    true if user maintenance is implemented.
   */
  public boolean getUserMaintOK() {
    return userAuth.getUserMaintOK();
  }

  public void setAuthUser(AuthUserVO val) {
    authUser = val;
  }

  public AuthUserVO getAuthUser() {
    return authUser;
  }

  public AuthUserVO[] getAllAuthUsers() {
    try {
      return retrieveUserAuth().getAll();
    } catch (Throwable t) {
      err.emit(t);
      return new AuthUserVO[0];
    }
  }

  /** Only called if the flag is set.
   */
  public void setAuthUserAlerts(boolean val) {
    newAuthUserType |= UserAuth.alertUser;
  }

  public boolean getAuthUserAlerts() {
    return (getAuthUser().getUsertype() & UserAuth.alertUser) != 0;
  }

  /** Only called if the flag is set.
   */
  public void setAuthUserPublicEvents(boolean val) {
    newAuthUserType |= UserAuth.publicEventUser;
  }

  public boolean getAuthUserPublicEvents() {
    return (getAuthUser().getUsertype() & UserAuth.publicEventUser) != 0;
  }

  /** Only called if the flag is set.
   */
  public void setAuthUserSuperUser(boolean val) {
    newAuthUserType |= UserAuth.superUser;
  }

  public boolean getAuthUserSuperUser() {
    return (getAuthUser().getUsertype() & UserAuth.superUser) != 0;
  }

  public void setAuthUserId(String val) {
    authUserId = val;
  }

  public String getAuthUserId() {
    return authUserId;
  }

  /** ===================================================================
   *                   Admin groups
   *  =================================================================== */

  /** Implant the current user group (or null for none) in the form.
   * Do NOT call the setAdminGroup - it should not be visible to the request
   * stream.
   *
   * @param val      AdminGroupROVO representing users group or null
   */
  public void assignAdminGroup(AdminGroupROVO val) {
    adminGroup = val;
    groupSet = true;
  }

  public AdminGroupROVO getAdminGroup() {
    return adminGroup;
  }

  public void assignGroupSet(boolean val) {
    groupSet = val;
  }

  public boolean getGroupSet() {
    return groupSet;
  }

  public void assignChoosingGroup(boolean val) {
    choosingGroup = val;
  }

  public boolean retrieveChoosingGroup() {
    return choosingGroup;
  }

  /** The groups of which our user is a member
   */
  public void setUserAdminGroups(AdminGroupVO[] val) {
    userAdminGroups = val;
  }

  public AdminGroupVO[] getUserAdminGroups() {
    return userAdminGroups;
  }

  /** Show whether admin group maintenance is available.
   * Some sites may use other mechanisms.
   *
   * @return boolean    true if admin group maintenance is implemented.
   */
  public boolean getAdminGroupMaintOK() {
    try {
      return getCalSvcI().getAdminGroups().getGroupMaintOK();
   } catch (Throwable t) {
      err.emit(t);
      return false;
    }
  }

  public AdminGroupVO[] getAdminGroups() {
    try {
      AdminGroupVO[] ags = getCalSvcI().getAdminGroups().getAll();
      if (ags == null) {
        return new AdminGroupVO[0];
      }

      return ags;
   } catch (Throwable t) {
      err.emit(t);
      return new AdminGroupVO[0];
    }
  }

  public void setUpdAdminGroup(AdminGroupVO val) {
    updAdminGroup = val;
  }

  public AdminGroupVO getUpdAdminGroup() {
    if (updAdminGroup == null) {
      updAdminGroup = new AdminGroupVO();
    }

    return updAdminGroup;
  }

  public void setUpdGroupMember(String val) {
    updGroupMember = val;
  }

  public String getUpdGroupMember() {
    return updGroupMember;
  }

  public void setShowAgMembers(boolean val) {
    showAgMembers = val;
  }

  public boolean getShowAgMembers() {
    return showAgMembers;
  }

  /** ===================================================================
   *                   Validation methods
   *  =================================================================== */

  /** Ensure the event has all required fields and all are valid.
   *
   * @return boolean   true   event looks OK
   */
  public boolean validateEvent() throws Throwable {
    boolean ok = validateEventKeyword();
    EventVO ev = getEvent();

    if (!validateEventSponsor()) {
      ok = false;
    }

    if (!validateEventLocation()) {
      ok = false;
    }

    ev.setShortdesc(checkNull(getEvent().getShortdesc()));
    ev.setLongdesc(checkNull(getEvent().getLongdesc()));

    if (ev.getShortdesc() == null) {
      err.emit("org.ucal.calendar.pubevents.error.missingfield",
               "Title");
      ok = false;
    }

    if (ev.getLongdesc() == null) {
      err.emit("org.ucal.calendar.pubevents.error.missingfield",
               "Description");
      ok = false;
    } else if (ev.getLongdesc().length() > EventVO.maxLongdescLength) {
      err.emit("org.ucal.calendar.pubevents.error.toolongfield",
               "Description", String.valueOf(EventVO.maxLongdescLength));
      ok = false;
    }

    StringBuffer sb = new StringBuffer();

    if (!CheckData.checkDateTimes(getEventStartDate().getDate(),
                                  getEventStartDate().getTime(),
                                  getEventEndDate().getDate(),
                                  getEventEndDate().getTime(),
                                  sb)) {
      err.emit("org.ucal.calendar.error.baddates", sb);
      ok = false;
    }

    return ok;
  }

  public boolean validateEventKeyword() throws Throwable {
    boolean ok = true;
    int id = keyId.getVal();

    if (id <= 0) {
      err.emit("org.ucal.calendar.pubevents.error.missingfield",
               "Category");
      return false;
    }

    if (getCalSvcI().getKeyAlert(id) != alertEvent) {
      err.emit("org.ucal.calendar.pubevents.error.badcatclass");
      return false;
    }

    if (!keyId.getChanged()) {
      return true;
    }

//    oldKeyword = getEvent().getKeyword(0);

    try {
      KeywordVO k = getCalSvcI().getPublicKeyword(id);
      if (k == null) {
        err.emit("org.ucal.calendar.pubevents.error.missingkeyword", id);
        return false;
      }

      getEvent().setKeyword(0, k);
      setKeyword(k);

      return true;
    } catch (Throwable t) {
      err.emit(t);
      return false;
    }
  }

  /** Validate the sponsor provided for an event
   *
   * @return boolean  true OK, false not OK and message(s) emitted.
   */
  public boolean validateEventSponsor() {
    boolean ok = true;

    if (!spId.getChanged()) {
      if (getSession().getAutoCreateSponsors()) {
        return validateNewSponsor();
      }

      if (event.getSponsor() == null) {
        err.emit("org.ucal.calendar.pubevents.error.missingfield",
                 "Sponsor");
        return false;
      }

      return ok;
    }

    // The user selected one from the list
    int id = spId.getVal();

    try {
      SponsorVO s = getCalSvcI().getPublicSponsor(id);
      if (s == null) {
        // Somebody's faking
        setSponsor(null);
        err.emit("org.ucal.calendar.pubevents.error.missingfield",
                 "Sponsor");
        return false;
      }

      getEvent().setSponsor(s);

      setSponsor(s);
      return true;
    } catch (Throwable t) {
      err.emit(t);
      return false;
    }
  }

  public boolean validateNewSponsor() {
    boolean ok = true;

    SponsorVO s = getSponsor();

    if (checkNull(s.getName()) == null) {
      err.emit("org.ucal.calendar.pubevents.error.missingfield",
               "Sponsoring Organization");
      ok = false;
    }

    if (!makePhone()) {
      ok = false;
    }

    s.setEmail(checkNull(s.getEmail()));
    s.setLink(checkNull(s.getLink()));

    String link = s.getLink();
    if (link != null && link.indexOf("://") == -1) {
      s.setLink("http://" + link);
    }

    return ok;
  }

  /** Validate the location provided for an event
   *
   * @return boolean  true OK, false not OK and message(s) emitted.
   */
  public boolean validateEventLocation() {
    boolean ok = true;

    if (!locId.getChanged()) {
      if (getSession().getAutoCreateLocations()) {
        return validateNewLocation();
      }

      if (event.getLocation() == null) {
        err.emit("org.ucal.calendar.pubevents.error.missingfield",
                 "Location");
        return false;
      }

      return ok;
    }

    // The user selected one from the list
    int id = locId.getVal();

    try {
      LocationVO l = getCalSvcI().getPublicLocation(id);
      if (l == null) {
        // Somebody's faking
        setLocation(null);
        err.emit("org.ucal.calendar.pubevents.error.missingfield",
                 "Location");
        return false;
      }

      getEvent().setLocation(l);

      setLocation(l);
      return true;
    } catch (Throwable t) {
      err.emit(t);
      return false;
    }
  }

  public boolean validateNewLocation() {
    boolean ok = true;

    LocationVO l = getLocation();

    l.setAddress(checkNull(l.getAddress()));

    if (l.getAddress() == null) {
      err.emit("org.ucal.calendar.pubevents.error.missingfield",
               "Location");
      ok = false;
    }

    l.setLink(checkNull(l.getLink()));

    String link = l.getLink();
    if (link != null && link.indexOf("://") == -1) {
      l.setLink("http://" + link);
    }

    return ok;
  }

  public boolean validateAuthUser() throws Throwable {
    boolean ok = true;

    AuthUserVO au = getAuthUser();

    au.setUsertype(getCalSvcI().getUserAuth().adjustUsertype(newAuthUserType));
    au.setLastname(checkNull(au.getLastname()));
    au.setFirstname(checkNull(au.getFirstname()));
    au.setUphone(checkNull(au.getUphone()));
    au.setUemail(checkNull(au.getUemail()));
    au.setDept(checkNull(au.getDept()));

    return ok;
  }

  /** Validate a keyword entry after add/mod
   */
  public boolean validateKeyword() throws Throwable {
    boolean ok = true;

    KeywordVO k = getKeyword();

    k.setWord(checkNull(k.getWord()));
    k.setTitle(checkNull(k.getTitle()));
    k.setDescription(checkNull(k.getDescription()));

    if (k.getWord() == null) {
      err.emit("org.ucal.calendar.pubevents.error.missingfield",
               "Keyword");
      ok = false;
    }

    int id = k.getId();

    return ok;
  }

  public boolean validateActionGroup() throws Throwable {
    boolean ok = true;

    if (updAdminGroup == null) {
      // bogus call.
      return false;
    }

    updAdminGroup.setName(checkNull(updAdminGroup.getName()));
    updAdminGroup.setDescription(checkNull(updAdminGroup.getDescription()));
    updAdminGroup.setGroupOwnerid(checkNull(updAdminGroup.getGroupOwnerid()));
    updAdminGroup.setOwnerid(checkNull(updAdminGroup.getOwnerid()));

    if (updAdminGroup.getName() == null) {
      err.emit("org.ucal.calendar.pubevents.error.missingfield",
               "Name");
      ok = false;
    }

    if (updAdminGroup.getDescription() == null) {
      err.emit("org.ucal.calendar.pubevents.error.missingfield",
               "description");
      ok = false;
    }

    if (updAdminGroup.getGroupOwnerid() == null) {
      err.emit("org.ucal.calendar.pubevents.error.missingfield",
               "groupOwnerid");
      ok = false;
    }

    if (updAdminGroup.getOwnerid() == null) {
      updAdminGroup.setOwnerid(updAdminGroup.getName());
    }

    String prefix =
        mres.getMessage("org.ucal.calendar.app.admingroupsidprefix");

    if (!updAdminGroup.getOwnerid().startsWith(prefix)) {
      updAdminGroup.setOwnerid(prefix + updAdminGroup.getOwnerid());
    }

    return ok;
  }


  /** Make up a phone number from the available material. This is a (sort of)
   * copy of the original pubevents code. We should probably default area
   * code and the exhange number (from the properties).
   *
   * <p>We should also turn it into the standard form with country code etc
   *
   * @return boolean  false means something wrong, message emitted
   */
  public boolean makePhone() {
    if (!checkDigits("area code", phoneAreaCode, 3) ||
        !checkDigits("phone number", phoneBegin, 3) ||
        !checkDigits("phone number", phoneEnd, 4)) {
      return false;
    }

    StringBuffer sb = new StringBuffer(12);

    sb.append(phoneAreaCode);
    sb.append('-');
    sb.append(phoneBegin);
    sb.append('-');
    sb.append(phoneEnd);
    if (phoneExtra != null) {
      sb.append(' ');
      sb.append(phoneExtra);
    }

    getSponsor().setPhone(sb.toString());
    return true;
  }

  public void initFields() {
    event = null;
    keyword = null;
//    oldKeyword = null;
    sponsor = null;
    doUnmakePhone = true;
    location = null;
    authUser = null;
    updGroupMember = null;
  }

  /** Update an authorised users preferences to reflect usage.
   *
   * @param keywords Vector of KeywordVO objects
   * @param sp       SponsorVO object
   * @param loc      LocationVO object
   */
  public void updatePrefs(Vector keywords, SponsorVO sp, LocationVO loc)
          throws CalFacadeException {
    AuthUserVO au = getCurAuthUser();
    UserAuth ua = retrieveUserAuth();

    /** If we have auto add on, add to the users preferred list(s)
     */

    if ((keywords != null) && (au.getAutoAddKeywords())) {
      Enumeration enum = keywords.elements();
      while (enum.hasMoreElements()) {
        KeywordVO k = (KeywordVO)enum.nextElement();
        if (ua.addKeyword(au, k.getId())) {
          clearPreferredKeywords();
        }
      }
    }

    if ((sp != null) && (au.getAutoAddSponsors())) {
      if (ua.addSponsor(au, sp.getId())) {
        clearPreferredSponsors();
      }
    }

    if ((loc != null) && (au.getAutoAddLocations())) {
      if (ua.addLocation(au, loc.getId())) {
        clearPreferredLocations();
      }
    }
  }

  /** ===================================================================
   *                   Private methods
   *  =================================================================== */

  /** Check if a String is composed only of digits and is the proper length
   *
   * @param
   * @param s              String in question
   * @param expectedLength int length the string should be
   * @return boolean       true if the fiedl is OK. False a message was
   *                       emitted
   */
  private boolean checkDigits(String fldName, String s, int expectedLength)
      throws NumberFormatException {
    if (s == null || s.length() != expectedLength) {
      err.emit("org.ucal.calendar.pubevents.error.badfield",
               fldName);
      return false;
    }

    if (!allDigits(s)) {
      err.emit("org.ucal.calendar.pubevents.error.badfield",
               fldName);
      return false;
    }

    return true;
  }

  /** Check if the string is all digits
   *
   * @return boolean  true means string made up entirely of digits
   */
  private boolean allDigits(String s) {
    for (int i = 0; i < s.length(); i++) {
      if (!Character.isDigit(s.charAt(i))) {
        return false;
      }
    }

    return true;
  }

  /** Take the phone number in the sponsor object and distribute it over
   * the various fields.
   */
  private void unMakePhone() {
    if (!doUnmakePhone) {
      return;
    }

    doUnmakePhone = false;

    String phone = getSponsor().getPhone();

    defaultPhone();

    if (phone == null) {
      return;
    }

    /** These numbers should all be of the form
        nnn-nnn-nnnn
        optionally followed by a single space and extra information
        (probably extension)

       If it's not we'll dump it all in phoneExtra
     */
    if ((phone.length() < 12) ||
        (phone.charAt(3) != '-') ||
        (phone.charAt(7) != '-') ||
        ((phone.length() > 12) && (phone.charAt(12) != ' '))) {
      // Invalid
      phoneExtra = phone;
      return;
    }

    phoneAreaCode = phone.substring(0, 3);
    phoneBegin = phone.substring(4, 7);
    phoneEnd = phone.substring(8, 12);

    if (phone.length() > 13) {
      phoneExtra = phone.substring(13);
    }
  }

  /** Set default phone values from properties.
   */
  private void defaultPhone() {
    phoneAreaCode = null;
    phoneBegin = null;
    phoneEnd = null;
    phoneExtra = null;
  }

  /**
   * Reset properties to their default values.
   *
   * @param mapping The mapping used to select this instance
   * @param request The servlet request we are processing
   */
  public void reset(ActionMapping mapping, HttpServletRequest request) {
    if (debug) {
      servlet.log("--" + getClass().getName() + ".reset--");
    }

    newAuthUserType = 0;
    keyAlertFlag = false;
  }
}

