/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.cct.uwcal.admin;

import edu.rpi.cct.uwcal.webcommon.UWCalAbstractAction;
import edu.rpi.cct.uwcal.webcommon.UWCalActionFormBase;
import edu.rpi.cct.uwcal.webcommon.UWCalSession;
import edu.rpi.sss.util.jsp.JspUtil;

import edu.washington.cac.calfacade.shared.AdminGroups;
import edu.washington.cac.calfacade.shared.AdminGroupVO;
import edu.washington.cac.calfacade.shared.AuthUserVO;
import edu.washington.cac.calfacade.shared.UserAuth;

import javax.servlet.http.HttpServletRequest;

/** This provides some pubevents specific services to subclasses
 *
 * @author Mike Douglass   douglm@rpi.edu
 */
public abstract class PEAbstractAction extends UWCalAbstractAction {
  /** This is an admin client
   *
   * @return boolean  true for a public events admin client
   */
  public boolean getPublicAdmin() {
    return true;
  }

  /** This is the routine which does the work.
   *
   * @param request   Needed to locate session
   * @param sess      UWCalSession calendar session object
   * @param frm       Action form
   * @return String   forward name
   */
  public String doAction(HttpServletRequest request,
                         UWCalSession sess,
                         UWCalActionFormBase frm) throws Throwable {
    PEActionForm form = (PEActionForm)frm;

    /** Show the owner we are administering */
    form.setAdminUserId(form.getCalSvcI().getUserVO().getName());

    if (debug) {
      logIt("-------- isSuperUser: " + form.getUserAuth().isSuperUser());
    }

    if (!form.getAuthorisedUser()) {
      return "noAccess";
    }

    String temp = checkGroup(request, form, true);
    if (temp != null) {
      return temp;
    }

    /** Ensure we have an AuthUser object for the logged in user
     */
    AuthUserVO au = form.getCurAuthUser();
    if (au == null) {
      UserAuth ua = form.retrieveUserAuth();

      au = ua.getUser(form.getCurrentUser());
      form.setCurAuthUser(au);
    }

    String reqpar = request.getParameter("cancelled");

    if (reqpar != null) {
      /** Set the objects to null so we get new ones.
       */
      form.initFields();
      form.setEvent(null);

      msg.emit("org.ucal.calendar.message.cancelled");
      return "cancelled";
    }

    return doAction(request, sess, form);
  }

  /** This is the abstract routine which does the work.
   *
   * @param request   Needed to locate session
   * @param sess      UWCalSession calendar session object
   * @param form      Action form
   * @return String   forward name
   */
  public abstract String doAction(HttpServletRequest request,
                                  UWCalSession sess,
                                  PEActionForm form) throws Throwable;

  /** Return null if group is chosen else return a forward name.
   *
   * @param request   Needed to locate session
   * @param form      Action form
   * @param initCheck true if this is a check to see if we're initialised,
   *                  otherwise this is an explicit request to change group.
   * @return String   forward name
   */
  String checkGroup(HttpServletRequest request,
                    PEActionForm form,
                    boolean initCheck) throws Throwable {
    if (form.getGroupSet()) {
      return null;
    }

    AdminGroups adgrps = form.getCalSvcI().getAdminGroups();

    if (form.retrieveChoosingGroup()) {
      /** This should be the response to presenting a list of groups.
          We handle it here rather than in a separate action to ensure our
          client is not trying to bypass the group setting.
       */

      String reqpar = request.getParameter("chooseGroup");
      if (reqpar == null) {
        // Make them do it again.

        return "chooseGroup";
      }

      return setGroup(request, form, adgrps,
                      request.getParameter("adminGroupName"));
    }

    /** If the user is in no group or in one group we just go with that,
        otherwise we ask them to select the group
     */

    AdminGroupVO[] adgs;

    if (initCheck || !form.getUserAuth().isSuperUser()) {
      // Always restrict to groups we're a member of
      adgs = adgrps.getAdminGroups(form.getCurrentUser());
    } else {
      adgs = adgrps.getAll();
    }

    if (adgs == null) {
      /** If we require that all users be in a group we return to an error
          page. The only exception will be superUser.
       */

      boolean noGroupAllowed = "true".equals(
                JspUtil.getProperty(getMessages(),
                                    "org.ucal.admin.nogroupallowed",
                                    "false"));
      if (form.getUserAuth().isSuperUser() || noGroupAllowed) {
        form.assignAdminGroup(null);
        return null;
      }

      return "noGroupAssigned";
    }

    if (adgs.length == 1) {
      form.assignAdminGroup(adgs[0]);
      String s = setAdminUser(request, form, adgs[0].getOwnerid(), true);

      if (s != null) {
        return s;
      }

      form.setAdminUserId(form.getCalSvcI().getUserVO().getName());
      return null;
    }

    /** Go ahead and present the possible groups
     */
    form.setUserAdminGroups(adgs);
    form.assignChoosingGroup(true); // reset

    return "chooseGroup";
  }

  private String setGroup(HttpServletRequest request,
                          PEActionForm form,
                          AdminGroups adgrps,
                          String groupName) throws Throwable {
    if (groupName == null) {
      // We require a name
      return "chooseGroup";
    }

    AdminGroupVO ag = adgrps.findGroup(groupName);

    if (debug) {
      if (ag == null) {
        logIt("No user admin group with name " + groupName);
      } else {
        logIt("Retrieved user admin group " + ag.getName());
      }
    }

    form.assignAdminGroup(ag);

    String s = setAdminUser(request, form, ag.getOwnerid(),
                            isMember(ag, form));

    if (s != null) {
      return s;
    }

    form.setAdminUserId(form.getCalSvcI().getUserVO().getName());

    return null;
  }

  protected String setAdminUser(HttpServletRequest request,
                                PEActionForm form,
                                String user,
                                boolean isMember) {
    try {
      int access = getAccess(request, getMessages());

      if (!checkSvci(request, form, access, user, false, true,
                     isMember, debug)) {
        return "error";
      }

      return null;
    } catch (Throwable t) {
      err.emit(t);
      return "error";
    }
  }

  private boolean isMember(AdminGroupVO ag,
                           PEActionForm form) throws Throwable {
    String[] mbrs = ag.getMembers();
    String id = String.valueOf(form.getCurrentUser());

    if (mbrs == null) {
      return false;
    }

    for (int i = 0; i < mbrs.length; i++) {
      if (mbrs[i].equals(id)) {
        return true;
      }
    }

    return false;
  }
}

