/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.sss.util.servlets;

import java.util.Locale;

/** This is what affects the state of ConfiguredXSLTFilter.
 *
 * @author Mike Douglass douglm@rpi.edu
 * @version June 18th 2003
 */
public class XSLTFilterConfigInfo {
  /** appRoot does not get reset when the filter tries a default path.
   * This must be set to some valid value.
   */
  private String appRoot;

  /** Default values (for some)
   */

  /** Use this if we can't derive it from the system.
   */
  private static final String localeInfoDefaultDefault = "default";

  /** Try to obtain from Locale.getDefault()
   */
  private static final String localeInfoDefault;
  private static final Locale localeDefault = Locale.getDefault();
  private static final String langDefault;
  private static final String countryDefault;

  static {
    langDefault = localeDefault.getLanguage();
    countryDefault = localeDefault.getCountry();

    localeInfoDefault = makeLocale(langDefault, countryDefault);
  }

  private static final String browserTypeDefault = "default";
  private static final String skinNameDefault = "default";

  private String localeInfo;
  private String browserType;
  private String skinName;

  /** These don't take part in equality checks.
   */
  private boolean dontFilter;

  /** Should be set during call to obtainConfigInfo to force a reload
   * of the transformer
   */
  private boolean forceReload;

  private boolean reloadAlways;

  /** Will be set on call to updatedConfigInfo if the filter forced us
   * back to default locale.
   *
   * <p>The superclass should respond by preserving the values and
   * representing them on the next call to obtainConfigInfo.
   */
  private boolean forceDefaultLocale;

  /** Will be set on call to updatedConfigInfo if the filter forced us
   * back to default browser type.
   *
   * <p>The superclass should respond by preserving the values and
   * representing them on the next call to obtainConfigInfo.
   */
  private boolean forceDefaultBrowserType;

  /** Will be set on call to updatedConfigInfo if the filter forced us
   * back to default skin name.
   *
   * <p>The superclass should respond by preserving the values and
   * representing them on the next call to obtainConfigInfo.
   */
  private boolean forceDefaultSkinName;

  /** Content type may be set to force the content type to a certain value.
   * Normally it will be set by the transform.
   */
  private String contentType;

  /** Reset all defaults, used before setting current values.
   */
  public void reset() {
    localeInfo = localeInfoDefault;
    browserType = browserTypeDefault;
    skinName = skinNameDefault;
  }

  public void resetBrowserType() {
    browserType = browserTypeDefault;
  }

  public void resetSkinName() {
    skinName = skinNameDefault;
  }

  /* ====================================================================
             Properties methods
     ==================================================================== */

  public void setAppRoot(String val) {
    appRoot = val;
  }

  public String getAppRoot() {
    return appRoot;
  }

  public void setLocaleInfo(String val) {
    localeInfo = val;
  }

  public String getLocaleInfo() {
    if (getForceDefaultLocale()) {
      return localeInfoDefault;
    }
    return localeInfo;
  }

  public String getDefaultLang() {
    return langDefault;
  }

  public String getDefaultCountry() {
    return countryDefault;
  }

  public void setBrowserType(String val) {
    browserType = val;
  }

  public String getBrowserType() {
    if (getForceDefaultBrowserType()) {
      return browserTypeDefault;
    }
    return browserType;
  }

  public void setSkinName(String val) {
    skinName = val;
  }

  public String getSkinName() {
    if (getForceDefaultSkinName()) {
      return skinNameDefault;
    }
    return skinName;
  }

  public void setDontFilter(boolean val) {
    dontFilter = val;
  }

  public boolean getDontFilter() {
    return dontFilter;
  }

  public void setForceReload(boolean val) {
    forceReload = val;
  }

  public boolean getForceReload() {
    return forceReload;
  }

  public void setReloadAlways(boolean val) {
    reloadAlways = val;
  }

  public boolean getReloadAlways() {
    return reloadAlways;
  }

  public void setForceDefaultLocale(boolean val) {
    forceDefaultLocale = val;
  }

  public boolean getForceDefaultLocale() {
    return forceDefaultLocale;
  }

  public void setForceDefaultBrowserType(boolean val) {
    forceDefaultBrowserType = val;
  }

  public boolean getForceDefaultBrowserType() {
    return forceDefaultBrowserType;
  }

  public void setForceDefaultSkinName(boolean val) {
    forceDefaultSkinName = val;
  }

  public boolean getForceDefaultSkinName() {
    return forceDefaultSkinName;
  }

  public void setContentType(String val) {
    contentType = val;
  }

  public String getContentType() {
    return contentType;
  }

  public void updateFrom(XSLTFilterConfigInfo that) {
    appRoot = that.getAppRoot();
    localeInfo = that.getLocaleInfo();
    browserType = that.getBrowserType();
    skinName = that.getSkinName();
  }

  public int hashCode() {
    int i = String.valueOf(appRoot).hashCode() +
            String.valueOf(localeInfo).hashCode() +
            String.valueOf(browserType).hashCode() +
            String.valueOf(skinName).hashCode();

    return i;
  }

  /** If either the lang or country is null we provide a default value for
   * the whole locale. Otherwise we construct one.
   */
  public static String makeLocale(String lang, String country) {
    if ((lang == null) || (country == null)) {
      return localeInfoDefaultDefault;
    }

    return lang + "_" + country;
  }

  public boolean equals(Object o) {
    if (!(o instanceof XSLTFilterConfigInfo)) {
      return false;
    }

    if (o.hashCode() != this.hashCode()) {
      return false;
    }

    XSLTFilterConfigInfo that = (XSLTFilterConfigInfo)o;

    return isEqual(appRoot, that.appRoot) &&
           isEqual(localeInfo, that.localeInfo) &&
           isEqual(browserType, that.browserType) &&
           isEqual(skinName, that.skinName);
  }

  private boolean isEqual(String a, String b) {
    if (a == null) {
      return (b == null);
    }

    if (b == null) {
      return false;
    }

    return a.equals(b);
  }
}

