/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.sss.util.servlets;

import edu.rpi.sss.util.servlets.io.WrappedResponse;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.io.IOException;
import java.io.PrintWriter;
//import java.net.URL;
import java.util.Enumeration;
import java.util.Properties;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.xml.transform.SourceLocator;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

/** Class to implement a basic XSLT filter. The final configuration of this
 *  object can be carried out by overriding init.
 *  <p>Loosely based on some public example of filter code.</p>
 */
public class XSLTFilter extends AbstractFilter {
  private String xslt;
  private Transformer xmlTransformer;

  private TransformerFactory tf = TransformerFactory.newInstance();

  // For debugging
  private String reason = null;

  public void setXslt(String val) {
    xslt = val;
    // Force reload of the transformer
    xmlTransformer = null;
  }

  public String getXslt() {
    return xslt;
  }

  /** This method will only do something if there is no current XML transformer.
   *  A previous call to setXslt will discard any previous transformer.
   *  <p>Subclasses could call setXslt then call this method to check that the
   *  stylesheet is valid. A TransformerException provides inforamtion about
   *  where any error occuured.
   *
   * @return  Transformer  Existing or new XML transformer
   */
  public Transformer getXmlTransformer()
      throws TransformerException, ServletException, FileNotFoundException {
    if (xmlTransformer != null) {
      return xmlTransformer;
    }

    if (getXslt() == null) {
      return null;
    }

    // Time to get a transformer
//    URL u = null;
    InputStream is = null;

    try {
//      u = new URL(getXslt());

      xmlTransformer = tf.newTransformer(new StreamSource(getXslt()));
    } catch (TransformerConfigurationException tce) {
      /** Work our way down the chain to see if we have an embedded file
       * not found. If so, throw that to let the caller try another path.
       */
      Throwable cause = tce.getCause();
      while (cause instanceof TransformerException) {
        cause = ((TransformerException)cause).getCause();
      }

      if (!(cause instanceof FileNotFoundException)) {
        throw tce;
      }

      throw (FileNotFoundException)cause;
    } catch (TransformerException te) {
      throw te;
    } catch (Exception e) {
      e.printStackTrace();
      ctx.log("Could not initialize transform", e);
      throw new ServletException("Could not initialize transform", e);
    } finally {
  /*    if (is != null) {
        try {
          is.close();
        } catch (Exception fe) {}
      } */
    }

    return xmlTransformer;
  }

  /** This will attempt to initialise from the filter configuration.
   *  This may not provide all we need. The actual style sheet for example, may
   *  be set by the doPreFilter method
   *
   * @param filterConfig  The filter configuration
   */
  public void init(FilterConfig filterConfig) throws ServletException {
    super.init(filterConfig);

    xslt = filterConfig.getInitParameter("xslt");

    if (debug) {
      ctx.log("Filter " + filterConfig.getFilterName() +
                                        " using xslt " + xslt);
    }
  }

  public void doFilter(ServletRequest req,
                       ServletResponse response,
                       FilterChain filterChain)
         throws IOException, ServletException {
    HttpServletRequest hreq = (HttpServletRequest)req;
    final HttpServletResponse resp = (HttpServletResponse)response;

    long startTime = System.currentTimeMillis();

    reason = null;

    if (debug) {
      ctx.log("XSLTFilter: Accessing filter for " +
              HttpServletUtils.getReqLine(hreq) + " " +
              hreq.getMethod());
    }

    final boolean[] xformNeeded = new boolean[1];
    ByteArrayOutputStream baos = new ByteArrayOutputStream();

    WrappedResponse wrappedResp = new WrappedResponse(resp, ctx, debug) {
      public void setContentType(String type) {
        if (debug) {
          getLogger().debug("XSLTFilter: setContentType(" + type + ")");
        }
        if ((type.startsWith("text/xml")) ||
            (type.startsWith("application/xml"))) {
          if (debug) {
            ctx.log("XSLTFilter: Converting xml to html");
          }
          xformNeeded[0] = true;
        } else {
          super.setContentType(type);
          if (debug) {
            reason = "Content-type = " + type;
          }
        }
      }
    };

    synchronized (this) {
      filterChain.doFilter(req, wrappedResp);
    }

    /* We don't get a session till we've been through to the servlet.
     */
    HttpSession sess = hreq.getSession(false);
    String sessId;
    if (sess == null) {
      sessId = "NONE";
    } else {
      sessId = sess.getId();
    }

    logTime("PRETRANSFORM", sessId,
            System.currentTimeMillis() - startTime);

    /** Ensure we're all set up to handle content
     */
    doPreFilter(hreq);

    byte[] bytes = wrappedResp.toByteArray();
    if (bytes == null || (bytes.length == 0)) {
      if (debug) {
        ctx.log("No content!");
      }

      xformNeeded[0] = false;
    }

    try {
      if ((!dontFilter) && (xformNeeded[0])) {
        if (debug) {
          getLogger().debug("+*+*+*+*+*+*+*+*+*+*+* about to transform");
        }

        TransformerException te = null;
        Transformer xmlt = null;

        try {
          xmlt = getXmlTransformer();
        } catch (TransformerException te1) {
          te = te1;
        }

        if (te != null) {
          /** We had an exception getting the transformer.
              Output error information instead of the transformed output
           */

          outputInitErrorInfo(te, baos);
          contentType = "text/html";
        } else {
          /** We seem to be getting invalid bytes occassionally
           */
          for (int i = 0; i < bytes.length; i++) {
            if ((bytes[i] & 0x0ff) > 128) {
              getLogger().warn("Found byte > 128 at " + i +
                             " bytes = " + (bytes[i] & 0x0ff));
              bytes[i] = (int)('?');
            }
          }
          /** Managed to get a transformer. Do the thing.
              Note: This can be _very_ inefficient for large transforms
              Such transforms should be pre-calculated.
            */
          try {
            xmlt.transform(
                new StreamSource(new ByteArrayInputStream(bytes)),
                          new StreamResult(baos));
          } catch (TransformerException e) {
            outputTransformErrorInfo(e, baos);
            contentType = "text/html";
          }

          if (debug) {
            Properties pr = xmlt.getOutputProperties();
            if (pr != null) {
              Enumeration enum = pr.propertyNames();
              while (enum.hasMoreElements()) {
                String key = (String)enum.nextElement();
                getLogger().debug("--------- xslt-output property " +
                                key + "=" + pr.getProperty(key));
              }
            }
          }
        }

        if (contentType != null) {
          /** Set explicitly by caller.
           */
          resp.setContentType(contentType);
        } else {
          /** The encoding and media type should be available from the
           *  Transformer. Letting the stylesheet dictate the media-type
           *  is the right thing to do as only the stylesheet knows what
           *  it's producing.
           */
          Properties pr = xmlt.getOutputProperties();
          if (pr != null) {
            String encoding = pr.getProperty("encoding");
            String mtype = pr.getProperty("media-type");

            if (mtype != null) {
              if (encoding != null) {
                resp.setContentType(mtype + ";charset=" + encoding);
              } else {
                resp.setContentType(mtype);
              }
            }
          }
        }

        byte[] xformBytes = baos.toByteArray();

        resp.setContentLength(xformBytes.length);
        resp.getOutputStream().write(xformBytes);

        if (debug) {
          ctx.log("XML -> HTML conversion completed");
        }
      } else {
        if (debug) {
          if (dontFilter) {
            reason = "dontFilter";
          }

          if (reason == null) {
            reason = "Unknown";
          }

          getLogger().debug("+*+*+*+*+*+*+*+*+*+*+* transform suppressed" +
                          " reason = " + reason);
        }
        resp.setContentLength(bytes.length);
        resp.getOutputStream().write(bytes);
        if (contentType != null) {
          /** Set explicitly by caller.
           */
          resp.setContentType(contentType);
        }
      }
    } catch (Exception e) {
      throw new ServletException("Unable to transform document", e);
    } finally {
      if (wrappedResp != null) {
        wrappedResp.close();
      }
      if (baos != null) {
        try {
          baos.close();
        } catch (Exception bae) {}
      }
    }

    logTime("POSTTRANSFORM", sessId,
            System.currentTimeMillis() - startTime);
  }

  public void destroy() {
    super.destroy();
  }

  private void outputInitErrorInfo(TransformerException te, ByteArrayOutputStream wtr) {
    PrintWriter pw = new PrintWriter(wtr);

    outputErrorHtmlHead(pw, "XSLT initialization errors");
    pw.println("<body>");

    SourceLocator sl = te.getLocator();

    if (sl != null) {
      pw.println("<table>");
      outputErrorTr(pw, "Line", "" + sl.getLineNumber());
      outputErrorTr(pw, "Column", "" + sl.getColumnNumber());
      pw.println("</table>");
    }

    outputErrorException(pw, te.getCause());

    pw.println("</body>");
    pw.println("</html>");
  }

  private void outputTransformErrorInfo(Exception e,
                                        ByteArrayOutputStream wtr) {
    PrintWriter pw = new PrintWriter(wtr);

    outputErrorHtmlHead(pw, "XSLT transform error");
    pw.println("<body>");

    outputErrorPara(pw, "There was an error transforming content.");
    outputErrorPara(pw, "This is possibly due to incorrectly formatted " +
                        "content.");
    outputErrorPara(pw, "Following is a trace to help us locate the cause.");

    outputErrorException(pw, e);

    pw.println("</body>");
    pw.println("</html>");
  }

  private void outputErrorHtmlHead(PrintWriter pw, String head) {
    pw.println("<html>");
    pw.println("<head>");
    pw.println("<title>" + head + "</title>");
    pw.println("</head>");
  }

  private void outputErrorTr(PrintWriter pw, String s1, String s2) {
    pw.println("<tr>");
    pw.println("<td>" + s1 + "</td>");
    pw.println("<td>" + s2 + "</td>");
    pw.println("</tr>");
  }

  private void outputErrorPara(PrintWriter pw, String s) {
    pw.println("<p>");
    pw.println(s);
    pw.println("</p>");
  }

  private void outputErrorException(PrintWriter pw, Throwable e) {
    pw.println("<h2>Cause:</h2>");

    if (e == null) {
      pw.println("<br />********Unknown<br />");
    } else {
      pw.println("<pre>");
      e.printStackTrace(pw);
      pw.println("</pre>");
    }
  }

  private void logTime(String recId, String sessId, long timeVal) {
    StringBuffer sb = new StringBuffer(recId);

    sb.append(":");
    sb.append(sessId);
    sb.append(":");
    sb.append(timeVal);

    getLogger().info(sb.toString());
  }
}


