/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.sss.util.servlets;

import edu.rpi.sss.util.PropertyUtilException;

import java.beans.PropertyChangeSupport;
import java.util.Enumeration;
import javax.servlet.GenericServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.ServletConfig;
import java.util.Enumeration;
import org.apache.log4j.Logger;

/**
   A bunch of useful routines for servlets.
 */
public class ServletUtil {
  private transient PropertyChangeSupport propertyChangeSupport;
  private transient GenericServlet svlt;
  private transient ServletProperties pr;
  private boolean traceIt;
  private transient ServletConfig config;
  private Logger log;

  private String[] propertyNames;
  private String[] propertyDefaults;

  public ServletUtil(GenericServlet svlt, Logger log) {
    this.svlt = svlt;
    traceIt = (log != null) && (log.isDebugEnabled());
    propertyChangeSupport = new PropertyChangeSupport(svlt);
  }

  public ServletProperties setProperties(String[] propertyNames,
                                         String[] propertyDefaults) {
    this.propertyNames = propertyNames;
    this.propertyDefaults = propertyDefaults;

    try {
      pr = new ServletProperties(propertyNames, propertyDefaults, log);
    } catch (PropertyUtilException e) {
      e.printStackTrace(System.out);
      svlt.log("setProperties failed with exception " + e.getMessage());
      return null;
    }

    return pr;
  }

  public boolean setConfig(ServletConfig config) {
    try {
      pr.setConfig(config);
      if ((log != null) && (log.isDebugEnabled())) {
        dumpConfig(config, log);
      }
      return true;
    } catch (PropertyUtilException e) {
      svlt.log("setConfig failed with exception " + e.getMessage());
      return false;
    }
  }

  public static void dumpConfig(ServletConfig config, Logger log) {
    if (log == null) {
      return;
    }

    if (config == null) {
      log.debug("null config");
      return;
    }

    Enumeration names = config.getInitParameterNames();

    log.debug("Initial parameters:");

    while (names.hasMoreElements()) {
      String key = (String)names.nextElement();
      String val = config.getInitParameter(key);
      log.debug("  " + key + " = " + val);
    }
  }

  public String getStringProperty(int index) {
    if (pr != null) {
      // Get the current value
      return pr.prop(index);
    }

    // return the default if any
    if (propertyDefaults == null) return null;

    /** Should check index here!!!!!!!!!!! */
    return propertyDefaults[index];
  }

  public void setStringProperty(int index, String val) {
    Object oldValue = getStringProperty(index);

    if (pr != null) pr.propSet(index, val);

    if (propertyNames == null) return;
    if (propertyChangeSupport != null)
      propertyChangeSupport.firePropertyChange(
              propertyNames[index], oldValue, val);
  }

  public int getIntProperty(int index) {
    String val = getStringProperty(index);

    if (val == null) return 0;

    try {
      return Integer.parseInt(val, 10);
    } catch (Exception e) {
      return 0;
    }
  }

  public void setIntProperty(int index, int val) {
    Object oldValue = new Integer(getIntProperty(index));
    Integer newValue = new Integer(val);

    if (pr != null) pr.propSet(index, newValue.toString());

    if (propertyNames == null) return;
    if (propertyChangeSupport != null)
      propertyChangeSupport.firePropertyChange(
        propertyNames[index], oldValue, newValue);
  }

  public boolean getBoolProperty(int index) {
    String val = getStringProperty(index);

    if (val == null) return false;

    val = val.toLowerCase();
    return ((val.equals("true") || val.equals("yes")));
  }

  public void setBoolProperty(int index, boolean val) {
    Object oldValue = new Boolean(getBoolProperty(index));
    Boolean newValue = new Boolean(val);

    if (pr != null) pr.propSet(index, newValue.toString());

    if (propertyNames == null) return;
    if (propertyChangeSupport != null)
      propertyChangeSupport.firePropertyChange(
         propertyNames[index], oldValue, newValue);
  }

  public static void dumpRequestParameters(String title,
             HttpServletRequest req,
             Logger log) {
    if ((log == null) || (!log.isDebugEnabled())) {
      return;
    }

    Enumeration names = req.getParameterNames();

    log.debug(title + " - global info and uris");
    log.debug("getRequestURI = " + req.getRequestURI());
    log.debug("getRemoteUser = " + req.getRemoteUser());
    log.debug("getRequestedSessionId = " + req.getRequestedSessionId());
    log.debug("HttpUtils.getRequestURL(req) = " + req.getRequestURL());
    log.debug("query=" + req.getQueryString());
    log.debug("contentlen=" + req.getContentLength());
    log.debug("request=" + req);
    log.debug("parameters:");

    log.debug(title);

    while (names.hasMoreElements()) {
      String key = (String)names.nextElement();
      String val = req.getParameter(key);
      log.debug("  " + key + " = \"" + val + "\"");
    }
  }

}

