/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.sss.util.servlets;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpServletResponseWrapper;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import org.w3c.tidy.Tidy;

/** Class to implement a JTidy filter. The final configuration of this
 *  object can be carried out by overriding init and preFilter.
 *  <p>This filter will run the data through JTidy converting it to valid XML
 *
 * @author Mike Douglass
 */
public class JTidyFilter implements Filter {
  private ServletContext ctx;
  private boolean tidyIt = true;
  private boolean tidyXmlOut = true;
  private boolean tidyQuiet = false;
  private boolean debug = false;

  public void setTidyIt(boolean val) {
    tidyIt = val;
  }

  public boolean getTidyIt() {
    return tidyIt;
  }

  public void setTidyXmlOut(boolean val) {
    tidyXmlOut = val;
  }

  public boolean getTidyXmlOut() {
    return tidyXmlOut;
  }

  public void setDebug(boolean val) {
    debug = val;
  }

  public boolean getDebug() {
    return debug;
  }

  /** This will attempt to initialise from the filter configuration.
   *  This may not provide all we need. The actual style sheet for example, may
   *  be set by the doPreFilter method
   *
   * @param filterConfig  The filter configuration
   */
  public void init(FilterConfig filterConfig) throws ServletException {
    ctx = filterConfig.getServletContext();
    String temp = filterConfig.getInitParameter("debug");
    debug = (String.valueOf(temp).equals("true"));

    temp = filterConfig.getInitParameter("tidyit");
    tidyIt = (String.valueOf(temp).equals("true"));

    if (debug) {
      ctx.log("Filter " + filterConfig.getFilterName() +
                                        " tidyIt = " + tidyIt);
    }
  }

  /** This method can be overridden to allow a subclass to set up ready for a
   *  transformation.
   *
   * @param   request    Incoming HttpServletRequest object
   */
  public void doPreFilter(HttpServletRequest request)
    throws ServletException {
  }

  public void doFilter(ServletRequest req,
                       ServletResponse response,
                       FilterChain filterChain)
         throws IOException, ServletException {
    HttpServletRequest hreq = (HttpServletRequest)req;
    final HttpServletResponse resp = (HttpServletResponse)response;

    doPreFilter(hreq);

    if (debug) {
      ctx.log("Accessing filter for " +
              HttpServletUtils.getReqLine(hreq) + " " +
              hreq.getMethod());
    }

    final ByteArrayPrintWriter pw = new ByteArrayPrintWriter();
    final boolean[] xformNeeded = new boolean[1];

    HttpServletResponse wrappedResp = new HttpServletResponseWrapper(resp) {
      public PrintWriter getWriter() {
        return pw.getWriter();
      }

      public ServletOutputStream getOutputStream() {
        return pw.getStream();
      }

      public void flushBuffer() {
      }

      public void setContentType(String type) {
        // I guess we just set it to xml
        resp.setContentType("text/xml");
      }
    };

    if (!tidyIt) {
      filterChain.doFilter(req, resp);
      return;
    }

    filterChain.doFilter(req, wrappedResp);

    byte[] bytes = pw.toByteArray();
    if (debug) {
      if (bytes == null || (bytes.length == 0)) {
        ctx.log("JTidy filter: No content.");
      } else {
        ctx.log("JTidy filter: Content length " + bytes.length);
      }
    }

    ByteArrayInputStream in = new ByteArrayInputStream(bytes);
    ByteArrayOutputStream out = new ByteArrayOutputStream();

    StringWriter errStrw = new StringWriter();
    PrintWriter err = new PrintWriter(errStrw);

    try {
      Tidy tidy = new Tidy();
      tidy.setXmlOut(tidyXmlOut);
      tidy.setXmlTags(true);
      tidy.setSmartIndent(true);
      tidy.setQuiet(tidyQuiet);
      tidy.setInputStreamName("");

      tidy.setErrout(err);
      tidy.parse(in, out);

      byte[] outBytes = out.toByteArray();

      resp.setContentLength(outBytes.length);
      resp.getOutputStream().write(outBytes);
      if (debug) {
        ctx.log("Tidy conversion completed - wrote " + outBytes.length +
                " bytes");
      }

      if (debug) {
        ctx.log("JTidy messages:");
        System.out.println(errStrw.getBuffer().toString());
      }
    } catch (Exception e) {
      System.out.println(errStrw.getBuffer().toString());
      e.printStackTrace();
      throw new ServletException("Unable to tidy document", e);
    }
  }

  public void destroy() {
    if (debug) {
      ctx.log("Destroying filter...");
    }
  }

  private static class ByteArrayServletStream extends ServletOutputStream {
    ByteArrayOutputStream baos;

    ByteArrayServletStream(ByteArrayOutputStream baos) {
      this.baos = baos;
    }

    public void write(int param) throws IOException {
      baos.write(param);
    }
  }

  private static class ByteArrayPrintWriter {
    private ByteArrayOutputStream baos = new ByteArrayOutputStream();
    private PrintWriter pw = new PrintWriter(baos);
    private ServletOutputStream sos = new ByteArrayServletStream(baos);

    public PrintWriter getWriter() {
      return pw;
    }

    public ServletOutputStream getStream() {
      return sos;
    }

    byte[] toByteArray() {
      return baos.toByteArray();
    }
  }
}

