/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.sss.util.servlets;

import edu.rpi.sss.util.servlets.io.WrappedResponse;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.HashMap;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import org.apache.fop.apps.Driver;
import org.apache.fop.apps.FOPException;
import org.xml.sax.InputSource;

/** Class to implement a basic FOP filter. The final configuration of this
 *  object can be carried out by overriding init.
 *  <p>This filter takes an input stream - probably peviously filtered by
 *  an xslt filter and uses fop to turn it into some other media type.
 *  </p>
 *  <p>The incoming content-type determines how we behave. If it is<ul>
 *  <li><em>application/fop-pdf</em>We try to generate pdf from the input
 *  </li>
 *  </ul>
 *  Otherwise we just pass it through.
 */
public class FOPFilter extends AbstractFilter {
  private boolean dump = false;

  /** If we see this come in we can act on the input.
   */
  private static final String contentTypePrefix = "application/fo-";
  private static final int contentTypePrefixLength = contentTypePrefix.length();

  /** The following defines the suffix on the input content-type, the
   *  corresponding renderer and the output content-type.
   * <p>
   * If it's not here but the suffix has the form "RENDER_?" where "?" is
   * any string, we assume it's a valid renderer name and pass it on.</p>
   */
  private final static class RendererInfo {
    String rendererName;
    String rendererFOPName;
    String contentType;

    RendererInfo(String rendererName, String rendererFOPName, String contentType) {
      this.rendererName = rendererName;
      this.rendererFOPName = rendererFOPName;
      this.contentType = contentType;
    }
  }

  private final static RendererInfo[] renderersInfo = {
    new RendererInfo("pdf", "RENDER_PDF", "application/pdf"),
    new RendererInfo("mif", "RENDER_MIF", "application/mif"),
    new RendererInfo("xml", "RENDER_XML", "application/xml"),
    new RendererInfo("pcl", "RENDER_PCL", "application/pcl"),
    new RendererInfo("ps", "RENDER_PS", "text/ps"),
    new RendererInfo("txt", "RENDER_TXT", "text/text"),
    new RendererInfo("svg", "RENDER_SVG", "text/svg")
  };

  private static final String rendererNamePrefix = "RENDER_";
  private static final int rendererNamePrefixLength = rendererNamePrefix.length();

  private static HashMap validRenderers = new HashMap(renderersInfo.length);
  static {
    for (int i = 0; i < renderersInfo.length; i++) {
      validRenderers.put(renderersInfo[i].rendererName, renderersInfo[i]);
    }
  }

  /** This will attempt to initialise from the filter configuration.
   *  This may not provide all we need. The actual style sheet for example, may
   *  be set by the doPreFilter method
   *
   * @param filterConfig  The filter configuration
   */
  public void init(FilterConfig filterConfig) throws ServletException {
    super.init(filterConfig);

    String temp = filterConfig.getInitParameter("dump");
    dump = (String.valueOf(temp).equals("true"));
  }

  public void doFilter(ServletRequest req,
                       ServletResponse response,
                       FilterChain filterChain)
         throws IOException, ServletException {
    HttpServletRequest hreq = (HttpServletRequest)req;
    final HttpServletResponse resp = (HttpServletResponse)response;

    if (debug) {
      ctx.log("FOPFilter: Accessing filter for " +
              HttpServletUtils.getReqLine(hreq) + " " +
              hreq.getMethod());
    }

    final boolean[] xformNeeded = new boolean[1];
    ByteArrayOutputStream baos = new ByteArrayOutputStream();

    FOPWrappedResponse wrappedResp = new FOPWrappedResponse(resp, ctx, debug);

    filterChain.doFilter(req, wrappedResp);

    /** Ensure we're all set up to handle content
     */
    doPreFilter(hreq);

    byte[] bytes = wrappedResp.toByteArray();
    if (debug) {
      if (bytes == null || (bytes.length == 0)) {
        ctx.log("No content!");
      } else if (dump) {
        dumpIt(new String(bytes), getLogger());
      }
    }

    try {
      if ((!dontFilter) && (xformNeeded[0])) {
        if (debug) {
          ctx.log("FOPFilter: about to transform");
        }

        RendererInfo ri = getRendererInfo(wrappedResp.getOutMethod());
        Driver fopDriver = new Driver(
          new InputSource(new ByteArrayInputStream(bytes)),
          new ByteArrayOutputStream());

        try {
          fopDriver.setRenderer(ri.rendererFOPName);
          fopDriver.run();
        } catch (FOPException fe) {
          outputErrorInfo(fe, baos);
        }

        if (contentType != null) {
          /** Set explicitly by caller.
           */
          resp.setContentType(contentType);
        } else if (ri.contentType != null) {
          resp.setContentType(ri.contentType);
        }

        byte[] xformBytes = baos.toByteArray();

        resp.setContentLength(xformBytes.length);
        resp.getOutputStream().write(xformBytes);
        ctx.log("FOPFilter conversion completed");
      } else {
        if (debug) {
          getLogger().debug("FOPFilter transform suppressed");
        }
        resp.setContentLength(bytes.length);
        resp.getOutputStream().write(bytes);
      }
    } catch (Exception e) {
      throw new ServletException("FOPFilter: Unable to transform document", e);
    } finally {
      if (wrappedResp != null) {
        wrappedResp.close();
      }
      if (baos != null) {
        try {
          baos.close();
        } catch (Exception bae) {}
      }
    }
  }

  public void destroy() {
    super.destroy();
  }

  private static class FOPWrappedResponse extends WrappedResponse {
    private String outMethod;

    public FOPWrappedResponse(HttpServletResponse resp,
                              ServletContext ctx, boolean debug) {
      super(resp, ctx, debug);
    }

    public void setContentType(String type) {
      if (type.startsWith(contentTypePrefix)) {
        outMethod = type.substring(contentTypePrefixLength);

        if (debug) {
          ctx.log("FOPFilter: Converting fo to " + outMethod);
        }
      } else {
        super.setContentType(type);
      }
    }

    public String getOutMethod() {
      return outMethod;
    }
  }

  private RendererInfo getRendererInfo(String outMethod) throws Exception {
    if (outMethod != null) {
      RendererInfo ri = (RendererInfo)validRenderers.get(outMethod);

      if (ri != null) {
        return ri;
      }

      if (outMethod.startsWith(rendererNamePrefix)) {
        return new RendererInfo(null, outMethod.substring(rendererNamePrefixLength),
                                null);
      }
    }

    throw new Exception("FOPFilter: invalid output method: " + outMethod);
  }

  private void outputErrorInfo(FOPException fe, ByteArrayOutputStream wtr) {
    PrintWriter pw = new PrintWriter(wtr);

    pw.println("<html>");
    pw.println("<head>");
    pw.println("<title>FOP exception</title>");
    pw.println("</head>");
    pw.println("<body>");

    pw.println("<h2>Message:</h2>");
    pw.println("<p>");
    pw.println(fe.getMessage());
    pw.println("</p>");

    pw.println("<pre>");
    fe.printStackTrace(pw);
    pw.println("</pre>");

    pw.println("</body>");
    pw.println("</html>");
  }
}
