/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.sss.util.servlets;

//import java.io.FileNotFoundException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.util.Locale;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import javax.servlet.ServletException;

/** This filter configures the superclass according to certain dynamic or
 * application parameters. These include some or all of the following:
 * <ul>
 * <li><b>approot</b> Root path to the css styles etc. probably a url</li>
 * <li><b>locale info</b> Allows different language and country versions</li>
 * <li><b>browser type</b> Different styles for different browsers</li>
 * <li><b>skin-name</b> Allow user configurability</li>
 * </ul>
 * <p>The above could be achieved by building a compound name from the various
 * components or by building a path. We choose to build a path.
 *
 * <p>The locale info is two components from the current locale. These are a
 * valid ISO Language Code, the lower-case two-letter codes as defined by
 * ISO-639. These codes can be found at a number of sites, such as
 * http://www.ics.uci.edu/pub/ietf/http/related/iso639.txt
 * <p>
 * The second part is a valid ISO Country Code. These codes are the upper-case
 * two-letter codes as defined by ISO-3166. A full list of these codes can be
 * found at a number of sites, such as <br/>
 * http://www.chemie.fu-berlin.de/diverse/doc/ISO_3166.html
 * <p>
 * The result is a string of the form "en_US"
 *
 * <p>The browser type can be any string. We choose to use the browser types
 * generated by HttpServletUtils.getBrowserType which maps user agents on
 * to a more restricted set of types.
 *
 * <p>The skin name is also an arbitrary string.
 *
 * <p>The values for these fields will be set by the doPreFilter method
 * which can be overriden. This method will be called before each invocation
 * of the filter. The values may be set at each call, however, the filter
 * will not be reinitialised unless the path has changed
 *
 * <p>The obtainConfigInfo method can be overridden to supply this filter
 * with the required information. This will be the more normal way of
 * supplying that information.
 *
 * @author Mike Douglass douglm@rpi.edu
 * @version June 18th 2003
 */
public class ConfiguredXSLTFilter extends XSLTFilter {
  /** Overide this to set the value or turn off presentation support
   * by returning null or the value "NONE".
   */
  public String getPresentationAttrName() {
    return PresentationState.presentationAttrName;
  }

  private XSLTFilterConfigInfo cfg = new XSLTFilterConfigInfo();
  private XSLTFilterConfigInfo nextCfg = new XSLTFilterConfigInfo();
  private XSLTFilterConfigInfo defaultCfg = new XSLTFilterConfigInfo();

  /** The path we derive from the above
   */
//  private String xsltPath;
//  private boolean pathChanged;

  public ConfiguredXSLTFilter() {
    super();

    cfg.reset();
    nextCfg.reset();
    defaultCfg.reset();
  }

  /** This method can be overridden to allow a subclass to set up ready for a
   *  transformation.
   *
   * <p>The default action provided here is to locate the PresentatonState
   * object in the session and use that to configure the filter.
   *
   * @param   request    Incoming HttpServletRequest object
   * @param   info       XSLTFilterConfigInfo for the next invocation.
   * @return  XSLTFilterConfigInfo updated info or new object. Must be non-null.
   */
  public XSLTFilterConfigInfo obtainConfigInfo(HttpServletRequest request,
                                               XSLTFilterConfigInfo info,
                                               PresentationState ps)
                                     throws ServletException {
    if (ps == null) {
      ps = getPresentationState(request);
    }

    if (ps == null) {
      // Still can't do a thing
      return info;
    }

    info.setAppRoot(ps.getAppRoot());

    /** Transfer the state */
    if (ps.getNoXSLTSticky()) {
      info.setDontFilter(true);
    } else {
      info.setDontFilter(ps.getNoXSLT());
      ps.setNoXSLT(false);
    }

    /* ============== Don't filter ================= */

    if (info.getDontFilter()) {
      // I think that's enough
      return info;
    }

    info.reset();

    /* ============== Locale ================= */

    Locale l = request.getLocale();
    String lang = l.getLanguage();
    if ((lang == null) || (lang.length() == 0)) {
      lang = info.getDefaultLang();
    }

    String country = l.getCountry();
    if ((country == null) || (country.length() == 0)) {
      country = info.getDefaultCountry();
    }

    info.setLocaleInfo(XSLTFilterConfigInfo.makeLocale(lang, country));

    /* ============== Browser type ================= */

    String temp = ps.getBrowserType();
    if (temp != null) {
      info.setBrowserType(temp);
    }
    if (!ps.getBrowserTypeSticky()) {
      ps.setBrowserType(null);
    }

    /* ============== Skin name ================= */

    temp = ps.getSkinName();
    if (temp != null) {
      info.setSkinName(temp);
    }
    if (!ps.getSkinNameSticky()) {
      ps.setSkinName(null);
    }

    /* ============== Content type ================= */

//    temp = ps.getContentType();
//    if (temp != null) {
//      info.setContentType(temp);
//    }

    info.setContentType(ps.getContentType());
    if (!ps.getContentTypeSticky()) {
      ps.setContentType(null);
    }

    /* ============== Refresh ================= */

    info.setForceReload(ps.getForceXSLTRefresh());
    ps.setForceXSLTRefresh(false);
    info.setReloadAlways(ps.getForceXSLTRefreshAlways());

    return info;
  }

  /** This method can be overridden to allow a subclass to determine what the
   * final config was. It will be called after the filter has set the path
   * which may be the default.
   *
   * It will only be called if any of the config elements changed.
   *
   * @param   info       XSLTFilterConfigInfo for the next invocation.
   */
  public void updatedConfigInfo(XSLTFilterConfigInfo info) {
  }

  /** Obtain the presentation state from the session. Override if you want
   * different behaviour.
   */
  protected PresentationState getPresentationState(HttpServletRequest request) {
    String attrName = getPresentationAttrName();

    if ((attrName == null) ||
         (attrName.equals("NONE"))) {
      return null;
    }

    HttpSession sess = request.getSession(false);

    if (sess == null) {
      return null;
    }

    Object o = sess.getAttribute(attrName);
    if (o == null) {
      return null;
    }

    if (debug) {
      ((PresentationState)o).debugDump("ConfiguredXSLTFilter",
                                       getLogger());
    }

    return (PresentationState)o;
  }

  /** This method can be overridden to allow a subclass to set up ready for a
   *  transformation.
   *
   * @param   request    Incoming HttpServletRequest object
   */
  public void doPreFilter(HttpServletRequest request)
    throws ServletException {
    nextCfg = obtainConfigInfo(request, nextCfg, null);
    setDontFilter(nextCfg.getDontFilter());

    if (getDontFilter()) {
      return;
    }

    if (nextCfg.getAppRoot() == null) {
      /** Either it hasn't been set - so we can't transform or it's too
          early in the session (login etc)
       */
      if (debug) {
        getLogger().debug("No app root");
      }

      setDontFilter(true);
      return;
    }

    nextCfg.setForceDefaultLocale(false);
    nextCfg.setForceDefaultBrowserType(false);
    nextCfg.setForceDefaultSkinName(false);

    if (debug) {
      getLogger().debug("About to try with forceDefaultBrowserType=" +
            nextCfg.getForceDefaultBrowserType() +
            ", forceDefaultSkinName=" +
            nextCfg.getForceDefaultSkinName() +
            ", contentType=" +
            nextCfg.getContentType());
    }

    if (!xsltPathChanged(cfg, nextCfg)) {
      /* We're not searching for a path and it's not changed. We should be
         OK */
      if (getDebug()) {
        getLogger().debug("Path did not change from " + getXslt());
      }

      return;
    }

    /** Try to discover a valid path. We work our way down the path trying
     * first the current element then the default. There are 3 elements to
     * try, locale, browser type and skin name.
     */

    StringBuffer xsltPath = new StringBuffer(nextCfg.getAppRoot());

    /* ============== Locale ================= */

    if (!tryPath(xsltPath, nextCfg.getLocaleInfo())) {
      nextCfg.setForceDefaultLocale(true);

      if (!tryPath(xsltPath, nextCfg.getLocaleInfo())) {
        throw new ServletException("File not found: " + xsltPath);
      }
    }

    /* ============== Browser type ================= */

    if (!tryPath(xsltPath, nextCfg.getBrowserType())) {
      nextCfg.setForceDefaultBrowserType(true);

      if (!tryPath(xsltPath, nextCfg.getBrowserType())) {
        throw new ServletException("File not found: " + xsltPath);
      }
    }

    /* ============== Skin name ================= */

    if (!tryPath(xsltPath, nextCfg.getSkinName() + ".xsl")) {
      nextCfg.setForceDefaultSkinName(true);

      if (!tryPath(xsltPath, nextCfg.getSkinName() + ".xsl")) {
        throw new ServletException("File not found: " + xsltPath);
      }
    }

    setContentType(nextCfg.getContentType());

    setXslt(xsltPath.toString());

    try {
      /** To get a transformer and see if everything seems OK
       */
      getXmlTransformer();
      if (getDebug()) {
        getLogger().debug("Got Transformer OK");
      }

      /** Make any forced defaults stick
       * /

      if (nextCfg.getForceDefaultBrowserType()) {
        /** Switch to defaults and make them stay
         * /
        nextCfg.resetBrowserType();
      }

      if (nextCfg.getForceDefaultSkinName()) {
        /** Switch to defaults and make them stay
         * /
        nextCfg.resetSkinName();
      }
      */

      cfg.updateFrom(nextCfg);
      updatedConfigInfo(cfg);
    } catch (Throwable t) {
      throw new ServletException("Could not initialize transform", t);
    }
  }

  /** This method is called to see if there is any change in the components
   *  which make up the path. If there is none we don't need to (re)configure.
   *  Otherwise we need to point the filter at the new xslt file.
   *
   * <p>We compare the working copy, nextCfg, to cfg which holds the
   * state of the filter across calls and see if any path elements are
   * different.
   *
   * <p>The elements making up the path are:<ul>
   * <li>App root</li>
   * <li>locale</li>
   * <li>browser type</li>
   * <li>skin name</li>
   * </ul>
   *
   * <p>All, except the app root, may be defaulted if an explicit one does
   * not exist. Searching for a stylesheet then becomes an excercise in
   * trying various paths until a stylesheet is located.
   *
   * <p>
   *
   * @param  req       Incoming HttpServletRequest object
   * @param  cfg       XSLTFilterConfigInfo object holding state
   * @param  nextCfg   XSLTFilterConfigInfo working copy
   * @result boolean   true if the path changed.
   */
  private boolean xsltPathChanged(XSLTFilterConfigInfo cfg,
                                  XSLTFilterConfigInfo nextCfg) {
    if (fieldChanged(cfg.getLocaleInfo(), nextCfg.getLocaleInfo()) ||
        fieldChanged(cfg.getBrowserType(), nextCfg.getBrowserType()) ||
        fieldChanged(cfg.getSkinName(), nextCfg.getSkinName())) {
      return true;
    }

    /** The path didn't change but we might be trying to force a reload
        If so just pretend the path changed.
     */
    boolean returnVal = nextCfg.getForceReload() ||
                        nextCfg.getReloadAlways();
    nextCfg.setForceReload(false);

    return returnVal;
  }

  /** This method is called to see if there is any change in the value
   *
   * @param   curVal      Current field value.
   * @param   newVal      Possible new field value.
   * @return  boolean     true if it changed
   */
  private boolean fieldChanged(String curVal,
                               String newVal) {
    if (curVal == null) {
      return true;
    }

    return !curVal.equals(newVal);
  }

  /** Try a path and see if it exists. If so append the element
   *
   * @param   prefix    StringBuffer current path prefix
   * @param   el        String path element to append
   * @return  boolean   true if path is OK
   */
  private boolean tryPath(StringBuffer prefix, String el) {
    if (debug) {
      getLogger().debug("trypath: " + prefix + "/" + el);
    }

    try {
      URL u = new URL(prefix + "/" + el);

      URLConnection uc = u.openConnection();

      if (!(uc instanceof HttpURLConnection)) {
        return false;
      }

      HttpURLConnection huc = (HttpURLConnection)uc;

      if (huc.getResponseCode() != 200) {
        return false;
      }

      prefix.append("/");
      prefix.append(el);

      return true;
    } catch (Throwable t) {
      if (debug) {
        getLogger().debug("trypath exception: ", t);
      }
    }

    return false;
  }
}

