/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.sss.util.servlets;

import java.io.IOException;
import java.io.LineNumberReader;
import java.io.StringReader;
import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import org.apache.log4j.Logger;

/** An abstract filter class to help build filters.
 *  <p>We provide abstract methods for the three required filter methods,
 *  init, destroy and doFilter.
 *  We also provide an additional method, doPreFilter, which should be
 *  called by the doFilter method at entry.
 *  </p>
 *
 * @author  Mike Douglass douglm@rpi.edu
 */
public abstract class AbstractFilter implements Filter {
  protected ServletContext ctx;

  protected boolean debug = false;

  /** If non-null we might want to set from here
   */
  protected String contentType;

  /** Set true if we don't want filtering to take place. This can be
   *  used to get a dump of the stream.
   */
  protected boolean dontFilter = false;

  private transient Logger log;

  /** This will attempt to initialise from the filter configuration.
   *  This may not provide all we need. The actual style sheet for example, may
   *  be set by the doPreFilter method
   *
   * @param filterConfig  The filter configuration
   */
  public void init(FilterConfig filterConfig) throws ServletException {
    ctx = filterConfig.getServletContext();
    String temp = filterConfig.getInitParameter("debug");

    try {
      int debugVal = Integer.parseInt(temp);

      debug = (debugVal > 2);
    } catch (Exception e) {}
  }

  public abstract void doFilter(ServletRequest req,
                                ServletResponse response,
                                FilterChain filterChain)
         throws IOException, ServletException;

  public void destroy() {
    if ((debug) && (ctx != null)) {
      ctx.log("Destroying filter...");
    }
  }

  /** This method can be overridden to allow a subclass to set up ready for a
   *  transformation.
   *
   * @param   request    Incoming HttpServletRequest object
   */
  public void doPreFilter(HttpServletRequest request)
    throws ServletException {
  }

  public void setDebug(boolean val) {
    debug = val;
  }

  public boolean getDebug() {
    return debug;
  }

  public void setContentType(String val) {
    contentType = val;
  }

  public String getContentType() {
    return contentType;
  }

  public void setDontFilter(boolean val) {
    dontFilter = val;
  }

  public boolean getDontFilter() {
    return dontFilter;
  }

  /** Get a logger for messages
   */
  public Logger getLogger() {
    if (log == null) {
      log = Logger.getLogger(this.getClass());
    }

    return log;
  }

  /** A debugging dump routine
   */
  public static void dumpIt(String val, Logger log) {
    StringReader dsr = new StringReader(val);
    LineNumberReader dlnr = new LineNumberReader(dsr);
    int i = 1;

    try {
      while (dlnr.ready()) {
        String s = dlnr.readLine();

        if (s == null) {
          break;
        }

        log.debug(i + ": " + s);
        i++;
      }
    } catch (Exception e) {
      log.debug(e.getMessage());
    }
  }
}
