/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.sss.util.process;

/** This class implements read/write locking.
 *
 * <p>There may be any number of readers of the object but only one writer.
 */
public class LockObject {
  /** Somewhere to place an associated object we may be protecting
   */
  Object thing;

  /** Identifying string
   */
  private String id;

  /* Number of readers */
  private int readLocks;

  /* True if a thread has the write lock */
  private boolean lockedWrite;

  /* Number of threads waiting to lock for write. */
  private int waitingWrite;

  private boolean debug = false;

  public LockObject(String id) {
    this(id, null, false);
  }

  public LockObject(String id, boolean debug) {
    this(id, null, debug);
  }

  public LockObject(String id, Object thing) {
    this(id, thing, false);
  }

  public LockObject(String id, Object thing, boolean debug) {
    this.id = id;
    this.thing = thing;
    this.debug = debug;
  }

  public void setDebug(boolean val) {
    debug = val;
  }

  public boolean getDebug() {
    return debug;
  }

  public void setThing(Object val) {
    thing = val;
  }

  public Object getThing() {
    return thing;
  }

  /** Called to lock the object for read. Return false if cannot lock.
   *
   * @return  boolean  true if lock request succesful
   */
  public boolean lockRead() {
    return lockRead(false);
  }

  /** Called to lock the object for read. Waits if object locked.
   */
  public void lockReadWait() throws Exception {
    lockRead(true);
  }

  /** Called to lock the object for read.
   *
   * @param   waitFlag if true wait for lock
   * @return  boolean  true if lock request succesful
   */
  public synchronized boolean lockRead(boolean waitFlag) {
    try {
      while((lockedWrite) || (waitingWrite != 0)) {
        if (!waitFlag) {
          return false;
        }

        if (debug) {
          System.out.println("LockObject(" + id + ") " +
                             Thread.currentThread() +
                             "waiting lock read");
        }
        wait();
      }
    } catch(java.lang.InterruptedException e) {
      if (debug) {
        System.out.println("LockObject(" + id + "): Wait interrupted ");
        System.out.println(e);
      }
    }

    readLocks++;

    if (debug) {
      System.out.println("LockObject(" + id + ") " +
                         Thread.currentThread() +
                         "got lock read");
    }

    return true;
  }

  /** Called to lock the object for write
   *
   * @return  boolean  True if locked
   */
  public boolean lockWrite() throws Exception {
    return lockWrite(false);
  }

  /** Called to lock the object for write. Waits if object locked.
   */
  public void lockWriteWait() throws Exception {
    lockWrite(true);
  }

  /** Called to lock the object for write.
   *
   * @param   waitFlag if true wait for lock
   * @return  boolean  true if lock request succesful
   */
  public synchronized boolean lockWrite(boolean waitFlag) {
    if ((readLocks != 0) || (lockedWrite)) {
      if (debug) {
        dumpOut("locked already - not waiting for lock write");
      }
      return false;
    }

    waitingWrite++;
    try {
      while(readLocks != 0) {
        if (debug) {
          dumpOut("waiting lock write");
        }
        wait();
      }
    } catch(java.lang.InterruptedException e) {
      if (debug) {
        dumpOut("Wait interrupted ");
        System.out.println(e);
      }
    }

    waitingWrite--;
    lockedWrite = true;

    if (debug) {
      dumpOut("got lock write");
    }

    return true;
  }

  /** Called to unlock an object
   *
   * @return  boolean  True if it was locked
   */
  public synchronized boolean unlock() {
    if (lockedWrite) {
      lockedWrite = false;
      return true;
    }

    if (readLocks != 0) {
      readLocks--;
      return true;
    }

    notifyAll();
    return false;
  }

  /** Called to check if the object is locked
   *
   * @return  int  -1 locked write, 0 - not locked, >0 - number of read locks
   */
  public int checkLock() {
    if (lockedWrite) {
      return -1;
    }

    return readLocks;
  }

  /** ===================================================================
                            Private
      =================================================================== */

  private void dumpOut(String msg) {
    System.out.println("!--LockObject(" + id + ") " +
                         Thread.currentThread() + msg);
  }
}
