/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.sss.util.process;

import java.util.Collection;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Vector;

/** This class implements the lock class.
 */
public class LockImpl implements Lock {
  /** Soemthing to identify this object
   */
  String id;

  /** This table contains objects with active locks. We should flush it out
   *  every so often.
   */
  private Hashtable lockObjects = new Hashtable();

  private int idle = 0;

  /** This table will contain owner objects. These contain a vector of
   *  references to locked entries.
   */
  private Hashtable owners = new Hashtable();

  /** Something to save recreating OnerInfo objects
   */
  private Vector inactiveOwners = new Vector();

  private boolean debug;

  private static class OwnerInfo {
    Object owner;
    boolean active;
    Vector reads = new Vector();
    Vector writes = new Vector();
  }

  public LockImpl(String id) {
    this(id, false);
  }

  public LockImpl(String id, boolean debug) {
    this.id = id;
    this.debug = debug;
  }

  public void setDebug(boolean val) {
    debug = val;
  }

  /** Called to lock an object for read
   *
   * @param   key      key of the required object
   * @return  boolean  True if locked
   */
  public boolean lockRead(Object key, Object owner) throws Exception {
    return lockRead(true, key, owner);
  }

  /** Called to lock an object for read. Waits if object locked.
   *
   * @param   key      key of the required object
   * @param   owner    object representing owner
   */
  public void lockReadWait(Object key, Object owner) throws Exception {
    lockRead(true, key, owner);
  }

  /** Called to lock the object for read.
   *
   * @param   waitFlag if true wait for lock
   * @param   key      key of the required object
   * @param   owner    object representing owner
   * @return  boolean  true if lock request succesful
   */
  public synchronized boolean lockRead(boolean waitFlag,
                                       Object key, Object owner)
                                       throws Exception {
    OwnerInfo oi = getOwner(owner);
    LockObject lo = getLockObject(key);

    if ((oi.reads.indexOf(lo) >= 0) ||
       (oi.writes.indexOf(lo) >= 0)) {
      throw new Exception("Object " + lo + " already locked by " + owner);
    }

    if (!lo.lockRead(waitFlag)) {
      return false;
    }

    oi.reads.addElement(lo);
    return true;
  }

  /** Called to lock an object for write
   *
   * @param   key      key of the required object
   * @param   owner    object representing owner
   * @return  boolean  True if locked
   */
  public boolean lockWrite(Object key, Object owner) throws Exception {
    return lockWrite(true, key, owner);
  }

  /** Called to lock an object for write. Waits if object locked.
   *
   * @param   key      key of the required object
   * @param   owner    object representing owner
   */
  public void lockWriteWait(Object key, Object owner) throws Exception {
    lockWrite(true, key, owner);
  }

  /** Called to lock the object for write.
   *
   * @param   waitFlag if true wait for lock
   * @param   key      key of the required object
   * @param   owner    object representing owner
   * @return  boolean  true if lock request succesful
   */
  public synchronized boolean lockWrite(boolean waitFlag,
                                        Object key, Object owner)
                                        throws Exception {
    OwnerInfo oi = getOwner(owner);
    LockObject lo = getLockObject(key);

    if ((oi.reads.indexOf(lo) >= 0) ||
       (oi.writes.indexOf(lo) >= 0)) {
      throw new Exception("Object " + lo + " already locked by " + owner);
    }

    if (!lo.lockWrite(waitFlag)) {
      if (debug) {
        dumpOut("return false from LockWrite(" + waitFlag + ", " + key +
                ")");
      }
      return false;
    }

    oi.writes.addElement(lo);
    return true;
  }

  /** Called to unlock an record
   *
   * @param   key      key of the required object
   * @return  boolean  True if it was locked
   */
  public synchronized boolean unlock(Object key, Object owner) throws Exception {
    OwnerInfo oi = getOwner(owner);
    LockObject lo = getLockObject(key);

    if ((oi.reads.indexOf(lo) < 0) &&
       (oi.writes.indexOf(lo) < 0)) {
      return false;
    }

    if (!lo.unlock()) {
      return false;
    }

    oi.reads.removeElement(lo);
    oi.writes.removeElement(lo);

    if ((oi.reads.size() == 0) && (oi.writes.size() == 0)) {
      removeOwnerInfo(oi);
    }

    return true;
  }

  /** Called to check an object is locked
   *
   * @param   key      key of the required object
   * @param   owner    object representing owner - null for any owner
   * @return  boolean  True if locked
   */
  public synchronized boolean checkLock(Object key, Object owner) throws Exception {
    LockObject lo = getLockObject(key);

    if (owner != null) {
      OwnerInfo oi = getOwner(owner);
      if ((oi.reads.indexOf(lo) < 0) &&
         (oi.writes.indexOf(lo) < 0)) {
        return false;
      }

      return true;
    }

    return (lo.checkLock() != 0);
  }

  /** ===================================================================
                            Owners
      =================================================================== */

  /** Return a list of all locks this owner.
   */
  public synchronized Lock.LockEntry[] getLocks(Object owner) {
    OwnerInfo oi = getOwner(owner);

    if (oi == null) {
      return null;
    }

    int sz = oi.reads.size() + oi.writes.size();

    Lock.LockEntry[] les = new Lock.LockEntry[sz];
    int lesi = 0;

    for (int i = 0; i < oi.reads.size(); i++) {
      LockObject lo = (LockObject)oi.reads.elementAt(i);

      Lock.LockEntry le = new Lock.LockEntry();
      le.key = lo.getThing();
      le.owner = new Object[1];
      le.owner[0] = owner;
      le.readLock = true;

      les[lesi] = le;
      lesi++;
    }

    for (int i = 0; i < oi.writes.size(); i++) {
      LockObject lo = (LockObject)oi.writes.elementAt(i);

      Lock.LockEntry le = new Lock.LockEntry();
      le.key = lo.getThing();
      le.owner = new Object[1];
      le.owner[0] = owner;
      le.readLock = false;

      les[lesi] = le;
      lesi++;
    }

    return les;
  }

  /** Unlock all locks for owner.
   */
  public synchronized void releaseLocks(Object owner) {
    OwnerInfo oi = getOwner(owner);

    if (oi == null) {
      if (debug) {
        dumpOut("releaseLocks(" + owner + ") found nothing");
      }
      return;
    }

    if (debug) {
      dumpOut("releaseLocks(" + owner + ") found owner");
    }

    while (oi.reads.size() > 0) {
      if (debug) {
        dumpOut("releaseLocks(" + owner + ") unlock read");
      }
      LockObject lo = (LockObject)oi.reads.lastElement();
      lo.unlock();
      oi.reads.removeElementAt(oi.reads.size() - 1);
    }

    while (oi.writes.size() > 0) {
      if (debug) {
        dumpOut("releaseLocks(" + owner + ") unlock write");
      }
      LockObject lo = (LockObject)oi.writes.lastElement();
      lo.unlock();
      oi.writes.removeElementAt(oi.writes.size() - 1);
    }

    removeOwnerInfo(oi);
  }

  /** Return a list of all owners we know about.
   */
  public synchronized Object[] getOwners() {
    Enumeration enum = owners.keys();

    Object[] os = new Object[owners.size()];
    for (int i = 0; i < os.length; i++) {
      os[i] = enum.nextElement();
    }

    return os;
  }

  /** ===================================================================
                            Misc
      =================================================================== */

  /** Return a list of all locks.
   */
  public Lock.LockEntry[] getLocks() {
    Collection c = lockObjects.values();
    Vector v = new Vector();
    Iterator it = c.iterator();

    while (it.hasNext()){
      LockObject lo = (LockObject)it.next();

      int cl = lo.checkLock();

      if (cl != 0) {
        Lock.LockEntry le = new Lock.LockEntry();

        le.key = lo.getThing();
        if (cl < 0) {
          le.readLock = (cl > 0);
          le.owner = findOwners(lo);

          v.addElement(le);
        }
      }
    }

    return (Lock.LockEntry[])v.toArray(new Lock.LockEntry[v.size()]);
  }

  /** ===================================================================
                            Misc
      =================================================================== */

  public Object[] findOwners(LockObject lo) {
    boolean read = (lo.checkLock() >= 0);
    Object key = lo.getThing();
    Vector v = new Vector();

    Collection c = owners.values();
    Iterator it = c.iterator();

    while (it.hasNext()){
      OwnerInfo oi = (OwnerInfo)it.next();

      if ((read) && (oi.reads.indexOf(key) >= 0)) {
        v.addElement(oi.owner);
      }

      if ((!read) && (oi.writes.indexOf(key) >= 0)) {
        v.addElement(oi.owner);
      }
    }

    return v.toArray();
  }

  /** Return an OwnerInfo object
   */
  public OwnerInfo getOwner(Object owner) {
    Object o = null;
    synchronized (owners) {
      o = owners.get(owner);

      if (o != null) {
        if (debug) {
          dumpOut("getOwner(" + owner + ") found something");
        }
        return (OwnerInfo)o;
      }

      /** See if we can locate an inactive one.
       */
      OwnerInfo oi = null;

      if (inactiveOwners.size() > 0) {
        oi = (OwnerInfo)inactiveOwners.lastElement();
        inactiveOwners.removeElementAt(inactiveOwners.size() - 1);
        oi.owner = owner;
        owners.put(owner, oi);
        if (debug) {
          dumpOut("getOwner(" + owner + ") found inactive");
        }
        return oi;
      }

      /** Create a new one
       */
      oi = new OwnerInfo();
      oi.owner = owner;
      owners.put(owner, oi);
      if (debug) {
        dumpOut("getOwner(" + owner + ") created new");
      }

      return oi;
    }
  }

  /** Remove an OwnerInfo object
   */
  public void removeOwnerInfo(OwnerInfo oi) {
    synchronized (owners) {
      owners.remove(oi.owner);
      oi.owner = null;
      inactiveOwners.addElement(oi);
    }
  }

  /** Locate a lock object
   */
  public LockObject getLockObject(Object key) {
    Object o = lockObjects.get(key);

    if (o != null) {
      if (debug) {
        dumpOut("getLockObject(" + key + ") found something");
      }
      return (LockObject)o;
    }

    LockObject lo = new LockObject(key.toString(), key, debug);
    lockObjects.put(key, lo);
    if (debug) {
      dumpOut("getLockObject(" + key + ") created new object");
    }

    return lo;
  }

  /** ===================================================================
                            Private
      =================================================================== */

  private void dumpOut(String msg) {
    System.out.println("!--LockImpl[" + id + "]: " + msg);
  }
}
