/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.sss.util.process;

import java.io.Serializable;

/** This interface describes how we lock and unlock objects.
 * <p>We assume that locks are owned by somebody represented by an object.
 *
 * <p>Each object type we want to maintain locks on is represented by an
 * implementation of this interface.
 *
 * <p>Each object is indexed by an object which must be unique</p>
 *
 * <p>Each owner may have a number of locks.</p>
 * <p> Owner and key objects must implement
 * the hashCode and equals methods correctly.</p>
 */
public interface Lock extends Serializable {
  public static class LockEntry {
    public Object key;
    public Object[] owner;
    public boolean readLock;
  }

  public static class OwnerEntry {
    public Object owner;
    public int numLocks;
  }

  public void setDebug(boolean val);

  /** Called to lock an object for read
   *
   * @param   key      key of the required object
   * @return  boolean  True if locked
   */
  public boolean lockRead(Object key, Object owner) throws Exception;

  /** Called to lock an object for read. Waits if object locked.
   *
   * @param   key      key of the required object
   * @param   owner    object representing owner
   */
  public void lockReadWait(Object key, Object owner) throws Exception;

  /** Called to lock the object for read.
   *
   * @param   waitFlag if true wait for lock
   * @param   key      key of the required object
   * @param   owner    object representing owner
   * @return  boolean  true if lock request succesful
   */
  public boolean lockRead(boolean waitFlag,
                          Object key, Object owner)
                           throws Exception;

  /** Called to lock an object for write
   *
   * @param   key      key of the required object
   * @param   owner    object representing owner
   * @return  boolean  True if locked
   */
  public boolean lockWrite(Object key, Object owner) throws Exception;

  /** Called to lock an object for write. Waits if object locked.
   *
   * @param   key      key of the required object
   * @param   owner    object representing owner
   */
  public void lockWriteWait(Object key, Object owner) throws Exception;

  /** Called to lock the object for write.
   *
   * @param   waitFlag if true wait for lock
   * @param   key      key of the required object
   * @param   owner    object representing owner
   * @return  boolean  true if lock request succesful
   */
  public boolean lockWrite(boolean waitFlag,
                           Object key, Object owner)
                           throws Exception;

  /** Called to unlock an object
   *
   * @param   key      key of the required object
   * @param   owner    object representing owner
   * @return  boolean  True if it was locked
   */
  public boolean unlock(Object key, Object owner) throws Exception;

  /** Called to check if an object is locked
   *
   * @param   key      key of the required object
   * @param   owner    object representing owner - null for any owner
   * @return  boolean  True if locked
   */
  public boolean checkLock(Object key, Object owner) throws Exception;

  /** ===================================================================
                            Owners
      =================================================================== */

  /** Return a list of all locks this owner.
   *
   * @param   owner        object representing owner
   * @return   LockEntry[] objects representing locks
   */
  public LockEntry[] getLocks(Object owner);

  /** Unlock all locks for owner.
   *
   * @param   owner        object representing owner
   * @return   LockEntry[] objects representing locks
   */
  public void releaseLocks(Object owner);

  /** Return a list of all owners we know about.
   *
   * @return   Object[]    objects representing owners
   */
  public Object[] getOwners();

  /** ===================================================================
                            Misc
      =================================================================== */

  /** Return a list of all locks.
   */
  public LockEntry[] getLocks();
}
