/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.sss.util.jsp;

import edu.rpi.sss.util.fmt.TimeDateFormatter;
import edu.rpi.sss.util.log.MessageEmit;
import edu.rpi.sss.util.servlets.PresentationState;
import edu.rpi.sss.util.Util;

import java.io.File;
import java.util.Date;
import java.util.HashMap;
import java.util.Set;
import java.util.Vector;

import org.apache.log4j.Logger;
import org.apache.struts.action.ActionError;
import org.apache.struts.action.ActionErrors;
import org.apache.struts.action.ActionForm;
import org.apache.struts.action.ActionMapping;
import org.apache.struts.util.MessageResources;

/**
 * This class provides some convenience methods for use by ActionForm objects.
 *
 * @author Mike Douglass
 */
public class UtilActionForm extends ActionForm {
  /** Are we debugging?
   */
  protected boolean debug;

  /** Have we initialised?
   */
  protected boolean initialised;

  /** Is nocache on?
   */
  protected boolean nocache;

  /** Current presentation state
   */
  protected PresentationState ps;

  protected transient Logger log;

  /** An error object reinitialised at each entry to the abstract action
   */
  protected transient MessageEmit err;

  /** A message object reinitialised at each entry to the abstract action
   */
  protected transient MessageEmit msg;

  /** So we can get hold of properties
   */
  protected MessageResources mres;

  /** Application variables. These can be set with request parameters and
   * dumped into the page for use by jsp and xslt.
   */
  protected HashMap appVars;

  /** Incoming URL.
   */
  protected String url;

  /** First part of URL. Allows us to target services on same host.
   */
  protected String schemeHostPort;

  protected String actionPath;

  /** The part of the URL that identifies the application -
   * Of the form "/" + name-of-app, e.g. /kiosk
   */
  protected String context;

  /** scheme + host + port part of the url together with the context.
   */
  protected String urlPrefix;

  /**
   * The current authenticated user. May be null
   */
  protected String currentUser;

  /**
   * Session id -
   */
  protected String sessionId;

  /**
   * Confirmation id - require this on forms
   */
  protected String confirmationId;

  /**
   * General yes/no answer
   */
  protected String yesno;

  /**
   * Browser type
   */
  protected String browserType = "default";

  /** We accululate errors in this vector as the form is processed.
   * We use processErrors to emit actual messages
   */
  private Vector valErrors = new Vector();

  public static class ValError {
    public String fldName;
    public String badVal;

    public ValError(String fldName, String badVal) {
      this.fldName = fldName;
      this.badVal = badVal;
    }
  }

  public static class IntValError extends ValError {
    public IntValError(String fldName, String badVal) {
      super(fldName, badVal);
    }
  }

  /** ================ Properties methods ============== */

  public void setDebug(boolean val) {
    debug = val;
  }

  public boolean getDebug() {
    return debug;
  }

  public void setInitialised(boolean val) {
    initialised = val;
  }

  public boolean getInitialised() {
    return initialised;
  }

  public void setNocache(boolean val) {
    nocache = val;
  }

  public boolean getNocache() {
    return nocache;
  }

  public void setPresentationState(PresentationState val) {
    ps = val;
  }

  public PresentationState getPresentationState() {
    return ps;
  }

  /** Returns a base for use in urls within generated pages.
   * If no presentation state is present, or the app root is null we return
   * a zero length string - this will make the resulting url relative to the
   * application.
   *
   * <p>Otherwise we return the app root, appending a path separator if not
   * present.
   *
   * @return  String     base for urls.
   */
  public String getAppBase() {
    if ((ps == null) || (ps.getAppRoot() == null)) {
      return "";
    }

    String ar = ps.getAppRoot();

    if (!ar.endsWith(File.separator)) {
      return ar + File.separator;
    }

    return ar;
  }

  public MessageResources getMres() {
    return mres;
  }

  public void setMres(MessageResources val) {
    mres = val;
  }

  public void setLog(Logger val) {
    log = val;
  }

  public Logger getLog() {
    if (log == null) {
      log = Logger.getLogger(this.getClass());
    }

    return log;
  }

  public void setErr(MessageEmit val) {
    err = val;
  }

  public MessageEmit getErr() {
    return err;
  }

  public boolean getErrorsEmitted() {
    return err.messagesEmitted();
  }

  public void setMsg(MessageEmit val) {
    msg = val;
  }

  public MessageEmit getMsg() {
    return msg;
  }

  public boolean getMessagesEmitted() {
    return msg.messagesEmitted();
  }

  /** Can be called by a page to signal an exceptiuon
   */
  public void setException(Throwable t) {
    err.emit(t);
  }

  public void setAppVarsTbl(HashMap val) {
    appVars = val;
  }

  public Set getAppVars() {
    if (appVars == null) {
      return new HashMap().entrySet();
    }
    return appVars.entrySet();
  }

  /** Get a list of property values and return as a string array. The
   *  properties are stored with consectutively numbered names as in
   *  <pre>
   *     prop1=aval
   *     prop2=bval
   *  </pre>
   *  There can be no gaps in the sequence.
   *  setMres must have been called previously.
   *
   * @param prop       Property name
   * @return String[]  values as a String array
   */
  public String[] getVals(String prop) {
    return getVals(null, prop, null);
  }

  /** Get a list of property values and return as a string array. The
   *  properties are stored with consectutively numbered names as in
   *  <pre>
   *     prop1=aval
   *     prop2=bval
   *  </pre>
   *  There can be no gaps in the sequence.
   *  If pre or post are non-null they are the property names of values to
   *  be added to the beginning or end.
   *  setMres must have been called previously.
   *
   * @param pre        Property name of prefix
   * @param prop       Property name
   * @param post       Property name of postfix
   * @return String[]  values as a String array
   */
  public String[] getVals(String pre, String prop, String post) {
    Vector v = new Vector();

    if (pre != null) {
      // Add at the front.
      String s = mres.getMessage(pre);
      if (s != null) {
        v.addElement(s);
      }
    }

    int i = 1;

    for (;;) {
      String u = mres.getMessage(prop + i);
      if (u == null) {
        // No more
        break;
      }

      v.addElement(u);
      i++;
    }

    if (post != null) {
      // Add at the front.
      String s = mres.getMessage(post);
      if (s != null) {
        v.addElement(s);
      }
    }

    return (String[])v.toArray(new String[v.size()]);
  }

  public void setUrl(String val) {
    url = val;
  }

  public String getUrl() {
    return url;
  }

  public void setSchemeHostPort(String val) {
    schemeHostPort = val;
  }

  public String getSchemeHostPort() {
    return schemeHostPort;
  }

  /** Set the part of the URL that identifies the application.
   *
   * @param val       context path in form "/" + name-of-app, e.g. /kiosk
   */
  public void setContext(String val) {
    context = val;
  }

  public String getContext() {
    return context;
  }

  /** Sets the scheme + host + port part of the url together with the
   *  path up to the servlet path. This allows us to append a new action to
   *  the end.
   *  <p>For example, we want val="http://myhost.com:8080/myapp"
   *
   *  @param  val   the URL prefix
   */
  public void setUrlPrefix(String val) {
    urlPrefix = val;
  }

  /** Returns the scheme + host + port part of the url together with the
   *  path up to the servlet path. This allows us to append a new action to
   *  the end.
   *
   *  @return  String   the URL prefix
   */
  public String getUrlPrefix() {
    return urlPrefix;
  }

  /** Set the part of the URL that identifies the action.
   *
   * @param val       path in form "/" + action, e.g. /update.do
   */
  public void setActionPath(String val) {
    actionPath = val;
  }

  public String getActionPath() {
    return actionPath;
  }

  /** This should not be setCurrentUser as that exposes it to the incoming
   * request.
   *
   * @param val      String user id
   */
  public void assignCurrentUser(String val) {
    currentUser = val;
  }

  public String getCurrentUser() {
    return currentUser;
  }

  /** This should not be setSessionId as that exposes it to the incoming
   * request.
   *
   * @param val      String session id
   */
  public void assignSessionId(String val) {
    sessionId = val;
  }

  public String getSessionId() {
    return sessionId;
  }

  /** This should not be setConfirmationId as that exposes it to the incoming
   * request.
   *
   * @param val      String confirmation id
   */
  public void assignConfirmationId(String val) {
    confirmationId = val;
  }

  public String getConfirmationId() {
    if (confirmationId == null) {
      confirmationId = Util.makeRandomString(16, 35);
    }

    return confirmationId;
  }

  public void setYesno(String val) {
    yesno = val;
  }

  public String getYesno() {
    return yesno;
  }

  public boolean isYes() {
    return ((yesno != null) && (yesno.equalsIgnoreCase("yes")));
  }

  public void setBrowserType(String val) {
    browserType = val;
  }

  public String getBrowserType() {
    return browserType;
  }

  /** ----------------------------------------------------------------
   *      <center>Value conversion and error processing.</center>
   *  ---------------------------------------------------------------- */

  /** Convert a string parameter so we can add an
   * error message for incorrect formats (instead of relying on Struts).
   *
   * <p>Struts tends to return 0 or null for illegal values, e.g., alpha
   * characters for a number.
   */
  public int intVal(String newVal, int curVal, String name) {
    int newInt;

    try {
      newInt = Integer.parseInt(newVal);
    } catch (Exception e) {
      valErrors.addElement(new IntValError(name, newVal));
      newInt = curVal;
    }

    return newInt;
  }

  /** processErrors is called to determine if there were any errors.
   * If so processError is called for each error adn the errors vector
   * is cleared.
   * Override the processError method to emit custom messages.
   *
   * @param err      MessageEmit object
   * @return boolean True if there were errors
   */
  public boolean processErrors(MessageEmit err) {
    if (valErrors.size() == 0) {
      return false;
    }

    for (int i = 0; i < valErrors.size(); i++) {
      processError(err, (ValError)valErrors.elementAt(i));
    }

    valErrors.clear();
    return true;
  }

  /** Override this to emit messages
   */
  public void processError(MessageEmit err, ValError ve) {
  }

  /** Current time and date formatting
   */
  public String getCurTime() {
    return new TimeDateFormatter(TimeDateFormatter.time).format(new Date());
  }

  public String getCurDate() {
    return new TimeDateFormatter(TimeDateFormatter.date).format(new Date());
  }

  public String getCurDateTime() {
    return new TimeDateFormatter(TimeDateFormatter.timeDate).format(new Date());
  }

  public String getCurShortDate() {
    return new TimeDateFormatter(TimeDateFormatter.dateShort).format(new Date());
  }

  public String getCurShortDateTime() {
    return new TimeDateFormatter(TimeDateFormatter.dateTimeShort).format(new Date());
  }

  /** We get a lot of zero length strings in the web world. This will return
   * null for a zero length.
   *
   * @param  val    String request parameter value
   * @return String null for null or zero lengt val, val otherwise.
   */
  public static String checkNull(String val) {
    if (val == null) {
      return null;
    }

    if (val.length() == 0) {
      return null;
    }

    return val;
  }
}

