/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.sss.util.jsp;

import edu.rpi.sss.util.log.MessageEmit;

import org.apache.log4j.Logger;
import org.apache.struts.action.ActionError;
import org.apache.struts.action.ActionErrors;
import org.apache.struts.util.MessageResources;

import java.io.Serializable;
import java.util.ArrayList;

/** This class allows error message generation in the servlet world.
 *
 * @author Mike Douglass douglm@rpi.edu
 * @version 1.0
 */
public class ErrorEmitSvlt implements MessageEmit {
  protected boolean debug;

  transient protected String id;
  transient protected Object caller;
  transient protected MessageResources messages;
  transient protected ActionErrors errors;
  transient protected String exceptionPname;

  /** We save the message property and the parameters in the following
   * class which we can return as an alternative to the struts message
   * generation.
   */
  public class Msg implements Serializable {
    private ArrayList params = new ArrayList();

    private String msgId;

    public Msg(String msgId) {
      this.msgId = msgId;
    }

    public Msg(String msgId, Object o) {
      this.msgId = msgId;
      params.add(o);
    }

    public Msg(String msgId, Object o1, Object o2) {
      this.msgId = msgId;
      params.add(o1);
      params.add(o2);
    }

    public Msg(String msgId, Object o1, Object o2, Object o3) {
      this.msgId = msgId;
      params.add(o1);
      params.add(o2);
      params.add(o3);
    }

    public String getMsgId() {
      return msgId;
    }

    public ArrayList getParams() {
      return params;
    }
  }

  protected ArrayList msgList = new ArrayList();

  public ErrorEmitSvlt() {
    this(false);
  }

  public ErrorEmitSvlt(boolean debug) {
    this.debug = debug;
  }

  /** Generation of errors in the servlet world means adding them to the
   *  errors object. We need to call this routine on every entry to the
   *  application
   *
   * @param id       An identifying name
   * @param caller   Used for log4j identification
   * @param messages Resources
   * @param errors   Error message will be appended on failure.
   * @param exceptionPname Property name for exceptions
   */
  public void reinit(String id,
                     Object caller,
                     MessageResources messages,
                     ActionErrors errors,
                     String exceptionPname) {
    this.id = id;
    this.caller = caller;
    this.messages = messages;
    this.errors = errors;
    this.exceptionPname = exceptionPname;

    msgList.clear();
  }

  public ArrayList getMsgList() {
    return msgList;
  }

  public void emit(String pname) {
    if (debug) {
      debugMsg(pname, null, null);
    }

    msgList.add(new Msg(pname));

    if ((messages == null) || !haveOutputObject()) {
      return;
    }

    try {
      errors.add(id, new ActionError(pname));
    } catch (Throwable t) {
      logError(className() + ": exception adding Action message", t);
    }
  }

  /** Convenience routine
   */
  public void emit(String pname, int num){
    if (debug) {
      debugMsg(pname, "int", String.valueOf(num));
    }

    emit(pname, new Integer(num));
  }

  /** Set the property name of the default exception message
   */
  public void setExceptionPname(String pname) {
    exceptionPname = pname;
  }

  /** Convenience routine
   */
  public void emit(Throwable t) {
    if (debug) {
      debugMsg(exceptionPname, "Throwable", String.valueOf(t.getMessage()));
    }

    Logger.getLogger(caller.getClass()).error(t.getMessage(), t);

    emit(exceptionPname, t.getMessage());
  }

  public void emit(String pname, Object o){
    if (debug) {
      debugMsg(pname, "object", String.valueOf(o));
    }

    msgList.add(new Msg(pname, o));

    if ((messages == null) || !haveOutputObject()) {
      return;
    }

    try {
      errors.add(id, new ActionError(pname, o));
    } catch (Throwable t) {
      logError(className() + ": exception adding Action error", t);
    }
  }

  public void emit(String pname, Object o1, Object o2){
    if (debug) {
      debugMsg(pname, "2objects",
               String.valueOf(o1) + "; " +
               String.valueOf(o2));
    }

    msgList.add(new Msg(pname, o1, o2));

    if ((messages == null) || !haveOutputObject()) {
      return;
    }

    try {
      errors.add(id, new ActionError(pname, o1, o2));
    } catch (Throwable t) {
      logError(className() + ": exception adding Action error", t);
    }
  }

  public void emit(String pname, Object o1, Object o2, Object o3){
    if (debug) {
      debugMsg(pname, "2objects",
               String.valueOf(o1) + "; " +
               String.valueOf(o2) + "; " +
               String.valueOf(o3));
    }

    msgList.add(new Msg(pname, o1, o2, o3));

    if ((messages == null) || !haveOutputObject()) {
      return;
    }

    try {
      errors.add(id, new ActionError(pname, o1, o2, o3));
    } catch (Throwable t) {
      logError(className() + ":exception adding Action error" + pname, t);
    }
  }

  /** Indicate no messages emitted. Null in this implementation.
   */
  public void clear() {}

  /** Return true if any messages emitted
   */
  public boolean messagesEmitted() {
    return !errors.empty();
  }

  public ActionErrors getErrors() {
    return errors;
  }

  public void debugOut(String msg) {
    try {
      Logger log = Logger.getLogger(caller.getClass());

      if (log.isDebugEnabled()) {
        log.debug(msg);
      }
    } catch (Throwable t) {
      System.out.println(msg);
    }
  }

  protected boolean haveOutputObject() {
    return errors != null;
  }

  protected String className() {
    return "ErrorEmitSvlt";
  }

  protected void debugMsg(String pname, String ptype, String pval) {
    debugOut("Error emitted: property=" + pname +
             " ptype=" + ptype +
             " val(s)=" + pval);
  }

  protected void logError(String msg, Throwable t) {
    try {
      Logger log = Logger.getLogger(caller.getClass());

      log.error(msg, t);
    } catch (Throwable t1) {
      System.out.println(caller.getClass().getName() +
                         "Error: " + msg);
      t.printStackTrace();
    }
  }
}

