/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.sss.util.fmt;

import java.io.Serializable;
import java.sql.Timestamp;
import java.text.DateFormatSymbols;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Locale;

/** Bean to help in display and construction of date and time fields.
 *  Should be handed an array of valid month values.
 */
public class DateBean extends GregorianCalendar implements Serializable {
  private boolean debug = false;

  private String[] months;

  private String shortPattern;

  private String longPattern;

  private String[] defaultMonthLabels =
    {"Jan", "Feb", "Mar", "Apr", "May", "Jun",
     "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};

  /* Rather than do string conversion, we'll just pick out an indexed
     value from here */
  private static final String[] nums = {
    "0", "1", "2", "3", "4", "5", "6", "7", "8", "9",
    "10", "11", "12", "13", "14", "15", "16", "17", "18", "19",
    "20", "21", "22", "23", "24", "25", "26", "27", "28", "29",
    "30", "31", "32", "33", "34", "35", "36", "37", "38", "39",
    "40", "41", "42", "43", "44", "45", "46", "47", "48", "49",
    "50", "51", "52", "53", "54", "55", "56", "57", "58", "59",
    "60"};

  private static final String[] nums2 = {
    "00", "01", "02", "03", "04", "05", "06", "07", "08", "09",
    "10", "11", "12", "13", "14", "15", "16", "17", "18", "19",
    "20", "21", "22", "23", "24", "25", "26", "27", "28", "29",
    "30", "31", "32", "33", "34", "35", "36", "37", "38", "39",
    "40", "41", "42", "43", "44", "45", "46", "47", "48", "49",
    "50", "51", "52", "53", "54", "55", "56", "57", "58", "59",
    "60"};

  private SimpleDateFormat shortFormatter;
  private SimpleDateFormat longFormatter;
  private String stringDt;
  private Date dateFromStringDt;

  private int numberDaysInWeek;

  /** Properties settable from forms
   */
  private int year = -1;
  private int month = -1;
  private int day = -1;
  private int hours;
  private int minutes;
  private int seconds;

  public DateBean(String[] months) throws Exception {
    this(months, null, null);
  }

  /** Provide the localized months, the pattern for short format dates,
   * and the pattern for long format dates.
   *
   * @param  months       String[] localized months.
   * @param  shortPattern String short format - may be null.
   * @param  longPattern  String long format - may be null.
   */
  public DateBean(String[] months, String shortPattern,
                  String longPattern) {
    this.months = months;
    this.shortPattern = shortPattern;
    this.longPattern = longPattern;

    /** Get the number of days in a week. Is this ever anything other than 7?
     */
    numberDaysInWeek = getMaximum(Calendar.DAY_OF_WEEK) -
                       getMinimum(Calendar.DAY_OF_WEEK) + 1;
  }

  public DateBean(String[] months, Date dt) {
    this(months, dt, null, null);
  }

  public DateBean(String[] months, Date dt, String shortPattern,
                  String longPattern) {
    this(months, shortPattern, longPattern);
    setDate(dt);
  }

  /** Use this constructor if getLongDate and getShortDate will never be
   * called
   */
  public DateBean(Date dt) {
    this.months = defaultMonthLabels;

    /** Get the number of days in a week. Is this ever anything other than 7?
     */
    numberDaysInWeek = getMaximum(Calendar.DAY_OF_WEEK) -
                       getMinimum(Calendar.DAY_OF_WEEK) + 1;

    setDate(dt);
  }

  public void setDebug(boolean val) {
    debug = val;
  }

  public void setDate(Date dt) {
    setTime(dt);

    year = get(Calendar.YEAR);
    month = get(Calendar.MONTH);
    day = get(Calendar.DAY_OF_MONTH);
    hours = get(Calendar.HOUR_OF_DAY);
    minutes = get(Calendar.MINUTE);
    seconds = get(Calendar.SECOND);

    if (debug) {
      //System.out.println("****************DateBean: date = " + dt +
      //                   " month = " + month);
    }

    stringDt = null;
  }

  public String[] getMonths() {
    return months;
  }

  public void setYear(String val) {
    stringDt = null;

    year = makeInt(val, -9999, 9999);
  }

  public String getYear() {
    return makeStr(year);
  }

  public void setMonth(String val) {
    stringDt = null;

    if (val == null) {
      month = -1;
      return;
    }

    for (int i = 0; i < months.length; i++) {
      if (val.equalsIgnoreCase(months[i])) {
        month = i;
        if (debug) {
          try {
            System.out.println("******DateBean: set month: " + month +
                               " timestamp: " + getTimestamp());
          } catch (Throwable t) {
            System.out.println("******DateBean: set month: " + month +
                               " got exception " + t);
          }
        }
        return;
      }
    }

    month = -1;
  }

  public String getMonth() {
    if (months == null) {
      return defaultMonthLabels[month];
    }
    return months[month];
  }

  public int getMonthNum() {
    return month;
  }

  public String getMonthNum2() {
    return nums2[month];
  }

  public void setDay(String val) {
    stringDt = null;

    day = makeInt(val, 0, 31);
  }

  public String getDay() {
    return nums[day];
  }

  public String getDay2() {
    return nums2[day];
  }

  /** Get the day of the week for this object.
   *
   * @return int    day of the week
   */
  public int getDayOfWeek() {
    try {
      return get(Calendar.DAY_OF_WEEK) % numberDaysInWeek;
    } catch (Throwable t) {
      t.printStackTrace();
      return -1;
    }
  }

  public void setHour(String val) {
    stringDt = null;

    hours = makeInt(val, 0, 24);
  }

  public String getHour() {
    return nums[hours];
  }

  public String getHour2() {
    return nums2[hours];
  }

  public void setMinute(String val) {
    stringDt = null;

    minutes = makeInt(val, 0, 59);
  }

  public String getMinute() {
    return nums[minutes];
  }

  public String getMinute2() {
    return nums2[minutes];
  }

  public void setSecond(String val) {
    stringDt = null;

    seconds = makeInt(val, 0, 24);
  }

  public String getSecond() {
    return nums[seconds];
  }

  public String getSecond2() {
    return nums2[seconds];
  }

  public String getStringDate() throws Exception {
    if (stringDt != null) {
      return stringDt;
    }

    stringDt = "" + year + "-" + (month + 1) + "-" + day + " " +
               hours + ":" + minutes + ":" + seconds;

    if ((year == -1) || (month == -1) || (day == -1) ||
        (hours == -1) || (minutes == -1) || (seconds == -1)) {
      throw new Exception(getClass().getName() + ": missing data for getDate " +
                          stringDt);
    }

    if (debug) {
      System.out.println("****************DateBean: getStringDate dt = " +
                         stringDt);
    }

    dateFromStringDt =
        new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").parse(stringDt);
    return stringDt;
  }

  /** Return a date based upon the stored values
   *  Only expects year + month + day
   */
  public Date getDate() throws Throwable {
    return new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").parse(getStringDate());
  }

  public String getShortDate() throws Throwable {
    if (shortFormatter == null) {
      Locale l = Locale.getDefault();
      shortFormatter = new SimpleDateFormat(shortPattern,
                                            new DateFormatSymbols(l));
    }

    getStringDate(); // Ensure we have date
    return shortFormatter.format(dateFromStringDt);
  }

  public String getLongDate() throws Throwable {
    try {
    if (longFormatter == null) {
      Locale l = Locale.getDefault();
      longFormatter = new SimpleDateFormat(longPattern,
                                           new DateFormatSymbols(l));
    }

    getStringDate(); // Ensure we have date
    return longFormatter.format(dateFromStringDt);
    } catch (Throwable t) {
      t.printStackTrace();
      throw t;
    }
  }

  public Timestamp getTimestamp() throws Exception {
    return Timestamp.valueOf(getStringDate() + ".0");
  }

  public void reset() {
    year = -1;
    month = -1;
    day = -1;
    hours = -1;
    minutes = -1;
    seconds = -1;
  }

  private int makeInt(String val, int min, int max) {
    if (val == null) {
      return -1;
    }

    int ival;

    try {
      ival = Integer.parseInt(val);
      if ((ival < min) || (ival > max)) {
        return -1;
      }
    } catch (Exception e) {
      return -1;
    }

    return ival;
  }

  private String makeStr(int val) {
    if (val < 0) {
      return "";
    }

    return String.valueOf(val);
  }

  /** Get a two-digit representation of a one to two-digit number
   *
   * @param  val       number
   * @return String    2 digit representation of the number
   */
  private String getPad2(int val) {
    if (val >= 10) {
      return String.valueOf(val);
    }

    return "0" + val;
  }

  public static void main(String[] args) {
    try {
      String[] months = {
              "Jan", "Feb", "Mar", "Apr", "May", "Jun",
              "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};

      DateBean db = new DateBean(months, "yyyy-MM-dd HH:mm",
                                 "yyyy-MM-dd HH:mm");

      db.setDate(new Date());

      System.out.println("long date: " + db.getLongDate());
      System.out.println("timestamp: " + db.getTimestamp());
    } catch (Throwable t) {
      t.printStackTrace();
    }
  }

}
