/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.sss.util;

import edu.rpi.sss.util.fmt.FmtUtil;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Properties;

/**
 *
 * @author  Mike Douglass
 * @version
 * PropertyInfo.java
 *
 * Created on August 4, 1999, 10:33 AM
 *
 * Class and methods to help maintain a set of program properties.
 */

public class PropertyUtil extends Properties {

  // HashTables (and Properties) cannot hold null values so we keep a table
  //               of all names we ever knew.
  private Hashtable allNames;
  private Integer dummyVal = new Integer(0);

  public PropertyUtil() {
    allNames = new Hashtable();
  }

  public PropertyUtil(PropertyUtil defaults) {
    super(defaults);
    allNames = new Hashtable();
  }

  /** Create a set of properties with defined defaults.
   */
  public PropertyUtil(String[] names, String[] defaults) throws PropertyUtilException {
    this();
    put(names, defaults);
  }

  /** Create a set of properties with defined defaults.
   */
  public PropertyUtil(String[][] defaults) throws PropertyUtilException {
    this();
    put(defaults);
  }

  public PropertyUtil(Hashtable tbl) throws PropertyUtilException {
    this();
    put(tbl);
  }

  public void put(String name, String val) {
    if (val == null) {
      remove(name);
    } else {
      setProperty(name, val);
    }
    allNames.put(name, dummyVal);
  }

  public void put(String[][] namevals) throws PropertyUtilException {
    for (int i = 0; i < namevals.length; i++) {
      if (namevals[i][0] != null) put(namevals[i][0], namevals[i][1]);
      allNames.put(namevals[i][0], dummyVal);
    }
  }

  public void put(String[] names, String[] vals) throws PropertyUtilException {
    if (names.length != vals.length)
      throw new PropertyUtilException("vals size doesn't match property list size");

    for (int i = 0; i < names.length; i++) {
      if (vals[i] != null) put(names[i], vals[i]);
      allNames.put(names[i], dummyVal);
    }
  }

  /** Transfer all the name/value pairs from the hashtable.
   */
  public void put(Hashtable tbl) {
    Enumeration e = tbl.keys();
    while (e.hasMoreElements()) {
      String name = (String)e.nextElement();
      String val = (String)tbl.get(name);

      put(name, val);
    }
  }

  public synchronized Object get(Object key) {
//  public String get(String name) {
    return getProperty((String)key);
  }

  public String getStr(String key) {
    return getProperty(key);
  }

  public String get(String name, String defaultVal) {
    return getProperty(name, defaultVal);
  }

  /** If the named property is present and has a value use that.
      Otherwise, set the value to the given default and use that.
   */
  public String checkProp(String name, String defaultVal) {
//    String val = get(name);
    String val = (String)get(name);

    if (val == null) {
      put(name, defaultVal);
      val = defaultVal;
    }

    return val;
  }

  /** Ensure we have a default value for named properties.
   */
  public void setDefault(String[] names, String[] defaultVals)
      throws PropertyUtilException {
    if (names.length != defaultVals.length)
      throw new PropertyUtilException("vals size doesn't match property list size");

    for (int i = 0; i < names.length; i++) {
      if (get(names[i]) == null) {
        put(names[i], defaultVals[i]);
      }
    }
  }

  /** Ensure we have a default value for named properties.
      Doesn't set the value if it is already set.
   */
  public void setDefault(String[][] nameVals) throws PropertyUtilException {
    for (int i = 0; i < nameVals.length; i++) {
      String[] nameVal = nameVals[i];
      if (get(nameVal[0]) == null) {
        put(nameVal[0], nameVal[1]);
      }
    }
  }

  public int getInt(String name) throws PropertyUtilException {
    try {
      return Integer.parseInt(getProperty(name), 10);
    } catch (NumberFormatException e) {
      throw new PropertyUtilException("Invalid integer value: " + e.getMessage());
    }
  }

  public int getInt(String name, int defaultVal) throws PropertyUtilException {
    String val = getProperty(name);
    if (val == null) return defaultVal;

    try {
      return Integer.parseInt(val, 10);
    } catch (NumberFormatException e) {
      throw new PropertyUtilException("Invalid integer value: " + e.getMessage());
    }
  }

  public boolean getBool(String name) throws PropertyUtilException {
    return getBool(name, null);
  }

  public boolean getBool(String name, boolean defaultVal) throws PropertyUtilException {
    if (defaultVal) return getBool(name, "true");

    return getBool(name, "false");
  }

  private boolean getBool(String name, String defaultVal) throws PropertyUtilException {
    String val = getProperty(name, defaultVal);
    if (val == null) val = defaultVal;

    if (val == null)
      throw new PropertyUtilException("Invalid boolean value: null");

    val = val.toLowerCase();
    if ((val.equals("true")) || (val.equals("yes"))) return true;
    if ((val.equals("false")) || (val.equals("no"))) return false;

    throw new PropertyUtilException("Invalid boolean value: not one of true, false, yes or no");
  }

  public void setBool(String name, boolean val) throws PropertyUtilException {
    if (val) put(name, "true"); else put(name, "false");
  }

  /* The Configurator class is passed to the configure method to allow
     it to retrieve property values from the environment in which the
     PropertyUtil class is being used. The get method will be called for each
     property which we know about. A null return means don't change the
     property value.
   */
  public abstract static class Configurator {
    public abstract String get(String name);
  }

  public void configure(Configurator c) {
    Enumeration e = allNames.keys();

    while (e.hasMoreElements()) {
      String name = (String)e.nextElement();
      String val = c.get(name);

      if (val != null) put(name, val);
    }
  }

  public Object clone() {
    PropertyUtil np = new PropertyUtil();

    Enumeration e = propertyNames();
    while (e.hasMoreElements()) {
      String name = (String)e.nextElement();
      String val = getStr(name);
      np.put(name, val);
    }

    return np;
  }

  /** Some useful formatting routines
   */

  // Format a message consisting of a format string saved as a property
  //  plus one string parameter
  public String fmtProp(String propName, String arg) {
    return FmtUtil.fmtMsg(getStr(propName), arg);
  }

  // Format a message consisting of a format string saved as a property
  //  plus two string parameters
  public String fmtProp(String propName, String arg1, String arg2) {
    return FmtUtil.fmtMsg(getStr(propName), arg1, arg2);
  }

  // Format a message consisting of a format string saved as a property
  //  plus 3 string parameters
  public String fmtProp(String propName, String arg1, String arg2, String arg3) {
    return FmtUtil.fmtMsg(getStr(propName), arg1, arg2, arg3);
  }

  // Format a message consisting of a format string saved as a property
  // plus one integer parameter
  public String fmtProp(String propName, int arg) {
    return FmtUtil.fmtMsg(getStr(propName), arg);
  }

}
