/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.sss.util;

/** Encode and decode base 64 data.

From RFC 1521 (MIME)
5.2.  Base64 Content-Transfer-Encoding

   The Base64 Content-Transfer-Encoding is designed to represent
   arbitrary sequences of octets in a form that need not be humanly
   readable.  The encoding and decoding algorithms are simple, but the
   encoded data are consistently only about 33 percent larger than the
   unencoded data.  This encoding is virtually identical to the one used
   in Privacy Enhanced Mail (PEM) applications, as defined in RFC 1421.
   The base64 encoding is adapted from RFC 1421, with one change: base64
   eliminates the "*" mechanism for embedded clear text.

   A 65-character subset of US-ASCII is used, enabling 6 bits to be
   represented per printable character. (The extra 65th character, "=",
   is used to signify a special processing function.)

      NOTE: This subset has the important property that it is
      represented identically in all versions of ISO 646, including US
      ASCII, and all characters in the subset are also represented
      identically in all versions of EBCDIC.  Other popular encodings,
      such as the encoding used by the uuencode utility and the base85
      encoding specified as part of Level 2 PostScript, do not share
      these properties, and thus do not fulfill the portability
      requirements a binary transport encoding for mail must meet.

   The encoding process represents 24-bit groups of input bits as output
   strings of 4 encoded characters. Proceeding from left to right, a
   24-bit input group is formed by concatenating 3 8-bit input groups.
   These 24 bits are then treated as 4 concatenated 6-bit groups, each
   of which is translated into a single digit in the base64 alphabet.
   When encoding a bit stream via the base64 encoding, the bit stream
   must be presumed to be ordered with the most-significant-bit first.
   That is, the first bit in the stream will be the high-order bit in
   the first byte, and the eighth bit will be the low-order bit in the
   first byte, and so on.

   Each 6-bit group is used as an index into an array of 64 printable
   characters. The character referenced by the index is placed in the
   output string. These characters, identified in Table 1, below, are
   selected so as to be universally representable, and the set excludes
   characters with particular significance to SMTP (e.g., ".", CR, LF)
   and to the encapsulation boundaries defined in this document (e.g.,
   "-").

                            Table 1: The Base64 Alphabet

      Value Encoding  Value Encoding  Value Encoding  Value Encoding
           0 A            17 R            34 i            51 z
           1 B            18 S            35 j            52 0
           2 C            19 T            36 k            53 1
           3 D            20 U            37 l            54 2
           4 E            21 V            38 m            55 3
           5 F            22 W            39 n            56 4
           6 G            23 X            40 o            57 5
           7 H            24 Y            41 p            58 6
           8 I            25 Z            42 q            59 7
           9 J            26 a            43 r            60 8
          10 K            27 b            44 s            61 9
          11 L            28 c            45 t            62 +
          12 M            29 d            46 u            63 /
          13 N            30 e            47 v
          14 O            31 f            48 w         (pad) =
          15 P            32 g            49 x
          16 Q            33 h            50 y

   The output stream (encoded bytes) must be represented in lines of no
   more than 76 characters each.  All line breaks or other characters
   not found in Table 1 must be ignored by decoding software.  In base64
   data, characters other than those in Table 1, line breaks, and other
   white space probably indicate a transmission error, about which a
   warning message or even a message rejection might be appropriate
   under some circumstances.

   Special processing is performed if fewer than 24 bits are available
   at the end of the data being encoded.  A full encoding quantum is
   always completed at the end of a body.  When fewer than 24 input bits
   are available in an input group, zero bits are added (on the right)
   to form an integral number of 6-bit groups.  Padding at the end of
   the data is performed using the '=' character.  Since all base64
   input is an integral number of octets, only the following cases can
   arise: (1) the final quantum of encoding input is an integral
   multiple of 24 bits; here, the final unit of encoded output will be
   an integral multiple of 4 characters with no "=" padding, (2) the
   final quantum of encoding input is exactly 8 bits; here, the final
   unit of encoded output will be two characters followed by two "="
   padding characters, or (3) the final quantum of encoding input is
   exactly 16 bits; here, the final unit of encoded output will be three
   characters followed by one "=" padding character.

   Because it is used only for padding at the end of the data, the
   occurrence of any '=' characters may be taken as evidence that the
   end of the data has been reached (without truncation in transit).  No
   such assurance is possible, however, when the number of octets
   transmitted was a multiple of three.

   Any characters outside of the base64 alphabet are to be ignored in
   base64-encoded data.  The same applies to any illegal sequence of
   characters in the base64 encoding, such as "====="

   Care must be taken to use the proper octets for line breaks if base64
   encoding is applied directly to text material that has not been
   converted to canonical form.  In particular, text line breaks must be
   converted into CRLF sequences prior to base64 encoding. The important
   thing to note is that this may be done directly by the encoder rather
   than in a prior canonicalization step in some implementations.

      NOTE: There is no need to worry about quoting apparent
      encapsulation boundaries within base64-encoded parts of multipart
      entities because no hyphen characters are used in the base64
      encoding.
*/

public class Base64 {
  public static char[] encode(byte[] bytes) {
    int len = bytes.length;
    int pos = 0;

    // We can calculate the output length
    char[] encoded = new char[((len + 2) / 3) * 4];
    int epos = 0;

    while (pos < len) {
      int r = len - pos;
      int bits = ((int)bytes[pos] & 0xff);

      bits <<= 8; // append 0s at least
      if (r > 1) {
        // Or in next char
        bits |= ((int)bytes[pos + 1] & 0xff);
      }

      bits <<= 8; // append 0s at least
      if (r > 2) {
        bits |= ((int)bytes[pos + 2] & 0xff);
      }

      // Now convert 6 bit chunks.
      if (r <= 2) {
        // last 6 bits don't count
        encoded[epos + 3] = '=';
      } else {
        encoded[epos + 3] = alphabet[(bits & 0x3f)];
      }
      bits >>= 6;

      if (r <= 1) {
        // last 6 bits don't count
        encoded[epos + 2] = '=';
      } else {
        encoded[epos + 2] = alphabet[(bits & 0x3f)];
      }
      bits >>= 6;
      encoded[epos + 1] = alphabet[(bits & 0x3f)];
      bits >>= 6;
      encoded[epos] = alphabet[(bits & 0x3f)];

      epos += 4;
      pos += 3;
    }

    return encoded;
  }

  public static byte[] decode(String s) {
    return decode(s.toCharArray());
  }

  public static byte[] decode(char[] chars) {
    int len = chars.length;
    int pos = 0;

    /* Calculate the output length. We have to check the chars at the end
       for padding. This may be too long if there are extra chars in the
       input (new lines, blanks etc)
     */
    int outlen = ((len + 3) / 4) * 3;

    if ((len > 0) && (chars[len - 1] == '=')) {
      outlen--;
    }

    if ((len > 1) && (chars[len - 2] == '=')) {
      outlen--;
    }

    byte[] decoded = new byte[outlen];

    // Each valid 4 char segment represents 3 bytes of data
    int bits = 0;
    int numchars = 0; // Number of chars in seg.
    int outi = 0;

    while (pos < len) {
      //int val = ((int)decoder[chars[pos] & 0xff]) & 0xff;

      int val = decoder[chars[pos] & 0xff];

      if (val >= 0) {
        // Add in to the bits.
        bits <<= 6;
        bits |= val;
        numchars++;
      }

      pos++;

      if ((numchars == 4) || ((numchars > 0) && (pos == len))) {
        // Output this seg.
        int sn = numchars;
        while (sn < 4) {
          bits <<= 6;
          sn++;
        }

        int numbytes = 1;

        if (numchars == 4) {
          // last byte valid
          decoded[outi + 2] = (byte)(bits & 0xff);
          numbytes++;
        }
        bits >>= 8;

        if (numchars > 2) {
          // second byte valid
          decoded[outi + 1] = (byte)(bits & 0xff);
          numbytes++;
        }
        bits >>= 8;

        decoded[outi] = (byte)(bits & 0xff);

        outi += numbytes;
        numchars = 0;
      }
    }

    if (outi == outlen) {
      return decoded;
    }

    byte[] shorter = new byte[outi];

    System.arraycopy(decoded, 0, shorter, 0, outi);

    return shorter;
  }

  /* Define the encoding and decoding tables.
   */

  private final static char[] alphabet = {
    'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M',
    'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z',
    'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm',
    'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z',
    '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
    '+', '/'
  };

  // Table for decoding.
  static private byte[] decoder = new byte[256];
  static {
    for (int i = 0; i < 256; i++) decoder[i] = -1; // Ignore

    for (int i = 0; i < alphabet.length; i++) {
      decoder[alphabet[i] & 0xff] = (byte)i;
    }
  }

}
