/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.cct.uwcal.webcommon;

import edu.rpi.sss.util.jsp.JspUtil;
import edu.rpi.sss.util.servlets.PresentationState;

import org.apache.struts.util.MessageResources;

/** This ought to be made pluggable. We need a session factory which uses
 * CalEnv to figure out which implementation to use.
 *
 * <p>This class represents a session for the UWCal web interface.
 * Some user state will be retained here.
 * We also provide a number of methods which act as the interface between
 * the web world and the calendar world.
 *
 * <p>The UW web interface has session support that may not be applicable
 * to all potential users of this application. We should try to interface to
 * it through this.
 *
 * @author Mike Douglass   douglm@rpi.edu
 */
public class UWCalSessionImpl implements UWCalSession {
  private boolean autoCreateSponsors;

  private boolean autoDeleteSponsors;

  private boolean autoCreateLocations;

  private boolean autoDeleteLocations;

  /** Not valid in the j2ee world but it's only used to count sessions.
   */
  private static int sessionNum = 0;

  /** True if we want debugging output
   */
  private boolean debug;

  /** The current user - null for guest
   */
  private String user;

  /** The application root
   */
  private String appRoot;

  /** The application name
   */
  private String appName;

  /** The current presentation state of the application
   */
  private PresentationState ps;

  /** Constructor for a UWCalSession
   *
   * @param user    String user id
   * @param appName String identifying particular application
   * @param debug   true for debugging output
   */
  public UWCalSessionImpl(String user,
                          String appRoot,
                          String appName,
                          PresentationState ps,
                          MessageResources messages,
                          String schemeHostPort,
                          boolean debug) throws Throwable {
    this.user = user;
    this.appRoot = appRoot;
    this.appName = appName;
    this.ps = ps;
    this.debug = debug;

    if (ps != null) {
      if (ps.getAppRoot() == null) {
        if (!appRoot.toLowerCase().startsWith("http")) {
          appRoot = schemeHostPort + "/" + appRoot;
        }

        ps.setAppRoot(appRoot);
      }
    }

    autoCreateSponsors = JspUtil.getProperty(messages,
                            "org.ucal.calendar.pubevents.autocreatesponsors",
                            "false").equals("true");

    autoDeleteSponsors = JspUtil.getProperty(messages,
                            "org.ucal.calendar.pubevents.autodeletesponsors",
                            "false").equals("true");

    autoCreateLocations = JspUtil.getProperty(messages,
                            "org.ucal.calendar.pubevents.autocreatelocations",
                            "false").equals("true");

    autoDeleteLocations = JspUtil.getProperty(messages,
                            "org.ucal.calendar.pubevents.autodeletelocations",
                            "false").equals("true");

    sessionNum++;
  }
  /** ===================================================================
   *                   Options set from application properties
   *  =================================================================== */

  /** True if we should auto-create sponsors. Some sites may wish to control
   * the creation of sponsors to enforce consistency in their use. If this
   * is true we create a sponsor as we create events. If false the sponsor
   * must already exist.
   */
  public boolean getAutoCreateSponsors() {
    return autoCreateSponsors;
  }

  /** True if we should auto-delete sponsors. Some sites may wish to control
   * the deletion of sponsors to enforce consistency in their use. If this
   * is true we delete a sponsor when it becomes unused.
   */
  public boolean getAutoDeleteSponsors() {
    return autoDeleteSponsors;
  }

  /** True if we should auto-create locations. Some sites may wish to control
   * the creation of locations to enforce consistency in their use. If this
   * is true we create a location as we create events. If false the location
   * must already exist.
   */
  public boolean getAutoCreateLocations() {
    return autoCreateLocations;
  }

  /** True if we should auto-delete locations. Some sites may wish to control
   * the deletion of locations to enforce consistency in their use. If this
   * is true we delete a location when it becomes unused.
   */
  public boolean getAutoDeleteLocations() {
    return autoDeleteLocations;
  }

  /** =====================================================================
   *                     Property methods
   *  ===================================================================== */

  public void setSessionNum(int val) {
    sessionNum = val;
  }

  public int getSessionNum() {
    return sessionNum;
  }

  public void setAppName(String val) {
    appName = val;
  }

  public String getAppName() {
    return appName;
  }

  public void setAppRoot(String val) {
    appRoot = val;
  }

  public String getAppRoot() {
    return appRoot;
  }

  public void setUser(String val) {
    user = val;
  }

  public String getUser() {
    return user;
  }

  public void setPresentationState(PresentationState val) {
    ps = val;
    if (ps != null) {
      ps.setAppRoot(getAppRoot());
    }
  }

  public PresentationState getPresentationState() {
    return ps;
  }

  /** Is this a guest user?
   *
   * @return boolean true for a guest
   */
  public boolean isGuest() {
    return user == null;
  }
}

