/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.cct.uwcal.webcommon;

// I only need this because request.getInitParameterNames doesn't work
import edu.rpi.cct.uwcal.common.UserAuthPar;

import edu.rpi.cct.uwcal.calsvci.CalSvc;
import edu.rpi.cct.uwcal.calsvci.CalSvcI;

import edu.rpi.sss.util.jsp.JspUtil;
import edu.rpi.sss.util.jsp.SessionListener;
import edu.rpi.sss.util.jsp.UtilAbstractAction;
import edu.rpi.sss.util.jsp.UtilActionForm;
import edu.washington.cac.calfacade.shared.CalFacadeAccessException;
import edu.washington.cac.calfacade.shared.CalFacadeException;
import edu.washington.cac.calfacade.shared.MyCalendarVO;
import edu.washington.cac.calfacade.shared.UserAuth;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import org.apache.struts.util.MessageResources;

/** This abstract action performs common setup actions before the real
 * action method is called.
 *
 * <p>Errors:<ul>
 *      <li>org.ucal.calendar.error.noaccess - when user has insufficient
 *            access</li>
 * </ul>
 *
 * @author  Mike Douglass  douglm@rpi.edu
 */
public abstract class UWCalAbstractAction extends UtilAbstractAction {
  public String getId() {
    return getClass().getName();
  }

  public String performAction(HttpServletRequest request,
                              HttpServletResponse response,
                              UtilActionForm frm,
                              MessageResources messages) throws Throwable {
    String forward = "success";
    UWCalActionFormBase form = (UWCalActionFormBase)frm;

    String adminUserId = form.getCurrentUser();

    if (getPublicAdmin()) {
      /** We may want to masquerade as a different user
       */

      String temp = request.getParameter("adminUserId");

      if (temp != null) {
        adminUserId = temp;
      }
    }

    UWCalSession s = getState(request,
                              form,
                              messages,
                              adminUserId,
                              false);

    form.setSession(s);
    form.setGuest(s.isGuest());

    if (form.getGuest()) {
      // force public view on - off by default
      form.setPublicView(true);
    }

    form.setToday(new MyCalendarVO());

    String refreshAction = JspUtil.getProperty(messages,
                                          "org.ucal.calendar.refreshaction",
                                          null);

    if (refreshAction == null) {
      refreshAction = form.getActionPath();
    }

    if (refreshAction != null) {
      setRefreshInterval(request, response,
                         JspUtil.getIntProperty(messages,
                              "org.ucal.calendar.refreshinterval",
                              0),
                         refreshAction, form);
    }

    String appBase = form.getAppBase();

    if (appBase != null) {
      // Embed in request for pages that cannot access the form (loggedOut)
      request.setAttribute("org.cal.action.appbase", appBase);
    }

    try {
      forward = doAction(request, s, form);
    } catch (CalFacadeAccessException cfae) {
      err.emit("org.ucal.calendar.error.noaccess", "for that action");
    }

    return forward;
  }

  /** Override to return true if this is an admin client
   *
   * @return boolean  true for a public events admin client
   */
  public boolean getPublicAdmin() {
    return false;
  }

  /** This is the routine which does the work.
   *
   * @param request   Needed to locate session
   * @param sess      UWCalSession calendar session object
   * @param frm       Action form
   * @return String   forward name
   */
  public abstract String doAction(HttpServletRequest request,
                                  UWCalSession sess,
                                  UWCalActionFormBase frm) throws Throwable;

  /** Get the session state object for a web session. If we've already been
   * here it's embedded in the current session. Otherwise create a new one.
   *
   * <p>We also carry out a number of web related operations.
   *
   * @param request       HttpServletRequest Needed to locate session
   * @param form          Action form
   * @param messages      MessageResources needed for the resources
   * @param adminUserId   id we want to administer
   * @param admin         Get this for the admin client
   * @return UWCalSession null on failure
   */
  public synchronized UWCalSession getState(
                                    HttpServletRequest request,
                                    UWCalActionFormBase form,
                                    MessageResources messages,
                                    String adminUserId,
                                    boolean admin) throws Throwable {
    UWCalSession s = UWCalWebUtil.getState(request);
    HttpSession sess = request.getSession(false);

    if (s != null) {
      if (debug) {
        getLogger().debug("getState-- obtainedfrom session");
        getLogger().debug("getState-- timeout interval = " +
                          sess.getMaxInactiveInterval());
      }
    } else {
      if (debug) {
        getLogger().debug("getState-- get new object");
      }

      String appRoot = messages.getMessage("org.ucal.calendar.app.root");
      String appName = messages.getMessage("org.ucal.calendar.app.name");

      /** The actual session class used is possibly site dependent
       */
      s = new UWCalSessionImpl(form.getCurrentUser(), appRoot, appName,
                               form.getPresentationState(), messages,
                               form.getSchemeHostPort(), debug);

      UWCalWebUtil.setState(request, s);

      setSessionAttr(request, "cal.pubevents.client",
                     messages.getMessage("org.ucal.calendar.pubevents.client"));

      setSessionAttr(request, "cal.user.client",
                     messages.getMessage("org.ucal.calendar.user.client"));

      form.setHour24("true".equals(
                     messages.getMessage("org.ucal.calendar.app.hour24")));

      String minInc = messages.getMessage("org.ucal.calendar.app.minincrement");
      form.setMinIncrement(Integer.parseInt(minInc));

      form.assignShowYearData("true".equals(
                   messages.getMessage("org.ucal.calendar.app.showyeardata")));

      String temp = messages.getMessage("org.ucal.calendar.pubevents.client.host");
      if (temp == null) {
        temp = form.getSchemeHostPort();
      }

      setSessionAttr(request, "cal.pubevents.client.host", temp);

      temp = messages.getMessage("org.ucal.calendar.user.client.host");
      if (temp == null) {
        temp = form.getSchemeHostPort();
      }

      setSessionAttr(request, "cal.user.client.host", temp);

      String raddr = request.getRemoteAddr();
      String rhost = request.getRemoteHost();
      SessionListener.setId(appName); // First time will have no name
      getLogger().info("===============" + appName + ": New session (" +
                       s.getSessionNum() + ") from " +
                       rhost + "(" + raddr + ")");

      if (!admin) {
        /** Ensure the session timeout interval is longer than our refresh period
         */
        //  Should come from db -- int refInt = s.getRefreshInterval();
        int refInt = 60; // 1 min refresh?

        if (refInt > 0) {
          int timeout = sess.getMaxInactiveInterval();

          if (timeout <= refInt) {
            // An extra minute should do it.
            getLogger().debug("@+@+@+@+@+ set timeout to " + (refInt + 60));
            sess.setMaxInactiveInterval(refInt + 60);
          }
        }
      }
    }

    int access = getAccess(request, messages);
    if (debug) {
      getLogger().debug("Container says that current user has the type: " +
                        access);
    }

    /** Ensure we have a CalAdminSvcI object
     */
    checkSvci(request, form, access, adminUserId, s.isGuest(),
              getPublicAdmin(), false, debug);

    /** Somewhere up there we may have to do more for user auth in the
        session. This is where we can figure out this is a first call.
     */

    UserAuth ua = null;
    try {
      ua = form.getCalSvcI().getUserAuth();
    } catch (Throwable t) {
      getLogger().error("Exception from getUserAuth: ", t);
    }

    if (ua.getUserid() == null) {
      UserAuthPar par = new UserAuthPar();
      par.svlt = servlet;
      par.req = request;
      ua.initialise(s.getUser(), par);

//      ua.initialise(s.getUser(),
//                    ua.adjustUsertype(access));
    }

    form.assignUserAuth(ua);

    if (debug) {
      try {
        getLogger().debug("UserAuth says that current user has the type: " +
                          ua.getUsertype());
      } catch (Throwable t) {
        getLogger().error("Exception from UserAuth: ", t);
      }
    }

    return s;
  }

  /** Ensure we have a CalAdminSvcI object for the given user.
   *
   * <p>For an admin client with a super user we may switch to a different
   * user to administer their events.
   *
   * @param request       Needed to locate session
   * @param form          Action form
   * @param access        int unadjusted access
   * @param user          String user we want to be
   * @param isGuest       true if they will be a guest user
   * @param publicAdmin   true if this is an administrative client
   * @param canSwitch     true if we should definitely allow user to switch
   *                      this allows a user to switch between and into
   *                      groups of which they are a member
   * @param debug         true for all that debugging stuff
   * @return boolean      false for problems.
   */
  protected boolean checkSvci(HttpServletRequest request,
                              UWCalActionFormBase form,
                              int access,
                              String user,
                              boolean isGuest,
                              boolean publicAdmin,
                              boolean canSwitch,
                              boolean debug) throws CalFacadeException {
    /** Do some checks first
     */
    boolean switching = false;
    String curUser = String.valueOf(form.getCurrentUser());

    if (!publicAdmin) {
      if (!curUser.equals(String.valueOf(user))) {
        return false;
      }
    } else {
      canSwitch = canSwitch || ((access & UserAuth.contentAdminUser) != 0) ||
                           ((access & UserAuth.superUser) != 0);

      if (!canSwitch && !curUser.equals(String.valueOf(user))) {
        /** Trying to switch but not allowed */
        return false;
      }
    }

    CalSvcI svci = UWCalWebUtil.getCalSvcI(request);

    if (svci != null) {
      getLogger().debug("CalSvcI-- Obtained from session for user " +
                        svci.getUserVO());

      if (!curUser.equals(String.valueOf(user))) {
        /** Switching user */
        svci = null;
        switching = true;
      }
    }

    if (svci == null) {
      if (debug) {
        getLogger().debug(".CalSvcI-- get new object for user " + user);
      }

      svci = new CalSvc();
      svci.init(servlet, user, access, isGuest,
                publicAdmin, debug);

      UWCalWebUtil.setCalSvcI(request, svci);
    }

    if (switching) {
      svci.touch();
    }

    form.setCalSvcI(svci);

    return true;
  }

  /** This class determines the access rights of the current user based on
   * their assigned roles. There are two sections to this which appear to do
   * the same thing.
   *
   * <p>They are there because some servlet containers (jetty for one)
   * appear to be broken. Role mapping does not appear to work reliably.
   * Thsi seems to have something to do with jetty doing internal redirects
   * to handle login. In the process it seems to lose the appropriate servlet
   * context and with it the mapping of roles.
   *
   */
  protected int getAccess(HttpServletRequest req,
                          MessageResources messages)
         throws CalFacadeException {
    int access = 0;

    /** This form works with broken containers.
     */
    if (req.isUserInRole(
          getMessages().getMessage("org.ucal.calendar.role.admin"))) {
      access += UserAuth.superUser;
    }

    if (req.isUserInRole(
          getMessages().getMessage("org.ucal.calendar.role.contentadmin"))) {
      access += UserAuth.contentAdminUser;
    }

    if (req.isUserInRole(
          getMessages().getMessage("org.ucal.calendar.role.alert"))) {
      access += UserAuth.alertUser;
    }

    if (req.isUserInRole(
          getMessages().getMessage("org.ucal.calendar.role.owner"))) {
      access += UserAuth.publicEventUser;
    }

    /** This is how it out to look
    if (req.isUserInRole("admin")) {
      access += UserAuth.superUser;
    }

    if (req.isUserInRole("contentadmin")) {
      access += UserAuth.contentAdminUser;
    }

    if (req.isUserInRole("alert")) {
      access += UserAuth.alertUser;
    }

    if (req.isUserInRole("owner")) {
      access += UserAuth.publicEventUser;
    } */

    return access;
  }
}

