/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.cct.uwcal.common;

import edu.washington.cac.calfacade.shared.Calintf;
import edu.washington.cac.calfacade.shared.EventVO;
import edu.washington.cac.calfacade.shared.MyCalendarVO;

import java.io.Serializable;
import java.util.Vector;

/** This class represents a view of the calendar from a startDate to an
 * endDate. The getTimePeriodInfo method always returns a tree structure
 * with a single year as the root containing months, weeks, then days.
 *
 * <p>Each day element can return data for that day.
 *
 * <p>This structure facilitates the production of calendar like views as
 * each month and week is padded out at the start and end with filler
 * elements.
 *
 * <p>The calendar is represented as a sequence of, possibly overlapping,
 * events which must be rendered in some manner by the display.
 *
 * @author  Mike Douglass douglm@rpi.edu
 */
public class TimeView implements Serializable {
  protected boolean debug;
  protected MyCalendarVO curDay;
  protected Calintf cal;
  protected String periodName;
  protected MyCalendarVO firstDay;
  protected MyCalendarVO lastDay;
  protected String prevDate;
  protected String nextDate;
  protected boolean showData;

  /** set on the first call to getTimePeriodInfo
   */
  private TimeViewDailyInfo[] tvdis;

  /** Constructor:
   *
   * @param  curDay    MyCalendarVO representing current day.
   * @param  periodName Name of period, capitalized, e.g. Week
   * @param  cal       Calintf calendar interface
   * @param  firstDay  MyCalendar representing first day of period.
   * @param  lastDay   MyCalendar representing last day of period.
   * @param  prevDate  previous date for this time period in YYYYMMDD form
   * @param  nextDate  next date for this time period in YYYYMMDD form
   * @param  showData  boolean true if this TimeView can be used to
   *                   display events or if it is used for structure only.
   *                   For example we may use the year for navigation only
   *                   to reduce the amount of data retrieved.
   * @param  debug     true for some debugging output
   */
  public TimeView(MyCalendarVO curDay,
                  String periodName,
                  Calintf cal,
                  MyCalendarVO firstDay,
                  MyCalendarVO lastDay,
                  String prevDate,
                  String nextDate,
                  boolean showData,
                  boolean debug) {
    this.curDay = curDay;
    this.periodName = periodName;
    this.cal = cal;
    this.firstDay = firstDay;
    this.lastDay = lastDay;
    this.curDay = curDay;
    this.prevDate = prevDate;
    this.nextDate = nextDate;
    this.showData = showData;
    this.debug = debug;
  }

  /** Override this for a single day view
   */
  public boolean isMultiDay() {
    return true;
  }

  /** This method returns the period name (week, month etc)
   *
   * @return  String  period name
   */
  public String getPeriodName() {
    return periodName;
  }

  public MyCalendarVO getFirstDay() {
    return firstDay;
  }

  public MyCalendarVO getCurDay() {
    return curDay;
  }

  public MyCalendarVO getLastDay() {
    return lastDay;
  }

  public boolean isFirstDay(MyCalendarVO date) {
    return firstDay.isSameDate(date);
  }

  public boolean isLastDay(MyCalendarVO date) {
    return lastDay.isSameDate(date);
  }

  /** This method returns the first day in YYYYMMDD format.
   *
   * @return  String  first date
   */
  public String getFirstDate() {
    return firstDay.getDateDigits();
  }

  /** This method returns the last day in YYYYMMDD format.
   *
   * @return  String  last date
   */
  public String getLastDate() {
    return lastDay.getDateDigits();
  }

  /** This method returns the previous date in YYYYMMDD format.
   *
   * @return  String  previous date
   */
  public String getPrevDate() {
    return prevDate;
  }

  /** This method returns the current date in YYYYMMDD format.
   *
   * @return  String  current date
   */
  public String getCurDate() {
    return curDay.getDateDigits();
  }

  /** This method returns the next date in YYYYMMDD format.
   *
   * @return  String  next date
   */
  public String getNextDate() {
    return nextDate;
  }

  /** Return showData flag.
   *
   * @return  boolean  true if we should show data
   */
  public boolean getShowData() {
    return showData;
  }

  /** Return the events for the given day as an array of value objects
   *
   * @param   date    MyCalendar object defining day
   * @return  EventVO[]  one days events or null for no events.
   */
  /* !!!!!!! Needs to throw an exception for an error */
  public EventVO[] getDaysEvents(MyCalendarVO date) throws Throwable {
    return cal.getDaysEvents(date);
  }

  /** Return an array of the days of the week indexed from 0
   * Elements have been adjusted so that the first day of the week is at 0
   *
   * @return String[]  days of week
   */
  public String[] getDayNamesAdjusted() {
    return MyCalendarVO.getDayNamesAdjusted();
  }

  /** Return an array of the short days of the week indexed from 0
   * Elements have been adjusted so that the first day of the week is at 0
   *
   * @return String[]  days of week
   */
  public String[] getShortDayNamesAdjusted() {
    return MyCalendarVO.getShortDayNamesAdjusted();
  }

  /** Return the dayOfWeek regarded as the first day
   *
   * @return int   firstDayOfWeek
   */
  public int getFirstDayOfWeek() {
    return MyCalendarVO.getFirstDayOfWeek();
  }

  /** Class to hold data needed while we build this thing
   */
  static class GtpiData {
    /** True if we've done the last entry
     */
    boolean isLast;

    /** True if we're doing the first entry
     */
    boolean isFirst = true;

    /** True if we're starting a new month
     */
    boolean newMonth = true;

    MyCalendarVO currentDay;

    MyCalendarVO first;
    MyCalendarVO last;
    boolean multi;

    String monthName = null;

    /** Cuurent month we are processing
     */
    int curMonth;

    /** todays month
     */
    String todaysMonth;

    /** True if we are in the current month */
    boolean inThisMonth = false;

    String year;

    int weekOfYear = 1;

    /** Need this so we can flag end of month
     */
    TimeViewDailyInfo prevTvdi;
  }

  /** Return an array of TimeViewDailyInfo describing the period this
   * view covers. This will not include events
   *
   * @return TimeViewDailyInfo[]  array of info - one entry per day
   */
  public TimeViewDailyInfo[] getTimePeriodInfo() {
    /* In the following code we assume that we never cross year boundaries
      so that the year is always constant.
     */
    if (tvdis != null) {
      return tvdis;
    }

    try {
      GtpiData gtpi = new GtpiData();

      Vector months = new Vector();
      Vector weeks = new Vector();

      gtpi.first = getFirstDay();
      gtpi.last = getLastDay();
      gtpi.multi = !gtpi.last.isSameDate(gtpi.first);
      gtpi.currentDay = new MyCalendarVO(gtpi.first.getCalendar());
      gtpi.year = "" + gtpi.currentDay.getYear();

      gtpi.todaysMonth = new MyCalendarVO().getTwoDigitMonth();

      initGtpiForMonth(gtpi);

      /* Our month entry */
      TimeViewDailyInfo monthTvdi = new TimeViewDailyInfo();
      initTvdi(monthTvdi, gtpi);

      /* Create a year entry */
      TimeViewDailyInfo yearTvdi = new TimeViewDailyInfo();
      yearTvdi.setCal(gtpi.currentDay);
      yearTvdi.setYear(gtpi.year);
      yearTvdi.setDate(gtpi.currentDay.getDateDigits());
      yearTvdi.setDateShort(gtpi.currentDay.getDateString());
      yearTvdi.setDateLong(gtpi.currentDay.getLongDateString());

      for (;;) {
        TimeViewDailyInfo weekTvdi = new TimeViewDailyInfo();

        initTvdi(weekTvdi, gtpi);

        weekTvdi.setEntries(getOneWeekTvdi(gtpi));
        weeks.addElement(weekTvdi);

        if (getFirstDayOfWeek() == gtpi.currentDay.getDayOfWeek()) {
          gtpi.weekOfYear++;
        }

        if (gtpi.isLast || gtpi.newMonth) {
          /** First add all the weeks to this month
           */

          if (gtpi.prevTvdi != null) {
            gtpi.prevTvdi.setLastDayOfMonth(true);
          }

          monthTvdi.setEntries(
             (TimeViewDailyInfo[])weeks.toArray(new TimeViewDailyInfo[
                  weeks.size()]));
          months.addElement(monthTvdi);

          if (gtpi.isLast) {
            break;
          }

          /** Set up for a new month.
           */

          initGtpiForMonth(gtpi);

          monthTvdi = new TimeViewDailyInfo();
          initTvdi(monthTvdi, gtpi);
          weeks = new Vector();
        }
      }

      yearTvdi.setEntries(
              (TimeViewDailyInfo[])months.toArray(new TimeViewDailyInfo[months.size()]));

      tvdis = new TimeViewDailyInfo[1];
      tvdis[0] = yearTvdi;

      return tvdis;
    } catch (Throwable t) {
      if (debug) {
        t.printStackTrace();
      }

      // !!!!!!!!!!! We need an error object

      return null;
    }
  }

  private void initGtpiForMonth(GtpiData gtpi) {
    gtpi.curMonth = gtpi.currentDay.getMonth();
    gtpi.monthName = gtpi.currentDay.getMonthName();
    gtpi.inThisMonth = gtpi.todaysMonth.equals(gtpi.currentDay.getTwoDigitMonth());
  }

  private void initTvdi(TimeViewDailyInfo tvdi, GtpiData gtpi) {
    tvdi.setView(this);
    tvdi.setCal(gtpi.currentDay);
    tvdi.setMultiDay(gtpi.multi);
    tvdi.setMonth(gtpi.currentDay.getTwoDigitMonth());
    tvdi.setMonthName(gtpi.monthName);
    tvdi.setYear(gtpi.year);
    tvdi.setDate(gtpi.currentDay.getDateDigits());
    tvdi.setDateShort(gtpi.currentDay.getDateString());
    tvdi.setDateLong(gtpi.currentDay.getLongDateString());
    tvdi.setCurrentMonth(gtpi.inThisMonth);
    tvdi.setWeekOfYear("" + gtpi.weekOfYear);
  }

  /** Build up to one weeks worth of days info. We assume that at least one
   * day will go into the current week. We exit at the end of the week, the
   * end of the month or the end of the time period.
   *
   * @param gtpi     GtpiData object supplying many parameters
   * @return TimeViewDailyInfo[] array of entries for this week.
   */
  private TimeViewDailyInfo[] getOneWeekTvdi(GtpiData gtpi) throws Throwable {
    Vector days = new Vector();
    TimeViewDailyInfo tvdi;

    /** First see if we need to insert leading fillers */
    int dayOfWeek = gtpi.currentDay.getDayOfWeek();
    int dayNum = getFirstDayOfWeek();

    while (dayNum != dayOfWeek) {
      tvdi = new TimeViewDailyInfo();
      tvdi.setFiller(true);

      days.addElement(tvdi);
      dayNum++;

      if (dayNum > 6) {
        dayNum = 0;
      }

      // Check we got this right
      if (days.size() > 7) {
        throw new Exception("Programming error in getOneWeekTvdi");
      }
    }

    for (;;) {
      dayOfWeek = gtpi.currentDay.getDayOfWeek();

      if (gtpi.currentDay.getMonth() != gtpi.curMonth) {
        gtpi.newMonth = true;
        break;
      }

      gtpi.isLast = gtpi.last.isSameDate(gtpi.currentDay);

      /* Create a day entry */
      tvdi = new TimeViewDailyInfo();

      initTvdi(tvdi, gtpi);

      tvdi.setDayEntry(true);

      tvdi.setFirstDay(gtpi.isFirst);
      tvdi.setLastDay(gtpi.isLast);
      tvdi.setDayOfMonth(gtpi.currentDay.getDay());
      tvdi.setDayOfWeek(dayOfWeek);

      /** Is this correct? The days of the week are rotated to adjust for
       *   first day differences. */
      tvdi.setDayName(MyCalendarVO.getDayNames()[dayOfWeek]);
      tvdi.setFirstDayOfMonth(gtpi.newMonth);
      gtpi.newMonth = false;

      tvdi.setFirstDayOfWeek(getFirstDayOfWeek() == dayOfWeek);
      tvdi.setLastDayOfWeek(MyCalendarVO.getLastDayOfWeek() == dayOfWeek);

      days.addElement(tvdi);
      gtpi.isFirst = false;

      gtpi.prevTvdi = tvdi;

      gtpi.currentDay = gtpi.currentDay.getTomorrow();

      if (gtpi.isLast || tvdi.isLastDayOfWeek()) {
        // Watch for it also being the last day of the month
        if (gtpi.currentDay.getMonth() != gtpi.curMonth) {
          gtpi.newMonth = true;
        }

        break;
      }
    }

    /** Pad it out to seven days
     */
    while (days.size() < 7) {
      tvdi = new TimeViewDailyInfo();
      tvdi.setFiller(true);

      days.addElement(tvdi);
    }

    return (TimeViewDailyInfo[])days.toArray(new TimeViewDailyInfo[
                    days.size()]);
  }

  public String toString() {
    StringBuffer sb = new StringBuffer();

    sb.append("TimeView{");
    sb.append(periodName);
    sb.append(", cur=");
    sb.append(String.valueOf(curDay));
    sb.append(", firstDay=");
    sb.append(String.valueOf(firstDay));
    sb.append(", lastDay=");
    sb.append(String.valueOf(lastDay));
    sb.append("}");

    return sb.toString();
  }
}

