/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.cct.uwcal.common;

import edu.washington.cac.calfacade.shared.EventVO;
import edu.washington.cac.calfacade.shared.MyCalendarVO;

import java.io.Serializable;
import java.sql.Date;
import java.sql.Time;

/** Object to provide formatting services for an EventVO.
 *
 * @author Mike Douglass   douglm@rpi.edu
 */
public class EventFormatter implements Serializable {
  /** The event
   */
  private EventVO event;

  /** The view currently in place.
   */
  private TimeView view;

  /** Set so that questions can be asked about the time */
  private MyCalendarVO today;

  /** Set dynamically on request to represent start date/time */
  private MyCalendarVO startCalendar;

  /** Set dynamically on request to represent end date/time */
  private MyCalendarVO endCalendar;

  public EventFormatter(EventVO event, TimeView view) {
    this.event = event;
    this.view = view;
  }

  /** =====================================================================
   *                     Property methods
   *  ===================================================================== */

  public void setEvent(EventVO val) {
    event = val;
  }

  public EventVO getEvent() {
    return event;
  }

  /** ===================================================================
                      Convenience methods
      =================================================================== */

  public void setToday(MyCalendarVO val) {
    today = val;
  }

  public MyCalendarVO getToday() {
    if (today == null) {
      today = new MyCalendarVO();
    }

    return today;
  }

  /** Get the event's starting day and time
   *
   * @return MyCalendar  object corresponding to the event's
   *                     starting day and time
   */
  public MyCalendarVO getStartCalendar() {
    if (startCalendar == null) {
      startCalendar = myCalendar(event.getStartdate(), event.getStarttime());
    }

    return startCalendar;
  }

  /** Get the event's ending day and time
   *
   * @return MyCalendar  object corresponding to the event's
   *                     ending day and time
   */
  public MyCalendarVO getEndCalendar() {
    if (endCalendar == null) {
      endCalendar = myCalendar(event.getEnddate(), event.getEndtime());
    }

    return endCalendar;
  }

  /** Return true if this event starts in the morning of the given day.
   *
   * @return  true     if it's in the morning
   */
  public boolean isAm() {
    MyCalendarVO st = getStartCalendar();

    if (st == null) {
      return false;
    }

    return st.getHasTime() &&
           st.isSameDate(getToday()) &&
           st.getHour() < 12;
  }

  /** Return true if this event starts in the afternoon of the given day.
   *
   * @return  true     if it's in the afternoon
   */
  public boolean isPm() {
    MyCalendarVO et = getEndCalendar();

    if (et == null) {
      return false;
    }

    return et.getHasTime() &&
           et.isSameDate(getToday()) &&
           et.getHour() > 11;
  }

  /** Return the events start date and time taking into account the current
   * day (supress date if it's today)
   *
   * @return String  formatted start date.
   */
  public String getStartDateTime() {
    return getDateTime(getStartCalendar(), true);
  }

  /** Return the events end date and time taking into account the current
   * day (supress date if it's today)
   *
   * @return String  formatted end date.
   */
  public String getEndDateTime() {
    return getDateTime(getEndCalendar(), false);
  }


  /** Return the events start and end date and time taking into account the
   * current day (supress date if it's today)
   *
   * @return String  formatted start to end date.
   */
  public String getDateTimeRange() {
    StringBuffer dt = new StringBuffer();

    dt.append(getStartDateTime());

    if (getEndCalendar() != null) {
      dt.append(" - ");
      dt.append(getEndDateTime());
    }

    return dt.toString();
  }

  /** ===================================================================
                      Private methods
      =================================================================== */

  private String getDateTime(MyCalendarVO cal, boolean start) {
    StringBuffer dt = new StringBuffer();

    if (shouldPrintDate(cal, start)) {
      dt.append(cal.getDateString());

      if (cal.getHasTime()) {
        dt.append(" ");
      }
    }

    if (cal.getHasTime()) {
      dt.append(cal.getTimeString().toLowerCase());
    }

    return dt.toString();
  }

  /** Is this date interesting enough to print?
   *
   * @param date     MyCalendar date in question
   * @param start    boolean true if the date is a start date?
   * @return boolean true if this date is interesting enough to print
   */
  private boolean shouldPrintDate(MyCalendarVO date, boolean start) {
    return dateIsToday(date) &&
           (date.getHasTime() || currentDayIsBoundaryDay(start));
  }

  /**  Is the current day the interesting boundary day for a start/end date?
   *
   *  @param start   boolean true if we are interested in start dates.
   *                 false if we are interested in end dates.
   *  @return boolean  true for start and we are printing the first day of the
   *                   view<br/>
   *                   true for !start and we are printing the last day of the
   *                   view<br/>
   *                   false otherwise
   */
  private boolean currentDayIsBoundaryDay(boolean start) {
    if (start) {
      return view.isFirstDay(getToday());
    }

    return view.isLastDay(getToday());
  }

  /** Is the given date today
   *
   * @param date     MyCalendar representing a date
    @return boolean  true if date is today
   */
  private boolean dateIsToday(MyCalendarVO date) {
    return !date.isSameDate(getToday());
  }

  /** Create a calendar object based on date/time
   *
   * @param date          Date object
   * @param time          Time object
   * @return MyCalendarVO Object representing date/time or null if both pars null.
   */
  private MyCalendarVO myCalendar(Date date, Time time) {
    if (time == null && date == null) {
        return null;
    }

    return new MyCalendarVO(date, time);
  }
}

