/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.cct.uwcal.calsvci;

import java.io.Serializable;
import java.sql.Date;
import java.sql.Time;
import java.text.DateFormat;
import java.util.Calendar;

import org.apache.log4j.Logger;

import edu.washington.cac.calendar.MyCalendar;
import edu.washington.cac.calendar.data.CaldataException;

/** A wrapper around MyCalendar objects that is also used to generate
    form elements

   @author   Mike Douglass douglm@rpi.edu
   @author   Greg Barnes
 */
public class TimeDateComponents implements Serializable {
  /** Label that indicates no time is specified */
  private static final String NO_TIME_LABEL = "None";
  /** Internal value to show no time is specified */
  private static final String NO_TIME_VALUE = "-1";

  // arrays of values and labels for dropdown menus for various units of time

  /** default labels for am and pm */
  // XXX: Should localize
  private static final String[] DEFAULT_AMPM_LABELS = {"am", "pm"};

  /** default labels for the dates in a month */
  private static String[] defaultDayLabels = 
      new String[maximumValues(Calendar.DAY_OF_MONTH)];
  /** default internal values for the dates in a month */
  private static String[] defaultDayVals =
      new String[maximumValues(Calendar.DAY_OF_MONTH)];

  /** default labels for the months of the year */
  private static String[] defaultMonthLabels =
      new String[maximumValues(Calendar.MONTH)];
  /** default internal values for the months of the year */
  private static String[] defaultMonthVals =
      new String[maximumValues(Calendar.MONTH)];

  // The hour arrays have an extra member to indicate no specified time 
  /** default labels for the hours of the day */
  private static String[] defaultHourLabels =
      new String[maximumValues(Calendar.HOUR) + 1];
  /** default internal values for the hours of the day */
  private static String[] defaultHourVals =
      new String[maximumValues(Calendar.HOUR) + 1];
  /** default labels for the hours of the day (24-hour clock) */
  private static String[] defaultHour24Labels =
      new String[maximumValues(Calendar.HOUR_OF_DAY) + 1];
  /** default internal values for the hours of the day (24-hour clock) */
  private static String[] defaultHour24Vals =
      new String[maximumValues(Calendar.HOUR_OF_DAY) + 1];

  /** default labels for the minutes of the hour */
  private static String[] defaultMinuteLabels =
      new String[maximumValues(Calendar.MINUTE)];
  /** default internal values for the minutes of the hour */
  private static String[] defaultMinuteVals =
      new String[maximumValues(Calendar.MINUTE)];

  /** 
    Get the maximum number of distinct values for an appropriate unit of time
    @param unit The unit of time.  Should be one of the constants in 
       <code>java.util.Calendar</code>, such as <code>DAY_OF_MONTH</code>
    @return the maximum number of distinct values for that unit.  E.g., for
      <code>DAY_OF_MONTH</code> and a Gregorian calendar, 31
   */
  private static int maximumValues(int unit) {
    return Calendar.getInstance().getMaximum(unit) - 
           Calendar.getInstance().getMinimum(unit) + 1;
  }

  /** Initialize the arrays of time labels and values */
  static {
    /** For convenience, Get localized version of a calendar */
    //XXX some of this needs to be localized still
    MyCalendar mc = new MyCalendar();

    for (int i=0, dateOfMonth = mc.getMinimum(Calendar.DAY_OF_MONTH); 
         i < defaultDayLabels.length; 
	 i++, dateOfMonth++) 
    {
      defaultDayLabels[i] = dateOfMonth + "";
      //XXX assuming max number of days in months is two-digits
      defaultDayVals[i] = mc.twoDigit(dateOfMonth);
    }

    MyCalendar currentMonth = 
        new MyCalendar(new MyCalendar().firstMonthOfThisYear()); 

    for (int i=0;
         i < defaultMonthLabels.length; 
         i++, currentMonth = currentMonth.addTime(Calendar.MONTH, 1)) 
    {
      // this gives abbreviated form of month name
      defaultMonthLabels[i] = currentMonth.getComponent(DateFormat.MONTH_FIELD,
                                                        DateFormat.MEDIUM);
      //XXX assuming max number of months is two-digits
      defaultMonthVals[i] = currentMonth.twoDigitMonth();
    }

    defaultHourLabels[0] = NO_TIME_LABEL;
    defaultHourVals[0] = NO_TIME_VALUE;

    /* Calendar.HOUR is 0 for 12 o'clock.  Skip 0, then add it to the end 
       labeled as 12, but with value 0 */
    defaultHourLabels[defaultHourLabels.length - 1] = 
        (defaultHourLabels.length-1) + "";
    defaultHourVals[defaultHourLabels.length - 1] = mc.twoDigit(0);

    for (int i=1, hour = mc.getMinimum(Calendar.HOUR) + 1;
         i < defaultHourLabels.length - 1; 
	 i++, hour++)
    {
      defaultHourLabels[i] = hour + "";
      //XXX assuming max hour is two digits
      defaultHourVals[i] = mc.twoDigit(hour);
    }


    defaultHour24Labels[0] = NO_TIME_LABEL;
    defaultHour24Vals[0] = NO_TIME_VALUE;

    for (int i=1, hourOfDay = mc.getMinimum(Calendar.HOUR_OF_DAY);
         i < defaultHour24Labels.length; 
	 i++, hourOfDay++) 
    {
      defaultHour24Labels[i] = mc.twoDigit(hourOfDay);
      //XXX assuming max hour of day is two digits
      defaultHour24Vals[i] = mc.twoDigit(hourOfDay);
    }

    for (int i=0, minute = mc.getMinimum(Calendar.MINUTE); 
         i < defaultMinuteLabels.length; 
	 i++, minute++) 
    {
      defaultMinuteLabels[i] = mc.twoDigit(minute);
      //XXX assuming max number of days in months is two-digits
      defaultMinuteVals[i] = mc.twoDigit(minute);
    }
  }

  private boolean debug;

  private transient Logger log;

  /** Holds time and date information */
  private MyCalendar mc;

  private String[] dayLabels;
  private String[] dayVals;
  private String[] monthLabels;
  private String[] monthVals;

  private String[] hourLabels;
  private String[] hourVals;

  /* We populate minuteLabels and minuteVals with the appropriate increments. */
  private String[] minuteLabels;
  private String[] minuteVals;

  /** Only accept minute values that are multiples of this */
  private int minuteIncrement;

  private String[] ampmLabels;

  /** Are we on the 24-hour clock? */
  private boolean hour24;

  /** current value of am/pm */
  private boolean isAm = true;

  public static class TimeDateException extends Exception {
    public TimeDateException(String msg) {
      super(msg);
    }
  }

  /** Set up instance of this class using default values.
   *
   * @param minuteIncrement  increment for minutes, <= 1 is all,
   *                         5 is every 5 minutes etc.
   * @param hour24        true if we ignore am/pm and use 24hr clock
   */
  public TimeDateComponents(int minuteIncrement,
                            boolean hour24,
                            boolean debug) throws TimeDateException {
    String[] hrl;
    String[] hrv;

    if (debug) {
        getLogger().debug("Init TimeDateComponents with hour24= " + hour24);
    }

    init(defaultDayLabels,
         defaultDayVals,
         defaultMonthLabels,
         defaultMonthVals,
         hour24 ? defaultHour24Labels : defaultHourLabels,
         hour24 ? defaultHour24Vals : defaultHourVals,
         defaultMinuteLabels,
         defaultMinuteVals,
         minuteIncrement,
         DEFAULT_AMPM_LABELS,
         hour24,
         debug);
  }

  /* See XXX note below before getter/setter methods if you want to
     make this constructor non-private.
   */
  /** Set up instance of this class.
   *
   * @param dayLabels     external array of labels used for timedate
   * @param dayVals       internal array of values used for timedate
   * @param monthLabels   external array of labels used for timedate
   * @param monthVals     internal array of values used for timedate
   * @param hourLabels    external array of labels used for timedate
   * @param hourVals      internal array of values used for timedate
   * @param minuteLabels  external array of labels used for timedate
   * @param minuteVals    internal array of values used for timedate
   * @param minuteIncrement  increment for minutes, 0, 1 is all,
   *                      5 is every 5  inutes etc.
   * @param ampmLabels    internal array of values used for timedate
   * @param hour24        true if we ignore am/pm and use 24hr clock
   */
  private TimeDateComponents(String[] dayLabels,
                             String[] dayVals,
                             String[] monthLabels,
                             String[] monthVals,
                             String[] hourLabels,
                             String[] hourVals,
                             String[] minuteLabels,
                             String[] minuteVals,
                             int minuteIncrement,
                             String[] ampmLabels,
                             boolean hour24,
                             boolean debug) throws TimeDateException {
    init(dayLabels,
         dayVals,
         monthLabels,
         monthVals,
         hourLabels,
         hourVals,
         minuteLabels,
         minuteVals,
         minuteIncrement,
         ampmLabels,
         hour24,
         debug);
  }

  /** Set up instance of this class.
   *
   * @param dayLabels     external array of labels used for timedate
   * @param dayVals       internal array of values used for timedate
   * @param monthLabels   external array of labels used for timedate
   * @param monthVals     internal array of values used for timedate
   * @param hourLabels    external array of labels used for timedate
   * @param hourVals      internal array of values used for timedate
   * @param minuteLabels  external array of labels used for timedate
   * @param minuteVals    internal array of values used for timedate
   * @param minuteIncrement  increment for minutes, <= 1 is all,
   *                      5 is every 5 minutes etc.
   * @param ampmLabels    internal array of values used for timedate
   * @param hour24        true if we ignore am/pm and use 24hr clock
   */
  private void init(String[] dayLabels,
                    String[] dayVals,
                    String[] monthLabels,
                    String[] monthVals,
                    String[] hourLabels,
                    String[] hourVals,
                    String[] minuteLabels,
                    String[] minuteVals,
                    int minuteIncrement,
                    String[] ampmLabels,
                    boolean hour24,
                    boolean debug) throws TimeDateException {
    this.dayLabels = dayLabels;
    this.dayVals = dayVals;
    this.monthLabels = monthLabels;
    this.monthVals = monthVals;
    this.hourLabels = hourLabels;
    this.hourVals = hourVals;

    setMinutes(minuteLabels, minuteVals, minuteIncrement);

    this.ampmLabels = ampmLabels;
    this.hour24 = hour24;
    this.debug = debug;
  }

  /**
    Set the minutes arrays to a subset of the values in two given arrays
    @param minuteLabels Array from which to draw the labels
    @param minuteVals Array from which to draw the values
    @param minuteIncrement Choose the 0th entry in each array, and every
       minuteIncrement'th one after that
    @exception TimeDateException If either of the arrays given is not
       the proper length
   */
  public void setMinutes(String[] minuteLabels,
                         String[] minuteVals,
                         int minuteIncrement) throws TimeDateException {
    this.minuteIncrement = (minuteIncrement <= 1) ? 1: minuteIncrement;

    if ((minuteLabels.length != defaultMinuteLabels.length) ||
        (minuteVals.length != defaultMinuteLabels.length)) {
      throw new TimeDateException("minute values/labels must have " + 
          defaultMinuteLabels.length +  " entries");
    }

    if (this.minuteIncrement == 1) {
      this.minuteLabels = minuteLabels;
      this.minuteVals = minuteVals;
    } else {
      int sz = defaultMinuteLabels.length / this.minuteIncrement;

      this.minuteLabels = new String[sz];
      this.minuteVals = new String[sz];

      for (int i=0, j=0; 
           j < minuteLabels.length; 
           i++, j += this.minuteIncrement) 
      {
        this.minuteLabels[i] = minuteLabels[j];
        this.minuteVals[i] = minuteVals[j];
      }
    }
  }

  public String[] getDayLabels() {
    return this.dayLabels;
  }

  public String[] getDayVals() {
    return this.dayVals;
  }

  public String[] getMonthLabels() {
    return this.monthLabels;
  }

  public String[] getMonthVals() {
    return this.monthVals;
  }

  public String[] getHourLabels() {
    return this.hourLabels;
  }

  public String[] getHourVals() {
    return this.hourVals;
  }

  public String[] getMinuteLabels() {
    return this.minuteLabels;
  }

  public String[] getMinuteVals() {
    return this.minuteVals;
  }

  public String[] getAmpmLabels() {
    return this.ampmLabels;
  }

  /** Sets this object's current date to now.  The time is not set.
   */
  public void setNow() {
    this.mc = new MyCalendar(new java.util.Date(), null);
  }

  /** Sets this object's current time from the given long value.
   *
   * @param val    the new time in UTC milliseconds from the epoch.
   */
  public void setTimeInMillis(long val) {
    Calendar cal = Calendar.getInstance();
    cal.setTimeInMillis(val);
    this.mc = new MyCalendar(cal);
    this.isAm = (this.mc.get(Calendar.AM_PM) == Calendar.AM);
  }

  /** Set this object's date and, optionally, time
   *
   * @param date date to use (may not be null)
   * @param time time to use (may be null)
   */
  public void setDateTime(Date date, Time time) {
    this.mc = new MyCalendar(date, time);
    this.isAm = (this.mc.get(Calendar.AM_PM) == Calendar.AM);
  }

  /** Get an SQL date object representing the date of the event (only)
   *
   * @return Date object representing the date 
   */
  public Date getDate() {
    return MyCalendar.normalizedDate(new Date(this.mc.getTime().getTime()));
  }

  /** Get an SQL Time object representing the time of the event (only)
   *
   * @return The time, or null if the object does not have an associated time
   */
  public Time getTime() {
    if (this.mc.hasTime()) {
      return MyCalendar.normalizedTime(new Time(this.mc.getTime().getTime()));
    } else {
      return null;
    }
  }

  /** Get a normalized version of the date 
   *
   * @return String  date as YYYYMMDD
   */
  public String getDateString() {
    return this.mc.getDateDigits();
  }

  /** Set the year
   *
   * @param val   String 4 digit year yyyy
   */
  public void setYear(String val) throws TimeDateException {
    try {
      this.mc.set(Calendar.YEAR, Integer.parseInt(val));
    } catch (NumberFormatException e) {
      throw new TimeDateException("Invalid year " + val);
    }
  }

  public String getYear() {
    return this.mc.fourDigitYear();
  }

  /* XXX 
     In the get methods below, we try to translate real values to the 
     'values' in the arrays defined above (e.g., dayVals).  
     We should also translate array values into real vals in the set methods.
     We don't.  The only reason this works is because currently we only
     use the default arrays, which map things in the obvious way (1 <-> 1).
     If one allows non-default arrays to be used (e.g., with Val arrays
     that contain non-numbers, the set methods would need to be changed.

     Note that the years are exempt here, as they are not selected by
     dropdown menus
   */
  /** Set the month number
   *
   * @param val   String 2 digit month number 01-xx
   */
  public void setMonth(String val) throws TimeDateException {
    try {
      this.mc.set(Calendar.MONTH, Integer.parseInt(val) - 1);
    } catch (NumberFormatException e) {
      throw new TimeDateException("Invalid month " + val);
    }
  }

  public String getMonth() {
    // Calendar.MONTH returns 0-11
    return this.monthVals[this.mc.get(Calendar.MONTH)];
  }

  /** Set the day number
   *
   * @param val   String 2 digit day number 01-xx
   */
  public void setDay(String val) throws TimeDateException {
    try {
      this.mc.set(Calendar.DAY_OF_MONTH, Integer.parseInt(val));
    } catch (NumberFormatException e) {
      throw new TimeDateException("Invalid day of month " + val);
    }
  }

  public String getDay() {
    // Calendar.DAY_OF_MONTH returns 1-31
    return this.dayVals[this.mc.get(Calendar.DAY_OF_MONTH) - 1];
  }

  /**
    Set the hour in the underlying calendar
    @param hour Hour of the day (0-23)
   */
  private void setHourOfDay(int hour) {
    this.mc.set(Calendar.HOUR_OF_DAY, hour);
  }

  /**
    Set the hour in the underlying calendar
    @param hour Hour of the day (0-11)
   */
  /* setting hour and am/pm separately doesn't seem to work, so we'll
     always set on a 24-hour clock
   */
  private void setHour(int hour) {
    setHourOfDay(hour + (this.isAm ? 0 : 12));
  }

  /** Set the hour
   *
   * @param val   String hour
   */
  /* Note that setting the hour can change a calendar from having
     no associated time to having an associated time, or vice versa.
     So we always recreate the associated MyCalendar to avoid
     retaining the 'time/no time' state of the previous incarnation.
   */
  public void setHour(String val) throws TimeDateException {
    if (val.equals(NO_TIME_VALUE)) {
      // create a calendar with no time attached
      try {
        this.mc = new MyCalendar(this.mc.getDateDigits());
      } catch (CaldataException e) {  // shouldn't happen
        e.printStackTrace();
        throw new IllegalStateException(
	    "MyCalendar.getDateDigits() returned bad string");
      }

      return;
    } else {
      try {
        this.mc = new MyCalendar(this.mc.getCalendar());

        if (hour24) {
	  setHourOfDay(Integer.parseInt(val));
        } else {
	  setHour(Integer.parseInt(val));
        }
      } catch (NumberFormatException e) {
        throw new TimeDateException("Invalid hour " + val);
      }
    }
  }

  public String getHour() {
    if (!this.mc.hasTime()) {
      return NO_TIME_VALUE;
    }
   
    /* Calendar.HOUR_OF_DAY returns 0-23; must be adjusted up due NO_TIME_VAL
       Calendar.HOUR returns 0-11; must adjust due to funny 12/0 problem */
    if (this.hour24) {
      return this.hourVals[this.mc.get(Calendar.HOUR_OF_DAY) + 1];
    } else if (this.mc.get(Calendar.HOUR) == 0) {
      return this.hourVals[this.hourVals.length - 1];
    } else {
      return this.hourVals[this.mc.get(Calendar.HOUR)];
    }
  }

  /** Set the minute. Will be rounded to minuteIncrement
   *
   * @param val   String minute
   */
  public void setMinute(String val) throws TimeDateException {
    this.mc.set(Calendar.MINUTE, validMinute(val));
  }

  public String getMinute() {
    if (!this.mc.hasTime()) {
      return this.minuteVals[0];	// dummy value
    }
   
    /* Calendar.MINUTE returns 0-59; adjusted to account for 
       minuteIncrement skipping */
    return this.minuteVals[this.mc.get(Calendar.MINUTE) / 
                           this.minuteIncrement];
  }

  /** Set the am/pm
   *
   * @param val   String am/pm
   */
  public void setAmpm(String val) {
    // am/pm is set with the hour
    this.isAm = val.equals(ampmLabels[0]);

    if (this.mc.hasTime()) {
      setHour(this.mc.get(Calendar.HOUR));
    }  // otherwise, wait until hour is set
  }

  public String getAmpm() {
    if (!this.mc.hasTime()) {
      return this.ampmLabels[0];	// dummy value
    }
   
    // Calendar.AM_PM returns 0-1
    return this.ampmLabels[this.isAm ? 0 : 1];
  }

  /** ===================================================================
   *                Private methods
   *  =================================================================== */

  private int validMinute(String val) throws TimeDateException {
    int imin;

    if ((val == null) || (val.equals(NO_TIME_VALUE))) {
      return 0;
    } else {
      try {
        imin = Integer.parseInt(val);
      } catch (NumberFormatException e) {
        throw new TimeDateException("Invalid minutes: " + val);
      }
    }

    if (this.minuteIncrement > 1) {
      return ((imin + 1) / this.minuteIncrement) * this.minuteIncrement;
    } else {
      return imin;
    }
  }

  /** Get a logger for messages
   */
  private Logger getLogger() {
    if (log == null) {
      log = Logger.getLogger(this.getClass());
    }

    return log;
  }

  public static void main(String[] args) {
    try {
      TimeDateComponents tdc = new TimeDateComponents(5, true, true);

      tdc.setYear("1953");
      tdc.setMonth("3");
      tdc.setDay("15");
      tdc.setHour("7");
      tdc.setMinute("23");
      tdc.setAmpm("pm");

      showTdc(tdc);

      tdc.setYear("53");

      showTdc(tdc);

      tdc = new TimeDateComponents(5, true, true);
      tdc.setNow();
      showTdc(tdc);
    } catch (Throwable t) {
      t.printStackTrace();
    }
  }

  /** For use with main for testing
   */
  private static void showTdc(TimeDateComponents tdc) {
    try {
      System.out.println("tdc.getDate().toString() says: " +
                         tdc.getDate().toString());
      System.out.println("tdc.getTime().toString() says: " +
                         tdc.getTime().toString());

      System.out.println("tdc.getDateString() says: " + tdc.getDateString());
    } catch (Throwable t) {
      t.printStackTrace();
    }
  }
}

