/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.cct.uwcal.calsvci;

import edu.washington.cac.calfacade.shared.CalFacadeException;
import edu.washington.cac.calfacade.shared.Calintf;
import edu.washington.cac.calfacade.shared.EventVO;
import edu.washington.cac.calfacade.shared.KeywordVO;
import edu.washington.cac.calfacade.shared.LocationVO;
import edu.washington.cac.calfacade.shared.MyCalendarVO;
import edu.washington.cac.calfacade.shared.SponsorVO;

import java.io.Serializable;

/** This is the service interface to the calendar suite. This will be
 * used by web applications and web services as well as other applications
 * which wish to integrate calendar actions into their interface.
 *
 * <p>By and large this acts as a stub interface between the caller and the
 * Calintf implementation. Some of the Calintf methods are implmented
 * differently, to perhaps add funcionality or to hide them.
 *
 * <p>Those modified methods are defined here.
 *
 * @author Mike Douglass       douglm@rpi.edu
 */
public interface CalSvcI extends Serializable, Calintf {
  /** Reserved keyword attribute names and values start with "cal"
   */
  public static final String catClassAttrName = "calcategoryclass";

  public static final String catClassAlerts = "calalerts";

  /** (Re)initialise the object for a particular use.
   *
   * @param initObj     object used to initialise properties, for example, a
   *                    servlet.
   * @param user        String user of the application, null for guest
   * @param rights      int access rights value as defined by
   *                    edu.washington.cac.calfacade.shared.UserAuth
   * @param publicAdmin boolean true if this is a public events admin app
   * @param debug       boolean true to turn on debugging trace
   * @return Calintf    object for interactions with back-end.
   */
  public void init(Object initObj,
                   String user,
                   int rights,
                   boolean publicAdmin,
                   boolean debug) throws CalFacadeException;

  /** (Re)initialise the object for a particular use.
   *
   * @param user        String user of the application, null for guest
   * @param rights      int access rights value as defined by
   *                    edu.washington.cac.calfacade.shared.UserAuth
   * @param publicAdmin boolean true if this is a public events admin app
   * @param debug       boolean true to turn on debugging trace
   * @return Calintf    object for interactions with back-end.
   */
  public void init(String user,
                   int rights,
                   boolean publicAdmin,
                   boolean debug) throws CalFacadeException;

  /** Must be called to initialise the new object.
   *
   * @param initObj     object used to initialise properties, for example, a
   *                    servlet.
   * @param user        String user of the application
   * @param rights      int access rights value as defined by
   *                    edu.washington.cac.calfacade.shared.UserAuth
   * @param isGuest     boolean true if user is guest
   * @param publicAdmin boolean true if this is a public events admin app
   * @param debug       boolean true to turn on debugging trace
   */
  public void init(Object initObj,
                   String user,
                   int rights,
                   boolean isGuest,
                   boolean publicAdmin,
                   boolean debug) throws CalFacadeException;

  /** ===================================================================
   *                   Keywords
   *  =================================================================== */

  /** Return public keywords excluding alerts.
   * This modifies th ebehaviour of the Calintf getPublicKeywords method.
   * (I don't think there's anything other than public keywords.
   *
   * @return keywordVO[]    array of keywords
   */
  public KeywordVO[] getPublicKeywords() throws CalFacadeException;

  /** Return public keywords including alerts.
   *
   * @return keywordVO[]    array of keywords
   */
  public KeywordVO[] getAllPublicKeywords() throws CalFacadeException;

  /** Return alert keywords only.
   *
   * @return keywordVO[]    array of alert keywords
   */
  public KeywordVO[] getAlertKeywords() throws CalFacadeException;

  /** Search the public keywords for a matching name
   *
   * @param k          KeywordVO object to match for
   * @return KeywordVO matching object or null if no match.
   */
  public KeywordVO findKeyword(KeywordVO k) throws CalFacadeException;

  /** Ensure a keyword exists. If it does returns the keyword id.
   * If not creates the keyword then returns the new id.
   *
   * @param keyword   KeywordVO value object. If this object has the id set,
   *                  we assume the check was made previously.
   * @return int      keywordid that matches or new keywordid
   */
  public int ensureKeywordExists(KeywordVO keyword) throws CalFacadeException;

  /** Return an array of the given keywords
   *
   * @param keys          Keys of the desired keywords
   * @return KeywordVO[] Alphabetically sorted array
   */
  public KeywordVO[] getPublicKeywords(int[] keys)
      throws CalFacadeException;

  /** ===================================================================
   *                   Keyword attributes
   *  =================================================================== */

  /** Set keyword alert on/off
   *
   * @param id       keyword id
   * @param val      boolean turn it on/off
   */
  public void setKeyAlert(int id, boolean val) throws CalFacadeException;

  /** Is given keyword an alert?
   *
   * @param id       keyword id
   * @return boolean true if this is an alert keyword
   */
  public boolean getKeyAlert(int id) throws CalFacadeException;

  /** ===================================================================
   *                   Public Locations
   *  =================================================================== */

  /** Search the public locations for a matching name, phone, email and link
   *
   * @param l          LocationVO object to match for
   * @return LocationVO matching object or null if no match.
   */
  public LocationVO findPublicLocation(LocationVO l) throws CalFacadeException;

  /** Ensure a location exists. If it does returns the location id.
   * If not creates the location then returns the new id.
   *
   * @param location   LocationVO value object. If this object has the id set,
   *                  we assume the check was made previously.
   * @return int      locationid that matches or new locationid
   */
  public int ensurePublicLocationExists(LocationVO location)
      throws CalFacadeException;

  /** Return an array of the given locations
   *
   * @param keys          Keys of the desired locations
   * @return LocationVO[] Alphabetically sorted array
   */
  public LocationVO[] getPublicLocations(int[] keys)
      throws CalFacadeException;

  /** Delete a location with the given id
   *
   * @param id       int id of the location
   * @return boolean false if it didn't exist, true if it was deleted.
   */
  public boolean deletePublicLocation(int id) throws CalFacadeException;

  /** ===================================================================
   *                   Sponsors
   *  =================================================================== */

  /** Return an array of SponsorVO objects
   *
   * @return SponsorVO[]    array of sponsors
   */
  public SponsorVO[] getSponsors() throws CalFacadeException;

  /** Search the public sponsors for a matching name, phone, email and link
   *
   * @param s          SponsorVO object to match for
   * @return SponsorVO matching object or null if no match.
   */
  public SponsorVO findSponsor(SponsorVO s) throws CalFacadeException;

  /** Ensure a sponsor exists. If it does returns the sponsor id.
   * If not creates the sponsor then returns the new id.
   *
   * @param sponsor   SponsorVO value object. If this object has the id set,
   *                  we assume the check was made previously.
   * @return int      sponsorid that matches or new sponsorid
   */
  public int ensureSponsorExists(SponsorVO sponsor) throws CalFacadeException;

  /** Return an array of the given sponsors
   *
   * @param keys          Keys of the desired sponsors
   * @return SponsorVO[] Alphabetically sorted array
   */
  public SponsorVO[] getPublicSponsors(int[] keys)
      throws CalFacadeException;

  /** ===================================================================
   *                   Events
   *  =================================================================== */

  /** Return public events for the current user which match the supplied
   * alert flag
   *
   * @param alerts      boolean true if we want alerts false for exclude alerts
   * @return EventVO[]  populated event value objects
   */
  public EventVO[] getPublicEvents(boolean alerts)
      throws CalFacadeException;

  /** Return active public events for the current user which match the
   * supplied alert flag
   *
   * @param alerts      boolean true if we want alerts false for exclude alerts
   * @return EventVO[]  populated event value objects
   */
  public EventVO[] getActivePublicEvents(boolean alerts)
      throws CalFacadeException;

  /* * Return the alerts for the given day as an array of value objects
   *
   * @param   date    MyCalendarVO object defining day
   * @return  EventVO[]  one days alerts or null for no events.
   * /
  public EventVO[] getDaysAlerts(MyCalendarVO date) throws CalFacadeException;*/

  /** Add an event and ensure location and sponsor exist. Some calendar
   * implementors choose to allow the dynamic creation of locations and
   * sponsors. For each of those, if we have an id, then the object represents
   * a preexisting database item.
   *
   * <p>Otherwise the client has provided information which will be used to
   * locate an already existing location or sponsor. Failing that we use the
   * information to create a new entry.
   *
   * @param loc        LocationVO object
   * @param sp         SponsorVO object
   * @param event      EventVO object to be added
   * @return EventVO   Update object with embedded location and sponsor.
   */
  public EventVO addPublicEvent(LocationVO loc, SponsorVO sp,
                                EventVO event) throws CalFacadeException;

  /** Update an event and ensure location and sponsor exist. Some calendar
   * implementors choose to allow the dynamic creation of locations and
   * sponsors. For each of those, if we have an id, then the object represents
   * a preexisting database item.
   *
   * <p>Otherwise the client has provided information which will be used to
   * locate an already existing location or sponsor. Failing that we use the
   * information to create a new entry.
   *
   * <p>If the location or sponsor changes, the old ones may no longer be
   * referenced by any events. If flagged this method will delete the old
   * unused entries.
   *
   * @param loc           LocationVO object
   * @param deleteOldLoc  true if we should delete old unused location
   * @param sp            SponsorVO object
   * @param deleteOldSp   true if we should delete old unused sponsor
   * @param oldKeywordId  Id of old event category. Same as newKeywordId
   *                      if unchanged.
   * @param newKeywordId  int id to use
   * @param event         EventVO object to be added
   * @return EventVO      Updated object with embedded location and sponsor.
   */
  public EventVO updatePublicEvent(LocationVO loc, boolean deleteOldLoc,
                                   SponsorVO sp, boolean deleteOldSp,
                                   int oldKeywordId,
                                   int newKeywordId,
                                   EventVO event) throws CalFacadeException;

  /** Delete an event and optionally delete the old location and/or sponsor
   * entries if now unused.
   *
   * @param deleteOldLoc  true if we should delete old unused location
   * @param deleteOldSp  true if we should delete old unused sponsor
   * @param event      EventVO object to be added
   */
  public void deletePublicEvent(boolean deleteOldLoc,
                                boolean deleteOldSp,
                                EventVO event) throws CalFacadeException;
}

