/* **********************************************************************
    Copyright 2003 Rensselaer Polytechnic Institute.

    All worldwide rights reserved. A license to use, copy, modify and
    distribute this software for noncommercial research purposes only is
    hereby granted, provided that this copyright notice and accompanying
    disclaimer is not modified or removed from the software.

    DISCLAIMER: The software is distributed" AS IS" without any express or
    implied warranty, including but not limited to, any implied warranties
    of merchantability or fitness for a particular purpose or any warrant)'
    of non-infringement of any current or pending patent rights. The authors
    of the software make no representations about the suitability of this
    software for any particular purpose. The entire risk as to the quality
    and performance of the software is with the user. Should the software
    prove defective, the user assumes the cost of all necessary servicing,
    repair or correction. In particular, neither Rensselaer Polytechnic
    Institute, nor the authors of the software are liable for any indirect,
    special, consequential, or incidental damages related to the software,
    to the maximum extent the law permits.
*/

package edu.rpi.cct.uwcal.calsvci;

import edu.washington.cac.calfacade.shared.AdminGroups;
import edu.washington.cac.calfacade.shared.CalFacadeException;
import edu.washington.cac.calfacade.shared.Calintf;
import edu.washington.cac.calfacade.shared.CalintfFactory;
import edu.washington.cac.calfacade.shared.EventVO;
import edu.washington.cac.calfacade.shared.KeywordAttrsVO;
import edu.washington.cac.calfacade.shared.KeywordVO;
import edu.washington.cac.calfacade.shared.LocationVO;
import edu.washington.cac.calfacade.shared.MyCalendarVO;
import edu.washington.cac.calfacade.shared.SponsorVO;
import edu.washington.cac.calfacade.shared.UserAuth;
import edu.washington.cac.calfacade.shared.UserVO;

import java.sql.Timestamp;
import java.util.Arrays;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.Vector;

/** This is an implementation of the admin service interface to the calendar
 * suite.
 *
 * @author Mike Douglass       douglm@rpi.edu
 */
public class CalSvc implements CalSvcI {
  private Object initObj;
  private String user;
  private int rights;
  private boolean isGuest;
  private boolean publicAdmin;
  private boolean debug;

  private transient Calintf cali;

  /** (Re)initialise the object for a particular use.
   *
   * @param initObj     object used to initialise properties, for example, a
   *                    servlet.
   * @param user        String user of the application, null for guest
   * @param rights      int access rights value as defined by
   *                    edu.washington.cac.calfacade.shared.UserAuth
   * @param publicAdmin boolean true if this is a public events admin app
   * @param debug       boolean true to turn on debugging trace
   * @return Calintf    object for interactions with back-end.
   */
  public void init(Object initObj,
                   String user,
                   int rights,
                   boolean publicAdmin,
                   boolean debug) throws CalFacadeException {
    init(initObj, user, rights, (user == null), publicAdmin, debug);
  }


  /** (Re)initialise the object for a particular use.
   *
   * @param user        String user of the application, null for guest
   * @param rights      int access rights value as defined by
   *                    edu.washington.cac.calfacade.shared.UserAuth
   * @param publicAdmin boolean true if this is a public events admin app
   * @param debug       boolean true to turn on debugging trace
   * @return Calintf    object for interactions with back-end.
   */
  public void init(String user,
                   int rights,
                   boolean publicAdmin,
                   boolean debug) throws CalFacadeException {
    init(user, rights, (user == null), publicAdmin, debug);
  }

  /** Must be called to initialise the new object.
   *
   * @param initObj     object used to initialise properties, for example, a
   *                    servlet.
   * @param user        String user of the application
   * @param rights      int access rights value as defined by
   *                    edu.washington.cac.calfacade.shared.UserAuth
   * @param isGuest     boolean true if user is guest
   * @param publicAdmin boolean true if this is a public events admin app
   * @param debug       boolean true to turn on debugging trace
   */
  public void init(Object initObj,
                   String user,
                   int rights,
                   boolean isGuest,
                   boolean publicAdmin,
                   boolean debug) throws CalFacadeException {
    this.initObj = initObj;
    this.user = user;
    this.rights = rights;
    this.isGuest = isGuest;
    this.publicAdmin =publicAdmin;
    this.debug = debug;
  }

  /** Must be called to initialise the new object.
   *
   * @param user        String user of the application
   * @param rights      int access rights value as defined by
   *                    edu.washington.cac.calfacade.shared.UserAuth
   * @param isGuest     boolean true if user is guest
   * @param publicAdmin boolean true if this is a public events admin app
   * @param debug       boolean true to turn on debugging trace
   */
  public void init(String user,
                   int rights,
                   boolean isGuest,
                   boolean publicAdmin,
                   boolean debug) throws CalFacadeException {
    this.user = user;
    this.rights = rights;
    this.isGuest = isGuest;
    this.publicAdmin =publicAdmin;
    this.debug = debug;
  }

  /** Set this up for the given user. Must be called before any other action
   *
   * @param val   String user id - null for guest
   * @param rights      int access rights value as defined by
   *                    edu.washington.cac.calfacade.shared.UserAuth
   * @param publicAdmin boolean true if this is a public events admin app
   */
  public void setUser(String val, int rights,
                      boolean publicAdmin) throws CalFacadeException {
    this.user = user;
    this.rights = rights;
    this.isGuest = user == null;
    this.publicAdmin =publicAdmin;
    getCal().setUser(val, rights, publicAdmin);
  }

  /** Returns a value object representing the current user.
   *
   * @return UserVO       representing the current user
   */
  public UserVO getUserVO() throws CalFacadeException {
    return getCal().getUserVO();
  }

  /** Get a UserAuth object which allows the application to determine what
   * special rights the user has.
   *
   * <p>Note that the returned object may require a one time initialisation
   * call to set it up correctly.
   * @see edu.washington.cac.calfacade.shared.UserAuth#initialise(String, int)
   * @see edu.washington.cac.calfacade.shared.UserAuth#initialise(String, Object)
   *
   * @return UserAuth    implementation.
   */
  public UserAuth getUserAuth() throws CalFacadeException {
    return getCal().getUserAuth();
  }

  /** Get an AdminGroups object which allows the application to handle
   * administrative groups.
   *
   * @return AdminGroups    implementation.
   */
  public AdminGroups getAdminGroups() throws CalFacadeException {
    return getCal().getAdminGroups();
  }

  /** Refresh the users events cache
   */
  public void refreshEvents() throws CalFacadeException {
    getCal().refreshEvents();
  }

  /** Update lastmod timestamp.
   *
   * @exception CalFacadeException If there's a database access problem
   */
  public void touch() throws CalFacadeException {
    getCal().touch();
  }

  /** Get the value of the current public events timestamp.
   *
   * @return long       current timestamp value
   */
  public long getPublicLastmod() throws CalFacadeException {
    return getCal().getPublicLastmod();
  }

  /** ===================================================================
   *                   Calendars and search
   *  =================================================================== */

  /** Return a list of calendar we can present to the client as a list of
   * available calendars.
   *
   * @return SortedSet   of CalendarVO
   */
  public SortedSet getCalendars() throws CalFacadeException {
    return getCal().getCalendars();
  }

  /** Set a search filter using the suppplied search string
   *
   * @param val    String search parameters
   */
  public void setSearch(String val) throws CalFacadeException {
    getCal().setSearch(val);
  }

  /** Return the current search string
   *
   * @return  String     search parameters
   */
  public String getSearch() throws CalFacadeException {
    return getCal().getSearch();
  }

  /** Set the calendar we are interested in. This is represented by the name
   * of a calendar.
   *
   * @param  val     int id of calendar - 0 for none
   * @return boolean true for OK, false for unknown calendar
   */
  public boolean setCalendar(int val) throws CalFacadeException {
    return getCal().setCalendar(val);
  }

  /** Get the name of the current calendar we are displaying
   *
   * @return String    name of calendar or null for all events
   */
  public String getCalendarTitle() throws CalFacadeException {
    return getCal().getCalendarTitle();
  }

  /** This method is called to add a calendar. No consistency checks are
   * carried out at the moment.
   *
   * @param val  String values for db
   */
  public void addCalendarDef(String val) throws CalFacadeException {
    getCal().addCalendarDef(val);
    getCal().touch();
  }

  /** Set a filter which returns events starting after a given Timestamp.
   *
   * @param val   Timestamp object.
   */
  public void setLastmodGECalendar(Timestamp val) throws CalFacadeException {
    getCal().setLastmodGECalendar(val);
  }

  /** Set a filter which returns events starting after a given date of the
   * form yyyymmdd.
   *
   * @param date   String of form yyyymmdd.
   */
  public void setLastmodGEDateCalendar(String date) throws CalFacadeException {
    getCal().setLastmodGEDateCalendar(date);
  }

  /** Set a filter which returns events starting after a given date and time
   * of the form yyyymmddhhmm.
   *
   * @param dateTime   String of form yyyymmddhhmm.
   */
  public void setLastmodGEDateTimeCalendar(String dateTime)
      throws CalFacadeException {
    getCal().setLastmodGEDateTimeCalendar(dateTime);
  }

  /** ===================================================================
   *                   Subscriptions
   *  =================================================================== */

  /** Ensure that the users subscriptions match a set of calendars
   *
   * @param crs      TreeSet of Integer calendar ids
   * @return int[]   Number of calendars subscribed to, and number of
   *                 calendars unsubscribed to
   */
  public int[] fixSubscriptions(TreeSet crs) throws CalFacadeException {
    return getCal().fixSubscriptions(crs);
  }

  /** Determine if user is subscribed to a calendar
   *
   * @param id       int calendar id
   * @return boolean true if subscribed to calendars
   */
  public boolean getSubscribed(int id) throws CalFacadeException {
    return getCal().getSubscribed(id);
  }

  /** ===================================================================
   *                   Keywords
   *  =================================================================== */

  /** Return all keywords for this user
   *
   * @return KeywordVO[]  array of keywords value objects
   */
  public KeywordVO[] getKeywords() throws CalFacadeException {
    return getCal().getKeywords();
  }

  /** Add a Keyword to the database. The id will be set in the parameter
   * object.
   *
   * @param val   KeywordVO object to be added
   * @return int  new sponsor id
   */
  public int addKeyword(KeywordVO val) throws CalFacadeException {
    int id = getCal().addKeyword(val);
    getCal().touch();

    return id;
  }

  /** Return a keyword with the given id
   *
   * @param id     int id of the keyword
   * @return KeywordVo object representing the keyword in question
   *                     null if it doesn't exist.
   */
  public KeywordVO getKeyword(int id) throws CalFacadeException {
    return getCal().getKeyword(id);
  }

  /** Check to see if a Keyword is referenced.
   *
   * @param id       int id of the Keyword
   * @return boolean true if the Keyword is referenced somewhere
   */
  public boolean checkKeywordRefs(int id) throws CalFacadeException {
    return getCal().checkKeywordRefs(id);
  }

  /** ===================================================================
   *                   Keywords
   *  =================================================================== */

  /** Return public keywords excluding alerts.
   * (I don't think there's anything other than public keywords.
   *
   * @return keywordVO[]    array of keywords
   */
  public KeywordVO[] getPublicKeywords() throws CalFacadeException {
    KeywordVO[] ks = getCal().getPublicKeywords();
    if (ks == null) {
      return null;
    }

    Vector v = new Vector();

    for (int i = 0; i < ks.length; i++) {
      if (!getKeyAlert(ks[i].getId())) {
        v.addElement(ks[i]);
      }
    }

    return (KeywordVO[])v.toArray(new KeywordVO[v.size()]);
  }

  /** Return public keywords including alerts.
   *
   * @return keywordVO[]    array of keywords
   */
  public KeywordVO[] getAllPublicKeywords() throws CalFacadeException {
    return getCal().getPublicKeywords();
  }

  /** Return alert keywords only.
   *
   * @return keywordVO[]    array of alert keywords
   */
  public KeywordVO[] getAlertKeywords() throws CalFacadeException {
    KeywordVO[] ks = getCal().getPublicKeywords();
    if (ks == null) {
      return null;
    }

    Vector v = new Vector();

    for (int i = 0; i < ks.length; i++) {
      if (getKeyAlert(ks[i].getId())) {
        v.addElement(ks[i]);
      }
    }

    return (KeywordVO[])v.toArray(new KeywordVO[v.size()]);
  }

  /** Return given public keyword
   *
   * @return keywordVO    keyword or null if it doesn't exist
   */
  public KeywordVO getPublicKeyword(int id) throws CalFacadeException {
    return getCal().getPublicKeyword(id);
  }

  /** Replace a keyword in the database.
   *
   * @param val   KeywordVO object to be replaced
   */
  public void replaceKeyword(KeywordVO val) throws CalFacadeException {
    getCal().replaceKeyword(val);
    getCal().touch();
  }

  /** Delete a keyword with the given id
   *
   * @param id       int id of the keyword
   * @return boolean false if it didn't exist, true if it was deleted.
   */
  public boolean deleteKeyword(int id) throws CalFacadeException {
    if (getCal().deleteKeyword(id)) {
      getCal().deleteKeywordAttrs(id);
      getCal().touch();
      return true;
    }

    return false;
  }

  /** Search the public keywords for a matching name
   *
   * @param k          KeywordVO object to match for
   * @return KeywordVO matching object or null if no match.
   */
  public KeywordVO findKeyword(KeywordVO k) throws CalFacadeException {
    KeywordVO[] kk = getPublicKeywords();

    if (kk == null) {
      return null;
    }

    String word = k.getWord();

    for (int i = 0; i < kk.length; i++) {
      KeywordVO ki = kk[i];

      if ((word.equals(ki.getWord()))) {
        return ki;
      }
    }

    return null;
  }

  /** Ensure a keyword exists. If it does returns the keyword id.
   * If not creates the keyword then returns the new id.
   *
   * @param keyword   KeywordVO value object. If this object has the id set,
   *                  we assume the check was made previously.
   * @return int      keywordid that matches or new keywordid
   */
  public int ensureKeywordExists(KeywordVO keyword) throws CalFacadeException {
    if (keyword.getId() != 0) {
      // We've already checked it exists
      return keyword.getId();
    }

    // Assume a new keyword - but see if we can find it.
    KeywordVO k = findKeyword(keyword);

    if (k != null) {
      return k.getId();
    }

    // doesn't exist at this point, so we add it to db table
    keyword.setIsPublic(true);

    int id = getCal().addKeyword(keyword);

    keyword.setId(id);
    getCal().touch();

    return id;
  }

  /** Return an array of the given keywords
   *
   * @param keys          Keys of the desired keywords
   * @return KeywordVO[] Alphabetically sorted array
   */
  public KeywordVO[] getPublicKeywords(int[] keys)
  throws CalFacadeException {
    if (keys == null) {
      return null;
    }

    int sz = keys.length;
    KeywordVO[] ks = new KeywordVO[sz];

    for (int i = 0; i < sz; i++) {
      ks[i] = getCal().getPublicKeyword(keys[i]);
    }

    Arrays.sort(ks);

    return ks;
  }

  /** ===================================================================
   *                   Keyword attributes
   *  =================================================================== */

  /** Set keyword alert on/off
   *
   * @param id       keyword id
   * @param val      boolean turn it on/off
   */
  public void setKeyAlert(int id, boolean val) throws CalFacadeException {
    KeywordAttrsVO ka = getCal().getKeywordAttrs(id);

    boolean on = ka.isMatching(catClassAttrName, catClassAlerts);
    if (val && !on) {
      addKeywordAttr(id, catClassAttrName, catClassAlerts);
    } else if (!val && on) {
      removeKeywordAttr(id, catClassAttrName, catClassAlerts);
    }
    getCal().touch();
  }

  /** Is given keyword an alert?
   *
   * @param id       keyword id
   * @return boolean true if this is an alert keyword
   */
  public boolean getKeyAlert(int id) throws CalFacadeException {
    KeywordAttrsVO ka = getCal().getKeywordAttrs(id);

    return ka.isMatching(catClassAttrName, catClassAlerts);
  }

  /** Get the keyword attributes entry for a keyword id
   *
   * @param keywordid       int id of the keyword entry
   * @return KeywordAttrsVO keyword attributes
   */
  public KeywordAttrsVO getKeywordAttrs(int keywordid)
      throws CalFacadeException {
    return getCal().getKeywordAttrs(keywordid);
  }

  /** Delete keyword attrs entry
   *
   * @param id         int Id of the keyword attrs entry
   */
  public void deleteKeywordAttrs(int id) throws CalFacadeException {
    getCal().deleteKeywordAttrs(id);
    getCal().touch();
  }

  /** Add a keyword attribute name/value pair for a keyword
   *
   * @param keywordid       int id of the keyword entry
   * @param name            String attribute name
   * @param val             String attribute value
   */
  public void addKeywordAttr(int keywordid,
                             String name,
                             String val) throws CalFacadeException {
    getCal().addKeywordAttr(keywordid, name, val);
    getCal().touch();
  }

  /** Remove a keyword attribute name/value pair for a keyword
   *
   * @param keywordid       int id of the keyword entry
   * @param name            String attribute name
   * @param val             String attribute value
   */
  public void removeKeywordAttr(int keywordid,
                                String name,
                                String val) throws CalFacadeException {
    getCal().removeKeywordAttr(keywordid, name, val);
    getCal().touch();
  }

  /** ===================================================================
   *                   Locations
   *  =================================================================== */

  /** Return list of locations for the current user.
   *
   * @return LocationVO[]  locations for this user.
   */
  public LocationVO[] getLocations() throws CalFacadeException {
    return getCal().getLocations();
  }

  /** Add a location to the database. The id will be set in the parameter
   * object.
   *
   * @param val   LocationVO object to be added
   * @return int  new location id
   */
  public int addLocation(LocationVO val) throws CalFacadeException {
    int id = getCal().addLocation(val);
    getCal().touch();

    return id;
  }

  /** Replace a location in the database.
   *
   * @param val   LocationVO object to be replaced
   */
  public void replaceLocation(LocationVO val) throws CalFacadeException {
    getCal().replaceLocation(val);
    getCal().touch();
  }

  /** Return a location with the given id
   *
   * @param id     int id of the location
   * @return LocationVo object representing the location in question
   *                     null if it doesn't exist.
   */
  public LocationVO getLocation(int id) throws CalFacadeException {
    return getCal().getLocation(id);
  }

  /** Delete a location with the given id
   *
   * @param id       int id of the location
   * @return boolean false if it didn't exist, true if it was deleted.
   */
  public boolean deleteLocation(int id) throws CalFacadeException {
    if (id <= LocationVO.maxReservedId) {
      // claim it doesn't exist
      return false;
    }

    if (getCal().deleteLocation(id)) {
      getCal().touch();
      return true;
    }

    return false;
  }

  /** Check to see if a location is referenced.
   *
   * @param id       int id of the location
   * @return boolean true if the location is referenced somewhere
   */
  public boolean checkLocationRefs(int id) throws CalFacadeException {
    return getCal().checkLocationRefs(id);
  }

  /** ===================================================================
   *                   Public Locations
   *  =================================================================== */

  /** Return list of public locations.
   *
   * @return LocationVO[]  public locations.
   */
  public LocationVO[] getPublicLocations() throws CalFacadeException {
    return getCal().getPublicLocations();
  }

  /** Return public locations - optionally including reserved entries.
   *
   * @param includeReserved boolean true include reserved, false exclude
   * @return LocationVO[]  array of location value objects
   */
  public LocationVO[] getPublicLocations(boolean includeReserved)
      throws CalFacadeException {
    return getCal().getPublicLocations(includeReserved);
  }

  /** Return given public location
   *
   * @param id          int id of the location
   * @return LocationVo object representing the location in question
   *                     null if it doesn't exist.
   */
  public LocationVO getPublicLocation(int id) throws CalFacadeException {
    return getCal().getPublicLocation(id);
  }

  /** Search the public locations for a matching name, phone, email and link
   *
   * @param l          LocationVO object to match for
   * @return LocationVO matching object or null if no match.
   */
  public LocationVO findPublicLocation(LocationVO l) throws CalFacadeException {
    LocationVO[] ll = getPublicLocations();

    if (ll == null) {
      return null;
    }

    String address = l.getAddress();
    String subaddress = l.getSubaddress();
    String link = l.getLink();

    for (int i = 0; i < ll.length; i++) {
      LocationVO li = ll[i];

      if ((address.equals(li.getAddress())) &&
          checkField(subaddress, li.getSubaddress()) &&
          checkField(link, li.getLink())) {
        return li;
      }
    }

    return null;
  }

  /** Ensure a location exists. If it does returns the location id.
   * If not creates the location then returns the new id.
   *
   * @param location   LocationVO value object. If this object has the id set,
   *                  we assume the check was made previously.
   * @return int      locationid that matches or new locationid
   */
  public int ensurePublicLocationExists(LocationVO location)
      throws CalFacadeException {
    if (location.getId() != 0) {
      // We've already checked it exists
      return location.getId();
    }

    // Assume a new location
    LocationVO l = findPublicLocation(location);

    if (l != null) {
      return l.getId();
    }

    // doesn't exist at this point, so we add it to db table
    location.setIsPublic(true);

    int id = getCal().addLocation(location);

    location.setId(id);
    getCal().touch();

    return id;
  }

  /** Return an array of the given locations
   *
   * @param keys          Keys of the desired locations
   * @return LocationVO[] Alphabetically sorted array
   */
  public LocationVO[] getPublicLocations(int[] keys)
      throws CalFacadeException {
    if (keys == null) {
      return null;
    }

    int sz = keys.length;
    LocationVO[] locs = new LocationVO[sz];

    for (int i = 0; i < sz; i++) {
      locs[i] = getCal().getPublicLocation(keys[i]);
    }

    Arrays.sort(locs);

    return locs;
  }

  /** Delete a location with the given id
   *
   * @param id       int id of the location
   * @return boolean false if it didn't exist, true if it was deleted.
   */
  public boolean deletePublicLocation(int id) throws CalFacadeException {
    if (id <= LocationVO.maxReservedId) {
      // claim it doesn't exist
      return false;
    }

    if (getCal().deletePublicLocation(id)) {
      getCal().touch();
      return true;
    }

    return false;
  }

  /** ===================================================================
   *                   Sponsors
   *  =================================================================== */

  /** Return an array of SponsorVO objects
   *
   * @return SponsorVO[]    array of sponsors
   */
  public SponsorVO[] getSponsors() throws CalFacadeException {
    return getCal().getSponsors();
  }

  /** Return all sponsors - optionally including reserved entries.
   *
   * @param includeReserved true to include reserved entries.
   * @return SponsorVO[]    array of sponsor value objects
   */
  public SponsorVO[] getSponsors(boolean includeReserved)
      throws CalFacadeException {
    return getCal().getSponsors(includeReserved);
  }

  /** Return a sponsor with the given id
   *
   * @param id     int id of the sponsor
   * @return SponsorVo object representing the sponsor in question
   *                     null if it doesn't exist.
   */
  public SponsorVO getSponsor(int id) throws CalFacadeException {
    return getCal().getSponsor(id);
  }

  /** Add a sponsor to the database. The id will be set in the parameter
   * object.
   *
   * @param val   SponsorVO object to be added
   * @return int  new sponsor id
   */
  public int addSponsor(SponsorVO val) throws CalFacadeException {
    int id = getCal().addSponsor(val);
    getCal().touch();

    return id;
  }

  /** Replace a sponsor in the database.
   *
   * @param val   SponsorVO object to be replaced
   */
  public void replaceSponsor(SponsorVO val) throws CalFacadeException {
    getCal().replaceSponsor(val);
  }

  /** Delete a sponsor with the given id
   *
   * @param id       int id of the sponsor
   * @return boolean false if it didn't exist, true if it was deleted.
   */
  public boolean deleteSponsor(int id) throws CalFacadeException {
    if (id <= SponsorVO.maxReservedId) {
      // claim it doesn't exist
      return false;
    }

    if (getCal().deleteSponsor(id)) {
      getCal().touch();
      return true;
    }

    return false;

  }

  /** Check to see if a sponsor is referenced.
   *
   * @param id       int id of the sponsor
   * @return boolean true if the sponsor is referenced somewhere
   */
  public boolean checkSponsorRefs(int id) throws CalFacadeException {
    return getCal().checkSponsorRefs(id);
  }

  /** Return given public sponsor
   *
   * @param id         int id of the sponsor
   * @return SponsorVo object representing the sponsor in question
   *                     null if it doesn't exist.
   */
  public SponsorVO getPublicSponsor(int id) throws CalFacadeException {
    return getCal().getPublicSponsor(id);
  }

  /** Search the public sponsors for a matching name, phone, email and link
   *
   * @param s          SponsorVO object to match for
   * @return SponsorVO matching object or null if no match.
   */
  public SponsorVO findSponsor(SponsorVO s) throws CalFacadeException {
    SponsorVO[] ss = getSponsors();

    if (ss == null) {
      return null;
    }

    String name = s.getName();
    String phone = s.getPhone();
    String email = s.getEmail();
    String link = s.getLink();

    for (int i = 0; i < ss.length; i++) {
      SponsorVO si = ss[i];

      if ((name.equals(si.getName())) &&
          checkField(phone, si.getPhone()) &&
          checkField(email, si.getEmail()) &&
          checkField(link, si.getLink())) {
        return si;
      }
    }

    return null;
  }

  /** Ensure a sponsor exists. If it does returns the sponsor id.
   * If not creates the sponsor then returns the new id.
   *
   * @param sponsor   SponsorVO value object. If this object has the id set,
   *                  we assume the check was made previously.
   * @return int      sponsorid that matches or new sponsorid
   */
  public int ensureSponsorExists(SponsorVO sponsor) throws CalFacadeException {
    if (sponsor.getId() != 0) {
      // We've already checked it exists
      return sponsor.getId();
    }

    // Assume user entered a new sponsor
    SponsorVO s = findSponsor(sponsor);

    if (s != null) {
      return s.getId();
    }

    // doesn't exist at this point, so we add it to db table
    sponsor.setIsPublic(true);

    int id = getCal().addSponsor(sponsor);

    sponsor.setId(id);
    getCal().touch();

    return id;
  }

  /** Return an array of the given sponsors
   *
   * @param keys          Keys of the desired sponsors
   * @return SponsorVO[] Alphabetically sorted array
   */
  public SponsorVO[] getPublicSponsors(int[] keys)
      throws CalFacadeException {
    if (keys == null) {
      return null;
    }

    int sz = keys.length;
    SponsorVO[] sps = new SponsorVO[sz];

    for (int i = 0; i < sz; i++) {
      sps[i] = getCal().getPublicSponsor(keys[i]);
    }

    Arrays.sort(sps);

    return sps;
  }

  /** ===================================================================
   *                   Events
   *  =================================================================== */

  /** Return a single event for the current user
   *
   * @param   eventId   int id of the event
   * @return  EventVO   value object representing event.
   */
  public EventVO getEvent(int eventId) throws CalFacadeException {
    return getCal().getEvent(eventId);
  }

  /** Return the events for the given day as an array of value objects
   *
   * @param   date    MyCalendarVO object defining day
   * @return  EventVO[]  one days events or null for no events.
   */
  public EventVO[] getDaysEvents(MyCalendarVO date) throws CalFacadeException {
    return getCal().getDaysEvents(date);
  }

  /** Add an event to the database. The id will be set in the parameter
   * object.
   *
   * @param val   EventVO object to be added
   * @return int  new event id
   */
  public int addEvent(EventVO val) throws CalFacadeException {
    return getCal().addEvent(val);
  }

  /** Replace an event wit the same id in the database.
   *
   * @param val   EventVO object to be replaced
   */
  public void replaceEvent(EventVO val) throws CalFacadeException {
    getCal().replaceEvent(val);
  }

  /** Delete an event
   *
   * @param eventid   int id of event to delete
   * @return boolean false if it didn't exist, true if it was deleted.
   * @exception CalFacadeException If there's a database access problem
   */
  public boolean deleteEvent(int eventid) throws CalFacadeException {
    return getCal().deleteEvent(eventid);
  }

  /** ===================================================================
   *                   Public Events
   *  =================================================================== */

  /** Add an event and ensure location and sponsor exist. Some calendar
   * implementors choose to allow the dynamic creation of locations and
   * sponsors. For each of those, if we have an id, then the object represents
   * a preexisting database item.
   *
   * <p>Otherwsie the client has provided information which will be used to
   * locate an already existing location or sponsor. Failing that we use the
   * information to create a new entry.
   *
   * @param loc        LocationVO object
   * @param sp         SponsorVO object
   * @param event      EventVO object to be added
   * @return EventVO   Update object with embedded location and sponsor.
   */
  public EventVO addPublicEvent(LocationVO loc, SponsorVO sp,
                                EventVO event) throws CalFacadeException {
    int sponsorid = ensureSponsorExists(sp);
    int locationid = ensurePublicLocationExists(loc);

    event.setSponsor(sp);
    event.setLocation(loc);
    event.setCreator(getCal().getUserVO());

    int id = getCal().addEvent(event);
    event.setId(id);

    // finally, add the eventkeyword entry
    getCal().addEventKeyword(id, event.getKeyword(0).getId());

    getCal().touch();

    return event;
  }

  /** Update an event and ensure location and sponsor exist. Some calendar
   * implementors choose to allow the dynamic creation of locations and
   * sponsors. For each of those, if we have an id, then the object represents
   * a preexisting database item.
   *
   * <p>Otherwise the client has provided information which will be used to
   * locate an already existing location or sponsor. Failing that we use the
   * information to create a new entry.
   *
   * <p>If the location or sponsor changes, the old ones may no longer be
   * referenced by any events. If flagged this method will delete the old
   * unused entries.
   *
   * @param loc           LocationVO object
   * @param deleteOldLoc  true if we should delete old unused location
   * @param sp            SponsorVO object
   * @param deleteOldSp   true if we should delete old unused sponsor
   * @param oldKeywordId  Id of old event category. Same as newKeywordId
   *                      if unchanged.
   * @param newKeywordId  int id to use
   * @param event         EventVO object to be added
   * @return EventVO      Updated object with embedded location and sponsor.
   */
  public EventVO updatePublicEvent(LocationVO loc, boolean deleteOldLoc,
                                   SponsorVO sp, boolean deleteOldSp,
                                   int oldKeywordId,
                                   int newKeywordId,
                                   EventVO event) throws CalFacadeException {
    int sponsorid = ensureSponsorExists(sp);
    int locationid = ensurePublicLocationExists(loc);

    int origLocationid = event.getLocationid();
    int origSponsorid = event.getSponsorid();

    event.setSponsor(sp);
    event.setLocation(loc);

    int eventid = event.getId();

    getCal().replaceEvent(event);

    // If keyword changed, we need to change event_keywords entry
    if (oldKeywordId != newKeywordId) {
      getCal().deleteEventKeyword(eventid);

      // create new event_keywords entry
      getCal().addEventKeyword(eventid, newKeywordId);
    }

    /** Get the, possibly updated, sponsor and location ids.*/

    /* If we auto delete and the location changed and if the old location
       id no longer exists in events table then delete that location.
       */
    if ((origLocationid != event.getLocationid()) &&
        deleteOldLoc &&
        !getCal().checkLocationRefs(event.getLocationid())) {
      getCal().deleteLocation(origLocationid);
    }

    /* If we auto delete and the sponsor changed and if the old sponsorid
       no longer exists in events table then delete that sponsor.
       */
    if ((origSponsorid != event.getSponsorid()) &&
        deleteOldSp &&
        !getCal().checkSponsorRefs(origSponsorid)) {
      getCal().deleteSponsor(origSponsorid);
    }

    getCal().touch();

    return event;
  }

  /** Return public events for the current user which match the supplied
   * alert flag
   *
   * @param alerts      boolean true if we want alerts false for exclude alerts
   * @return EventVO[]  populated event value objects
   */
  public EventVO[] getPublicEvents(boolean alerts)
      throws CalFacadeException {
    return getPublicEvents(alerts, false);
  }

  /** Return active public events for the current user which match the
   * supplied alert flag
   *
   * @param alerts      boolean true if we want alerts false for exclude alerts
   * @return EventVO[]  populated event value objects
   */
  public EventVO[] getActivePublicEvents(boolean alerts)
      throws CalFacadeException {
    return getPublicEvents(alerts, true);
  }

  private EventVO[] getPublicEvents(boolean alerts,
                                    boolean active)
      throws CalFacadeException {
    EventVO[] evs;

    if (active) {
      evs = getCal().findPublicEvents(new MyCalendarVO(), null);
    } else {
      evs = getCal().findPublicEvents();
    }

    if (evs == null) {
      return new EventVO[0];
    }

    Vector v = new Vector();

    for (int i = 0; i < evs.length; i++) {
      KeywordVO k = evs[i].getKeyword(0);

      if ((k != null) &&
          (alerts == getKeyAlert(k.getId()))) {
        v.addElement(evs[i]);
      }
    }

    return (EventVO[])v.toArray(new EventVO[v.size()]);
  }

  /* * Return the public events for the given day as an array of value objects
   *
   * @param   date    MyCalendar object defining day
   * @return  EventVO[]  one days events or null for no events.
   * /
  public EventVO[] getDaysPublicEvents(MyCalendarVO date)
      throws CalFacadeException {
    return getCal().getDaysPublicEvents(date);
  }*/

  /* * Return the alerts for the given day as an array of value objects
   *
   * @param   date    MyCalendarVO object defining day
   * @return  EventVO[]  one days alerts or null for no events.
   * /
  public EventVO[] getDaysAlerts(MyCalendarVO date) throws CalFacadeException {
    EventVO[] evs = getCal().getDaysPublicEvents(date);

    if (evs == null) {
      return new EventVO[0];
    }

    Vector v = new Vector();

    for (int i = 0; i < evs.length; i++) {
      if (getKeyAlert(evs[i].getKeyword(0).getId())) {
        v.addElement(evs[i]);
      }
    }

    return (EventVO[])v.toArray(new EventVO[v.size()]);
  }*/

  /** Return the public events for the current user. This is an administrative
   * function for the public events admin client.
   *
   * @return  EventVO[]  events or null for no events.
   */
  public EventVO[] findPublicEvents() throws CalFacadeException {
    //return getCal().findPublicEvents(creator);
    throw new CalFacadeException("No direct access to method");
  }

  /** Return the public events for the current user within the given date
   * range. This is an administrative function for the public events admin
   * client.
   *
   * @return  EventVO[]  events or null for no events.
   */
  public EventVO[] findPublicEvents(MyCalendarVO startDate,
                                    MyCalendarVO endDate)
          throws CalFacadeException {
    //return getCal().findPublicEvents(creator, startDate, endDate);
    throw new CalFacadeException("No direct access to method");
  }

  /** Delete an event and optionally delete the old location and/or sponsor
   * entries if now unused.
   *
   * @param deleteOldLoc  true if we should delete old unused location
   * @param deleteOldSp  true if we should delete old unused sponsor
   * @param event      EventVO object to be added
   */
  public void deletePublicEvent(boolean deleteOldLoc,
                                boolean deleteOldSp,
                                EventVO event) throws CalFacadeException {
    int eventid = event.getId();

    getCal().deleteEvent(eventid);
    getCal().deleteEventKeyword(eventid);

    /* If we auto delete and if sponsor id no longer exists in events
       table then delete this sponsor.
     */
    if (deleteOldLoc &&
        !getCal().checkSponsorRefs(event.getSponsorid())) {
      getCal().deleteSponsor(event.getSponsorid());
    }

    /* If we auto delete and if location id no longer exists in events
       table then delete this location.
     */
    if (deleteOldSp &&
        !getCal().checkSponsorRefs(event.getLocationid())) {
      getCal().deleteLocation(event.getLocationid());
    }

    getCal().touch();
  }

  /** ===================================================================
   *                   Event keywords
   *  =================================================================== */

  /** Given a eventid and keywordid add an entry to the event keyword table
   *
   * @param eventid   Id of event
   * @param keywordid Id of keyword
   * @exception CalFacadeException If there's a database access problem
   */
  public void addEventKeyword(int eventid, int keywordid)
      throws CalFacadeException {
    getCal().addEventKeyword(eventid, keywordid);
  }

  /** Given a eventid delete an entry from the event keyword table
   *
   * @param eventid   Id of event
   * @exception CalFacadeException If there's a database access problem
   */
  public void deleteEventKeyword(int eventid) throws CalFacadeException {
    getCal().deleteEventKeyword(eventid);
  }

  /** ===================================================================
   *                   EventRefs
   *  =================================================================== */

  /** Create an event ref for the current user.
   *
   * @param   eventId   int id of the event to add
   * @return  boolean   false for no such item.
   */
  public boolean addEventRef(int eventId) throws CalFacadeException {
    return getCal().addEventRef(eventId);
  }

  /** Delete an event ref for the current user.
   *
   * @param   eventId   int id of the event to delete
   * @return  boolean   false for no such item.
   */
  public boolean deleteEventRef(int eventId) throws CalFacadeException {
    return getCal().deleteEventRef(eventId);
  }

  /** ===================================================================
   *                   Private methods
   *  =================================================================== */

  private Calintf getCal() throws CalFacadeException {
    if (cali == null) {
      cali = CalintfFactory.getCalintf(initObj, user, rights, isGuest,
                                       publicAdmin, debug);
    }

    return cali;
  }

  private boolean checkField(String fld1, String fld2) {
    if (fld1 == null) {
      if (fld2 == null) {
        return true;
      }
    } else if (fld1.equals(fld2)) {
      return true;
    }

    return false;
  }
}

